#
# Copyright 2017 the original author or authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""gRPC server endpoint"""
import os
import sys
import uuid
from Queue import Queue
from collections import OrderedDict
from os.path import abspath, basename, dirname, join, walk
import grpc
from concurrent import futures
import zlib

import django
sys.path.append('/opt/xos')
os.environ.setdefault("DJANGO_SETTINGS_MODULE", "xos.settings")

from protos import xos_pb2
from xos_grpc_api import XosService
from google.protobuf.empty_pb2 import Empty

from xos.logger import Logger, logging
logger = Logger(level=logging.INFO)

class XOSGrpcServer(object):

    def __init__(self, port=50055):
        self.port = port
        logger.info('init-grpc-server port=%d' % self.port)
        self.thread_pool = futures.ThreadPoolExecutor(max_workers=10)
        self.server = grpc.server(self.thread_pool)
        self.services = []

    def start(self):
        logger.debug('starting')

        # add each service unit to the server and also to the list
        for activator_func, service_class in (
            (xos_pb2.add_xosServicer_to_server, XosService),
        ):
            service = service_class(self.thread_pool)
            self.register(activator_func, service)

        # open port
        self.server.add_insecure_port('[::]:%s' % self.port)

        # strat the server
        self.server.start()

        logger.info('started')
        return self

    def stop(self, grace=0):
        logger.debug('stopping')
        for service in self.services:
            service.stop()
        self.server.stop(grace)
        logger.debug('stopped')

    def register(self, activator_func, service):
        """
        Allow late registration of gRPC servicers
        :param activator_func: The gRPC "add_XYZServicer_to_server method
        autogenerated by protoc.
        :param service: The object implementing the service.
        :return: None
        """
        self.services.append(service)
        activator_func(service, self.server)


# This is to allow running the GRPC server in stand-alone mode

if __name__ == '__main__':
    django.setup()

    server = XOSGrpcServer().start()

    import time
    _ONE_DAY_IN_SECONDS = 60 * 60 * 24
    try:
        while 1:
            time.sleep(_ONE_DAY_IN_SECONDS)
    except KeyboardInterrupt:
        server.stop()


