#
# Copyright 2012 New Dream Network, LLC (DreamHost)
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
"""Handler for producing network counter messages from Neutron notification
   events.

"""

import oslo.messaging
from oslo_config import cfg

from ceilometer.agent import plugin_base
from ceilometer.openstack.common import log
from ceilometer import sample

OPTS = [
    cfg.StrOpt('vcpeservice_control_exchange',
               default='vcpeservice',
               help="Exchange name for VCPE notifications."),
]

cfg.CONF.register_opts(OPTS)

LOG = log.getLogger(__name__)


class VCPENotificationBase(plugin_base.NotificationBase):

    resource_name = None

    @staticmethod
    def get_targets(conf):
        """Return a sequence of oslo.messaging.Target

        This sequence is defining the exchange and topics to be connected for
        this plugin.
        """
        LOG.info("SRIKANTH: get_targets for VCPE Notification Listener")
        return [oslo.messaging.Target(topic=topic,
                                      exchange=conf.vcpeservice_control_exchange)
                for topic in conf.notification_topics]

class VCPENotification(VCPENotificationBase):

    resource_name = None
    event_types = ['vcpe$']

    def process_notification(self, message):
        LOG.info('SRIKANTH: Received VCPE notification: vcpe_id=%(vcpe_id)s' % {'vcpe_id': message['payload']['vcpe_id']})
        yield sample.Sample.from_notification(
            name='vcpe',
            type=sample.TYPE_GAUGE,
            unit='vcpe',
            volume=1,
            user_id=message['payload']['user_id'],
            project_id=message['payload']['tenant_id'],
            resource_id=message['payload']['vcpe_id'],
            message=message)

class VCPEDNSCacheSize(VCPENotificationBase):

    resource_name = None
    event_types = ['vcpe.dns.cache.size']

    def process_notification(self, message):
        LOG.info('SRIKANTH: Received VCPE cache.size notification')
        yield sample.Sample.from_notification(
            name='vcpe.dns.cache.size',
            type=sample.TYPE_GAUGE,
            unit='entries',
            volume=float(message['payload']['cache_size']),
            user_id=message['payload']['user_id'],
            project_id=message['payload']['tenant_id'],
            resource_id=message['payload']['vcpe_id'],
            message=message)

class VCPEDNSTotalInsertedEntries(VCPENotificationBase):

    resource_name = None
    event_types = ['vcpe.dns.total_instered_entries']

    def process_notification(self, message):
        LOG.info('SRIKANTH: Received VCPE total_instered_entries notification')
        yield sample.Sample.from_notification(
            name='vcpe.dns.total_instered_entries',
            type=sample.TYPE_CUMULATIVE,
            unit='entries',
            volume=float(message['payload']['total_instered_entries']),
            user_id=message['payload']['user_id'],
            project_id=message['payload']['tenant_id'],
            resource_id=message['payload']['vcpe_id'],
            message=message)

class VCPEDNSReplacedUnexpiredEntries(VCPENotificationBase):

    resource_name = None
    event_types = ['vcpe.dns.replaced_unexpired_entries']

    def process_notification(self, message):
        LOG.info('SRIKANTH: Received VCPE replaced_unexpired_entries notification')
        yield sample.Sample.from_notification(
            name='vcpe.dns.replaced_unexpired_entries',
            type=sample.TYPE_CUMULATIVE,
            unit='entries',
            volume=float(message['payload']['replaced_unexpired_entries']),
            user_id=message['payload']['user_id'],
            project_id=message['payload']['tenant_id'],
            resource_id=message['payload']['vcpe_id'],
            message=message)

class VCPEDNSQueriesForwarded(VCPENotificationBase):

    resource_name = None
    event_types = ['vcpe.dns.queries_forwarded']

    def process_notification(self, message):
        LOG.info('SRIKANTH: Received VCPE queries_forwarded notification')
        yield sample.Sample.from_notification(
            name='vcpe.dns.queries_forwarded',
            type=sample.TYPE_CUMULATIVE,
            unit='queries',
            volume=float(message['payload']['queries_forwarded']),
            user_id=message['payload']['user_id'],
            project_id=message['payload']['tenant_id'],
            resource_id=message['payload']['vcpe_id'],
            message=message)

class VCPEDNSQueriesAnsweredLocally(VCPENotificationBase):

    resource_name = None
    event_types = ['vcpe.dns.queries_answered_locally']

    def process_notification(self, message):
        LOG.info('SRIKANTH: Received VCPE queries_answered_locally notification')
        yield sample.Sample.from_notification(
            name='vcpe.dns.queries_answered_locally',
            type=sample.TYPE_CUMULATIVE,
            unit='queries',
            volume=float(message['payload']['queries_answered_locally']),
            user_id=message['payload']['user_id'],
            project_id=message['payload']['tenant_id'],
            resource_id=message['payload']['vcpe_id'],
            message=message)

class VCPEDNSServerQueriesSent(VCPENotificationBase):

    resource_name = None
    event_types = ['vcpe.dns.server.queries_sent']

    def process_notification(self, message):
        LOG.info('SRIKANTH: Received VCPE server.queries_sent notification')
        resource_id = message['payload']['vcpe_id'] + '-' + message['payload']['upstream_server']
        yield sample.Sample.from_notification(
            name='vcpe.dns.server.queries_sent',
            type=sample.TYPE_CUMULATIVE,
            unit='queries',
            volume=float(message['payload']['queries_sent']),
            user_id=message['payload']['user_id'],
            project_id=message['payload']['tenant_id'],
            resource_id=resource_id,
            message=message)

class VCPEDNSServerQueriesFailed(VCPENotificationBase):

    resource_name = None
    event_types = ['vcpe.dns.server.queries_failed']

    def process_notification(self, message):
        LOG.info('SRIKANTH: Received VCPE server.queries_failed notification')
        resource_id = message['payload']['vcpe_id'] + '-' + message['payload']['upstream_server']
        yield sample.Sample.from_notification(
            name='vcpe.dns.server.queries_failed',
            type=sample.TYPE_CUMULATIVE,
            unit='queries',
            volume=float(message['payload']['queries_failed']),
            user_id=message['payload']['user_id'],
            project_id=message['payload']['tenant_id'],
            resource_id=resource_id,
            message=message)

