/*
 * Copyright 2020-present Open Networking Foundation

 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at

 * http://www.apache.org/licenses/LICENSE-2.0

 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package core

import (
	"context"
	"encoding/binary"
	"errors"
	"fmt"
	"strconv"
	"strings"
	"sync"
	"time"

	"github.com/google/gopacket"
	"github.com/google/gopacket/layers"
	"github.com/opencord/voltha-openolt-adapter/internal/pkg/core/l2oam"
	"github.com/opencord/voltha-protos/v3/go/common"
	oop "github.com/opencord/voltha-protos/v3/go/openolt"
	"github.com/opencord/voltha-protos/v3/go/voltha"
)

// DeviceNameOlt is a constant for OLT
const DeviceNameOlt = "L2oamOltDevice"

// DeviceNameOnu is a constant for ONU
const DeviceNameOnu = "L2oamOnuDevice"

// KeepAliveInterval is an interval time for keep-alive
const KeepAliveInterval = 800

// KeepAliveCounter is max count of not receiving keep-alive response
const KeepAliveCounter = 10

// ResponseTimer is wait time for receiving response message(seconds)
const ResponseTimer = 20

//KeepAliveStatusE is state of keep-alive message
type KeepAliveStatusE int

const (
	KeepAliveDisable KeepAliveStatusE = iota
	KeepAliveStep1
	KeepAliveStep2
	KeepAliveStep3
)

// map for L2oamDevice, its key is MAC address without colon
var deviceMap map[string]L2oamDevice

// map for L2oamDevice, its key is device-ID
var deviceMacMap map[string]string

var mapMutex sync.Mutex

var l2oamDeviceHandler *DeviceHandler

func init() {
	deviceMap = map[string]L2oamDevice{}
	deviceMacMap = map[string]string{}
	monitoringOnuStatus()
}

// FindL2oamDevice returns a L2oamDevice using MAC address
func FindL2oamDevice(macAddress string) L2oamDevice {
	mapMutex.Lock()
	defer mapMutex.Unlock()
	mac := strings.Replace(macAddress, ":", "", -1)
	onu, ok := deviceMap[mac]
	if !ok {
		return nil
	}
	return onu

}

// FindL2oamDeviceByDeviceID returns a L2oamDevice using device-ID
func FindL2oamDeviceByDeviceID(deviceID string) L2oamDevice {
	mapMutex.Lock()
	defer mapMutex.Unlock()
	mac, ok := deviceMacMap[deviceID]
	if !ok {
		return nil
	}
	onu, ok := deviceMap[mac]
	if !ok {
		return nil
	}
	return onu
}

// SetupL2oamDeleteFlag sets delete flag
func SetupL2oamDeleteFlag(deviceID string) bool {
	mapMutex.Lock()
	defer mapMutex.Unlock()

	mac, ok := deviceMacMap[deviceID]
	if !ok {
		return false
	}
	device, ok := deviceMap[mac]
	if !ok {
		return false
	}
	if device.getDeleteFlag() {
		return false
	}
	device.setDeleteFlag()
	deviceMap[mac] = device

	return true
}

// NewL2oamOltDevice creates a new OLT device
func NewL2oamOltDevice(macAddress string, dh *DeviceHandler) (L2oamDevice, error) {
	mapMutex.Lock()
	defer mapMutex.Unlock()

	l2oamDeviceHandler = dh

	mac := strings.Replace(macAddress, ":", "", -1)
	_, ok := deviceMap[mac]
	if !ok {
		device := &L2oamOltDevice{
			Base: DeviceBase{
				ResCh:          nil,
				KeepAliveResCh: make(chan *layers.Ethernet, 100),
				DstMac:         mac,
				DH:             dh,
				DeviceID:       dh.device.Id,
				StopCh:         make(chan string, 1),
				ActiveState:    false,
				Name:           DeviceNameOlt,
			},
			L2SwitchDomainMap: map[uint32]*l2SwitchDomain{},
		}
		deviceMap[mac] = device
		deviceMacMap[dh.device.Id] = mac
		return device, nil
	}
	return nil, fmt.Errorf("NewL2oamOltDevice() Already registered. mac=%s", mac)
}

// NewL2oamOnuDevice creates a new ONU device
func NewL2oamOnuDevice(macAddress string, dh *DeviceHandler) (L2oamDevice, error) {
	mapMutex.Lock()
	defer mapMutex.Unlock()

	mac := strings.Replace(macAddress, ":", "", -1)
	_, ok := deviceMap[mac]
	if !ok {
		device := &L2oamOnuDevice{
			Base: DeviceBase{
				ResCh:          nil,
				KeepAliveResCh: make(chan *layers.Ethernet, 100),
				DstMac:         mac,
				DH:             dh,
				StopCh:         make(chan string, 1),
				ActiveState:    true,
				Name:           DeviceNameOnu,
			},
			keepAliveResponse: make(chan string, 1),
			MacAddress:        mac,
		}
		deviceMap[mac] = device
		return device, nil
	}
	return nil, fmt.Errorf("NewL2oamOnuDevice() Already registered. mac=%s", mac)
}

// DeleteAllDevice deletes all devices
func DeleteAllDevice() {
	mapMutex.Lock()
	defer mapMutex.Unlock()

	for _, value := range deviceMap {
		value.delete(context.Background())
	}
	deviceMap = map[string]L2oamDevice{}
	deviceMacMap = map[string]string{}
}

// DeleteDevice deletes device
func DeleteDevice(deviceId string, deviceMac string) {
	mapMutex.Lock()
	defer mapMutex.Unlock()

	_, ok := deviceMacMap[deviceId]
	if ok {
		delete(deviceMacMap, deviceId)
		logger.Info(context.Background(), fmt.Sprintf("DeleteDevice() deviceMacMap. deviceid=%s, ", deviceId))
	}
	mac := strings.Replace(deviceMac, ":", "", -1)
	_, ok = deviceMap[mac]
	if ok {
		delete(deviceMap, mac)
		logger.Info(context.Background(), fmt.Sprintf("DeleteDevice() deviceMap. mac=%s, ", mac))
	}
}

// L2oamDevice interface provides some methods for device
// OLT and ONU implement each functions
type L2oamDevice interface {
	getDeviceName() string
	send(message gopacket.SerializableLayer) error
	waitResponse(timeoutSecond int) (*layers.Ethernet, error)
	sendAndWait(message gopacket.SerializableLayer, timeoutSecond int) (*layers.Ethernet, error)
	recieve(etherPacket *layers.Ethernet)
	recieveKeepAlive(etherPacket *layers.Ethernet)
	startKeepAlive()
	stopKeepAlive()
	setActiveState(ctx context.Context, isActive bool)
	startMountSequence(ctx context.Context, pkgType string, cmd *L2oamCmd)
	reboot(ctx context.Context)
	setL2oamCmd(*L2oamCmd)
	getL2oamCmd() *L2oamCmd
	setObjectContext(*l2oam.TomiObjectContext)
	getObjectContext() *l2oam.TomiObjectContext
	setReferenceTable(*l2oam.GetTrafficControlReferenceTableRes)
	getReferenceTable() *l2oam.GetTrafficControlReferenceTableRes
	setProfile(*l2oam.SetGenericActionCreateRes, *l2oam.SetGenericActionCreateRes)
	getProfile() (*l2oam.SetGenericActionCreateRes, *l2oam.SetGenericActionCreateRes)
	delete(ctx context.Context)
	setDeleteFlag()
	getDeleteFlag() bool
	setAutonomousFlag(bool)
	getAutonomousFlag() bool
	updateMap()
	receiveEapol(etherPacket *layers.Ethernet)
	addFlow(ctx context.Context, cmd *L2oamCmd) error
	updateFlow(ctx context.Context, cmd *L2oamCmd) error
	removeFlow(ctx context.Context) error
}

type l2SwitchDomain struct {
	OcAction *l2oam.TomiObjectContext
	Tpid     []byte
	Vid      []byte
	Onus     map[string]bool
}

// DeviceBase ====================================
// DeviceBase contains common functions
// ===============================================
type DeviceBase struct {
	ResCh                chan *layers.Ethernet
	KeepAliveResCh       chan *layers.Ethernet
	DstMac               string
	DH                   *DeviceHandler
	DeviceID             string
	mu                   sync.Mutex
	muResponse           sync.Mutex
	muResCh              sync.Mutex
	StopCh               chan string
	ActiveState          bool
	KeepAliveStatus      KeepAliveStatusE
	KeepAliveRemoteValue []byte
	KeepAliveSendCounter int
	Name                 string
	DeleteFlag           bool
	AutonomousFlag       bool
	FlowAdded            bool
	ActionIds            map[int]uint32
	EapFlag              bool
}

const (
	ActionIDFilterPonPort = 1
	ActionIDFilterEthPort = 2
)

func (d *DeviceBase) getDeviceName() string {
	return d.Name
}

func (d *DeviceBase) send(message gopacket.SerializableLayer) error {
	var bytes []byte
	if vlanEnable {
		vlanLayer := &layers.Dot1Q{
			VLANIdentifier: vlanIdManagement,
			Type:           0xa8c8,
		}
		bytes = l2oam.CreateMessageVlan(
			SrcMac,
			convertMacString(d.DstMac),
			0x88a8,
			message,
			vlanLayer,
		)
	} else {
		bytes = l2oam.CreateMessage(
			SrcMac,
			convertMacString(d.DstMac),
			0xa8c8,
			message,
		)

	}
	GetL2oamHandle().send(bytes)
	return nil
}

func (d *DeviceBase) waitResponse(timeoutSecond int) (*layers.Ethernet, error) {
	d.muResponse.Lock()
	defer d.muResponse.Unlock()

	d.ResCh = make(chan *layers.Ethernet, 1)
	defer func() {
		d.muResCh.Lock()
		close(d.ResCh)
		d.ResCh = nil
		d.muResCh.Unlock()
	}()
	ctx := context.Background()
	var res *layers.Ethernet
	var err error
	select {
	case res = <-d.ResCh:
		// DeviceBase do nothing
		logger.Debug(ctx, fmt.Sprintf("[%s] DeviceBase.waitResponse() received", d.getDeviceName()))
	case <-time.After(time.Duration(timeoutSecond) * time.Second):
		err = fmt.Errorf("DeviceBase.waitResponse().ResCh receive timeout")
	}
	return res, err
}

func (d *DeviceBase) sendAndWait(message gopacket.SerializableLayer, timeoutSecond int) (*layers.Ethernet, error) {
	if err := d.send(message); err != nil {
		return nil, err
	}

	return d.waitResponse(timeoutSecond)
}

func (d *DeviceBase) recieve(etherPacket *layers.Ethernet) {
	d.muResCh.Lock()
	defer d.muResCh.Unlock()

	if d.ResCh != nil {
		if len(d.ResCh) == cap(d.ResCh) {
			logger.Error(context.Background(), fmt.Sprintf("[%s] recieve() ResCh channel is full. MacAddress=%s", d.getDeviceName(), d.DstMac))
		} else {
			d.ResCh <- etherPacket
		}
	}
}

func (d *DeviceBase) recieveKeepAlive(etherPacket *layers.Ethernet) {
	d.KeepAliveResCh <- etherPacket
}

func (d *DeviceBase) startKeepAlive() {
	go func() {
		ctx := context.Background()
		d.KeepAliveStatus = KeepAliveStep1
		d.countClearKeepAliveCounter()
		logger.Debug(ctx, fmt.Sprintf("[%s] startKeepAlive(). MacAddress=%s, %v", d.getDeviceName(), d.DstMac, d))

		for {
			select {
			case <-d.StopCh:
				logger.Debug(ctx, fmt.Sprintf("[%s] startKeepAlive() StopCh request. MacAddress=%s", d.getDeviceName(), d.DstMac))
				return
			case etherPacket := <-d.KeepAliveResCh:
				packet := &l2oam.OAMPDUInformation{}
				if err := packet.Decode(etherPacket.Payload); err != nil {
					break
				}
				logger.Debug(ctx, fmt.Sprintf("[%s] receive KeepAlive() mac=%s", d.getDeviceName(), d.DstMac))

				// discards received packet if the device is not active
				if d.isEnable() {
					// keep received LIValue for sending next messages
					d.KeepAliveRemoteValue = packet.LIValue
					if packet.Flags == 0x0028 {
						d.KeepAliveStatus = KeepAliveStep2
					} else {
						d.KeepAliveStatus = KeepAliveStep3
					}
					d.setActiveState(ctx, true)
					d.countClearKeepAliveCounter()
				}

			case <-time.After(KeepAliveInterval * time.Millisecond):

				if !d.isActive() {
					d.countClearKeepAliveCounter()
				} else {
					logger.Debug(ctx, fmt.Sprintf("[%s] KeepAlive timer expired. MacAddress=%s", d.getDeviceName(), d.DstMac))
					// First three messages are sent each messages.
					// After that, third message will be sent repeatedly
					var err error = nil
					if d.KeepAliveStatus == KeepAliveStep1 {
						err = d.send(l2oam.GeneateKeepAlive1(l2oam.OnuPkgType == l2oam.OnuPkgTypeA))
					} else if d.KeepAliveStatus == KeepAliveStep2 {
						err = d.send(l2oam.GeneateKeepAlive2(d.KeepAliveRemoteValue, l2oam.OnuPkgType == l2oam.OnuPkgTypeA))
					} else if d.KeepAliveStatus == KeepAliveStep3 {
						err = d.send(l2oam.GeneateKeepAlive3(d.KeepAliveRemoteValue))
					}
					if err != nil {
						break
					}
					d.countUpKeepAliveCounter()

					if d.isKeepAliveLimitOver() {
						d.setActiveState(ctx, false)
					}
				}

				if d.EapFlag {
					logger.Debug(ctx, fmt.Sprintf("[%s] EapFlag == true. MacAddress=%s", d.getDeviceName(), d.DstMac))
					/*
						onu := FindL2oamDevice(d.DstMac)
						if onu == nil {
							logger.Debug(ctx, "onu not found.")
						} else {
							go d.DH.L2oamAddFlowAndMount(ctx, d.DeviceID, []byte{0x64}, []byte{0x00, 0x64})
						}
					*/
					d.EapFlag = false
				}
			}
		}
	}()
}

func (d *DeviceBase) stopKeepAlive() {
	if len(d.StopCh) == cap(d.StopCh) {
		logger.Error(context.Background(), fmt.Sprintf("[%s] stopKeepAlive stop channel is full. MacAddress=%s", d.getDeviceName(), d.DstMac))
	} else {
		d.StopCh <- "Stop"
	}
}

func (d *DeviceBase) countUpKeepAliveCounter() {
	d.mu.Lock()
	defer d.mu.Unlock()
	d.KeepAliveSendCounter++
}

func (d *DeviceBase) countClearKeepAliveCounter() {
	d.mu.Lock()
	defer d.mu.Unlock()
	d.KeepAliveSendCounter = 0
}

func (d *DeviceBase) isKeepAliveLimitOver() bool {
	d.mu.Lock()
	defer d.mu.Unlock()
	return KeepAliveCounter <= d.KeepAliveSendCounter
}

func (d *DeviceBase) isActive() bool {
	if !d.isEnable() {
		return false
	}
	return d.ActiveState
}

func (d *DeviceBase) isEnable() bool {
	if d.DeviceID == "" {
		return true
	}
	device, err := d.DH.coreProxy.GetDevice(context.Background(), d.DeviceID, d.DeviceID)
	if err != nil || device == nil {
		logger.Debug(context.Background(), fmt.Sprintf("[%s] isEnable() coreProxy.GetDevice() deviceid=%s, error. %v", d.getDeviceName(), d.DeviceID, err))
		return false
	}
	return device.AdminState == common.AdminState_ENABLED
}

func (d *DeviceBase) setActiveState(ctx context.Context, isActive bool) {
	d.ActiveState = isActive

	switch d.getDeviceName() {
	case DeviceNameOlt:
		if !isActive {
			d.ActiveState = true
		}
		updateOltActiveStatus(ctx, d.DH, isActive)

	case DeviceNameOnu:
		// updateOnuActiveStatus(ctx, d.DstMac, isActive, d.DH)
		updateOnuActiveStatus2(ctx, d.DH, isActive, d.DeviceID)
	}
}

/*
func (d *DeviceBase) startMountSequence(ctx context.Context, pkgType string, cmd *L2oamCmd) {
}

func (d *DeviceBase) reboot(ctx context.Context) {
}
*/

func (d *DeviceBase) delete(ctx context.Context) {
	logger.Info(ctx, fmt.Sprintf("[%s] delete() deviceid=%s, mac=%s", d.getDeviceName(), d.DeviceID, d.DstMac))
	d.stopKeepAlive()
}

func (d *DeviceBase) setDeleteFlag() {
	d.DeleteFlag = true
}

func (d *DeviceBase) getDeleteFlag() bool {
	return d.DeleteFlag
}

func (d *DeviceBase) setAutonomousFlag(flag bool) {
	d.AutonomousFlag = flag
}

func (d *DeviceBase) getAutonomousFlag() bool {
	return d.AutonomousFlag
}

func (d *DeviceBase) receiveEapol(etherPacket *layers.Ethernet) {
	go func() {
		packetBytes := append(etherPacket.Contents, etherPacket.Payload...)
		pktInd := &oop.PacketIndication{
			IntfId:    0,
			GemportId: 1,
			PortNo:    16,
			Pkt:       packetBytes,
			IntfType:  "pon",
		}
		logger.Info(context.Background(), fmt.Sprintf("[%s] receiveEapol() deviceid=%s, pktInd=%x", d.getDeviceName(), d.DeviceID, pktInd))

		if err := d.DH.handlePacketIndication(context.Background(), pktInd); err != nil {
			logger.Error(context.Background(), fmt.Sprintf("[%s] receiveEapol() error. deviceid=%s, ", d.getDeviceName(), d.DeviceID))
		}
	}()
}

// L2oamOltDevice =================================
// L2oamOltDevice provides functions for OLT device
// ================================================
type L2oamOltDevice struct {
	Base                DeviceBase
	autonomousExecMutex sync.Mutex
	L2oamCmd            *L2oamCmd
	ObjectContext       *l2oam.TomiObjectContext
	ReferenceTable      *l2oam.GetTrafficControlReferenceTableRes
	DownProfile         *l2oam.SetGenericActionCreateRes
	UpProfile           *l2oam.SetGenericActionCreateRes
	L2SwitchDomainMap   map[uint32]*l2SwitchDomain
}

func (d *L2oamOltDevice) getDeviceName() string {
	return d.Base.getDeviceName()
}
func (d *L2oamOltDevice) send(message gopacket.SerializableLayer) error {
	return d.Base.send(message)
}
func (d *L2oamOltDevice) waitResponse(timeoutSecond int) (*layers.Ethernet, error) {
	return d.Base.waitResponse(timeoutSecond)
}
func (d *L2oamOltDevice) sendAndWait(message gopacket.SerializableLayer, timeoutSecond int) (*layers.Ethernet, error) {
	return d.Base.sendAndWait(message, timeoutSecond)
}

func (d *L2oamOltDevice) recieve(etherPacket *layers.Ethernet) {
	bytes := etherPacket.Payload
	opcode := bytes[0]
	oampduCode := bytes[3]
	tomiOpCode := bytes[7]
	logger.Debug(context.Background(), fmt.Sprintf("[%s] recieve. opcode=%v oampduCode=%v,  %x", d.getDeviceName(), opcode, oampduCode, bytes))

	if opcode == 0x03 && oampduCode == 0xfe {
		// Autonomous message
		if tomiOpCode == 0xae {
			go d.receiveAutonomounsEvent(etherPacket)
		} else {
			d.Base.recieve(etherPacket)
		}
	} else if opcode == 0xfd {
		// other messages
		d.Base.recieve(etherPacket)
	} else {
		logger.Error(context.Background(), fmt.Sprintf("[%s] receive error. invalid message %v", d.getDeviceName(), etherPacket))
	}

}
func (d *L2oamOltDevice) recieveKeepAlive(etherPacket *layers.Ethernet) {
	d.Base.recieveKeepAlive(etherPacket)
}
func (d *L2oamOltDevice) startKeepAlive() {
	d.Base.startKeepAlive()
}
func (d *L2oamOltDevice) stopKeepAlive() {
	d.Base.stopKeepAlive()
}
func (d *L2oamOltDevice) receiveAutonomounsEvent(etherPacket *layers.Ethernet) {
	d.autonomousExecMutex.Lock()
	defer d.autonomousExecMutex.Unlock()

	packet := &l2oam.AutonomousEvent{}
	if err := packet.Decode(etherPacket.Payload); err != nil {
		return
	}

	if packet.IsRegistrationStatusMessage() {
		if d.getAutonomousFlag() {
			d.addOnu(packet)
		} else {
			logger.Debug(context.Background(), fmt.Sprintf("[%s] receiveAutonomounsEvent() skip. AutonomousFlag is false. ", d.getDeviceName()))
		}
	}
}
func (d *L2oamOltDevice) addOnu(packet *l2oam.AutonomousEvent) {
	ctx := context.Background()
	logger.Debug(ctx, fmt.Sprintf("[%s] addOnu() Start. ", d.getDeviceName()))
	operIndication := &oop.Indication_IntfOperInd{
		IntfOperInd: &oop.IntfOperIndication{
			Type:      "nni",
			IntfId:    0,
			OperState: "up",
		},
	}
	indication := &oop.Indication{
		Data: operIndication,
	}
	d.Base.DH.handleIndication(ctx, indication)
	operIndication.IntfOperInd.Type = "pon"
	d.Base.DH.handleIndication(ctx, indication)

	oc := &l2oam.TomiObjectContext{
		Branch:   packet.ComResp.OCBranch,
		Type:     packet.ComResp.OCType,
		Length:   packet.ComResp.OCLength,
		Instance: packet.ComResp.OCInstance,
	}

	ether, err := d.sendAndWait(l2oam.GenerateGetMpcpMacAddress(oc), ResponseTimer)
	if err != nil {
		logger.Error(ctx, fmt.Sprintf("[%s] addOnu() GetMpcpMacAddress Send Error: %v", d.getDeviceName(), err))
		return
	}
	macPacket := &l2oam.GetMpcpMacAddressRes{}
	if err = macPacket.Decode(ether.Payload); err != nil {
		return
	}
	macAddress := macPacket.GetMacAddress()
	logger.Debug(ctx, fmt.Sprintf("[%s] addOnu() GetMpcpMacAddress Success: macAddress=%s, response=%x:%s", d.getDeviceName(), macAddress, ether.Payload, macPacket))

	var onu *L2oamOnuDevice
	device := FindL2oamDevice(macAddress)
	if device != nil {
		logger.Debug(ctx, fmt.Sprintf("addOnu() FindL2oamDevice() is find. macAddress=%s", macAddress))
		onu = device.(*L2oamOnuDevice)
		if onu != nil {
			logger.Debug(ctx, fmt.Sprintf("addOnu() assertion success. macAddress=%s, deviceId=%s", macAddress, onu.Base.DeviceID))
			onu.setActiveState(ctx, true)
		}
		return
	}

	logger.Debug(ctx, fmt.Sprintf("addOnu() NewL2oamOnuDevice() called. macAddress=%s", macAddress))
	device, _ = NewL2oamOnuDevice(macAddress, d.Base.DH)
	onu = device.(*L2oamOnuDevice)

	ether, err = d.sendAndWait(l2oam.GenerateGetMpcpLLId(oc), ResponseTimer)
	if err != nil {
		logger.Error(ctx, fmt.Sprintf("[%s] addOnu() GetMpcpLLId Send Error: %v", d.getDeviceName(), err))
		return
	}

	llidPacket := &l2oam.GetMpcpLLIdRes{}
	if err = llidPacket.Decode(ether.Payload); err != nil {
		return
	}
	llid := llidPacket.GetLLID()
	logger.Debug(ctx, fmt.Sprintf("[%s] addOnu() GetMpcpLLId Success: LLID=%d, response=%x:%s", d.getDeviceName(), llid, ether.Payload, llidPacket))

	vendorID := ""
	VendorSpecific := ""
	serialNumber := ""
	onu.setActiveState(ctx, false)

	if err := onu.send(l2oam.GeneateKeepAlive1(l2oam.OnuPkgType == l2oam.OnuPkgTypeA)); err != nil {
		logger.Debug(ctx, fmt.Sprintf("error: %v", err))
	}
	go func() {

		err = onu.waitKeepALiveResponse()
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] addOnu() onu keep alive error. %s", d.getDeviceName(), err))
			DeleteDevice(onu.Base.DeviceID, onu.Base.DstMac)
			return
		}
		onu.keepAliveResponse = nil
		logger.Info(ctx, fmt.Sprintf("[%s] addOnu() onu keep alive success.", d.getDeviceName()))
		onu.startKeepAlive()
		vendorID = "FEC"
		VendorSpecific = "EponONU"
		//serialNumber = macAddress[6:12]
		serialNumber = strings.Replace(macAddress, ":", "", -1)[6:12]

		onuDiscInd := &oop.OnuDiscIndication{
			IntfId: 0,
			SerialNumber: &oop.SerialNumber{
				VendorId:       []byte(vendorID),
				VendorSpecific: []byte(VendorSpecific),
			},
		}
		if err = d.Base.DH.onuDiscIndication2(ctx, onuDiscInd, serialNumber, macAddress); err != nil {
			return
		}

		onu.setObjectContext(oc)
		onu.updateMap()
		// onu := FindL2oamDevice(macAddress)
		// if onu == nil {
		// 	logger.Debug(ctx, fmt.Sprintf("[%s] addOnu() NewL2oamOnuDevice() called. macAddress=%s", d.getDeviceName(), macAddress))
		// 	onu, _ = NewL2oamOnuDevice(macAddress, d.Base.DH, deviceId)
		// 	onu.startKeepAlive()
		// }
		// onu.setActiveState(ctx, true)
		device = FindL2oamDevice(macAddress)
		onuInd := &oop.OnuIndication{
			IntfId:       0,
			OnuId:        device.(*L2oamOnuDevice).OnuID,
			OperState:    "up",
			AdminState:   "up",
			SerialNumber: onuDiscInd.SerialNumber,
		}
		if err := d.Base.DH.onuIndication(ctx, onuInd); err != nil {
			return
		}
	}()
}

func (dh *DeviceHandler) fakeOmciIndication(ctx context.Context, intfID uint32, onuID uint32, pkt []byte) error {
	omciInd := &oop.OmciIndication{
		IntfId: intfID,
		OnuId:  onuID,
		Pkt:    make([]byte, 48),
	}

	onuKey := dh.formOnuKey(intfID, onuID)
	onuDev, ok := dh.onus.Load(onuKey)
	if !ok {
		logger.Errorf(ctx, "not-found-onu-device. intf-id:%d, onu-id:%d", intfID, onuID)
		return nil
	}
	onuDeviceID := (onuDev.(*OnuDevice)).deviceID
	onu := FindL2oamDeviceByDeviceID(onuDeviceID)
	if onu == nil {
		logger.Errorf(ctx, "not-found-l2oam-onu-device. onu-device-id:%s", onuDeviceID)
		return nil
	}


	if len(pkt) < 11 {
		return nil
	}
	messageType := uint8(pkt[2])
	meID := binary.BigEndian.Uint16(pkt[4:6])
	meSubID := binary.BigEndian.Uint16(pkt[6:8])
	omciInd.Pkt[0] = pkt[0]
	omciInd.Pkt[1] = pkt[1]
	omciInd.Pkt[2] = (pkt[2] & 0xbf) | 0x20 // AR -> AK
	copy(omciInd.Pkt[3:8], pkt[3:8])
	contentIdx := 8

	switch messageType {
	case 0x49: // Get
		if meID == 0x0101 { // ONU2-G
			if pkt[contentIdx] == 0xe0 {
				copy(omciInd.Pkt[contentIdx:], []byte{
					0x00, 0xe0, 0x00, 0x31, 0x32, 0x33, 0x34, 0x35,
					0x31, 0x32, 0x33, 0x34, 0x35, 0x31, 0x32, 0x33,
					0x34, 0x35, 0x31, 0x32, 0x33, 0x34, 0x35, 0xb4,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				})
			} else if pkt[contentIdx] == 0x80 {
				copy(omciInd.Pkt[contentIdx:], []byte{
					0x00, 0x80, 0x00, 0x31, 0x32, 0x33, 0x34, 0x35,
					0x31, 0x32, 0x33, 0x34, 0x35, 0x31, 0x32, 0x33,
					0x34, 0x35, 0x31, 0x32, 0x33, 0x34, 0x35, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				})
			} else {
				logger.Error(ctx, fmt.Sprintf("fakeOmciIndication() unknown-format:message-type:%x, me-id:%x",
					messageType, meID))
			}
		} else if meID == 0x0100 { // ONU-G
			copy(omciInd.Pkt[contentIdx:], []byte{
				0x00, 0xa0, 0x00, 0x45, 0x50, 0x4f, 0x4e, 0x45,
				0x50, 0x4f, 0x4e, 0x00, 0x00, 0x00, 0x01, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			})
		} else if meID == 0x0007 { // Software image
			if meSubID == 0 {
				copy(omciInd.Pkt[contentIdx:], []byte{
					0x00, 0xa0, 0x00, 0x30, 0x30, 0x30, 0x30, 0x30,
					0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
					0x31, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				})
			} else {
				copy(omciInd.Pkt[contentIdx:], []byte{
					0x00, 0xa0, 0x00, 0x30, 0x30, 0x30, 0x30, 0x30,
					0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
					0x31, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				})
			}
		} else if meID == 0x0086 { // IP host config data
			var macOctets [6]byte
			onuDev := onu.(*L2oamOnuDevice)
			if len(onuDev.MacAddress) == 12 {
				for idx := 0; idx < 6; idx++ {
					if v, err := strconv.ParseInt(onuDev.MacAddress[idx*2:(idx*2)+2], 16, 8); err == nil {
						macOctets[idx] = byte(v)
					}
				}
			}
			// 0x09-0a: Mask
			// 0x0b-10: MAC Address
			copy(omciInd.Pkt[contentIdx:], []byte{
				0x00, 0x40, 0x00, macOctets[0], macOctets[1], macOctets[2], macOctets[3], macOctets[4],
				macOctets[5], 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			})
		} else {
			logger.Error(ctx, fmt.Sprintf("fakeOmciIndication() unknown-format:message-type:%x, me-id:%x",
				messageType, meID))
		}
	case 0x4f: // MIB reset
		copy(omciInd.Pkt[contentIdx:], []byte{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		})
	case 0x4d: // MIB upload
		copy(omciInd.Pkt[contentIdx:], []byte{
			// first 2 bytes: the number of upload message
			// upload messages are sent specified number of times
			// following this message
			0x01, 0x23 - 3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		})
	case 0x4e: // MIB upload next
		uploadIdx := binary.BigEndian.Uint16(pkt[contentIdx : contentIdx+2])
		copy(omciInd.Pkt[contentIdx:], [][]byte{
			{
				0x00, 0x02, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				// addr: memo
				// 0x08-09: Class ID(#6 Circuit Pack)
				// 0x0a-0b: Managed entity ID(#257 ONU2-G)
				// 0x0c-0d: field mask, 0xf000 means first 4 fields
				// are stored in this message
				// 0x0e: Type, 0x2f means 10/100/1000 BASE-T
				// 0x0f: Number of ports
				// 0x10-17: Serial Number
				// 0x18-25: Version
				0x00, 0x06, 0x01, 0x01, 0xf0, 0x00, 0x2f, 0x04, // 0x08 - 0x0f
				0x49, 0x53, 0x4b, 0x54, 0x71, 0xe8, 0x00, 0x80, // 0x10 - 0x17
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x18 - 0x1f
				0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, // 0x20 - 0x27
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x28 - 0x2f
			},
			{
				// addr: memo
				// 0x08-09: Class ID(#6 Circuit Pack)
				// 0x0a-0b: Managed entity ID(#257 ONU2-G)
				// 0x0c-0d: field mask, 0x0f00 means from 5th to 8th fields
				// are stored in this message
				// 0x0e-11: Vendor ID
				// 0x12: Administrative state
				// 0x13: Operational state
				// 0x14: Bridged or IP ind
				0x00, 0x06, 0x01, 0x01, 0x0f, 0x00, 0x42, 0x52, // 0x08 - 0x0f
				0x43, 0x4d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x10 - 0x17
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x18 - 0x1f
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x20 - 0x27
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x28 - 0x2f
			},
			{
				// addr: memo
				// 0x08-09: Class ID(#6 Circuit Pack)
				// 0x0a-0b: Managed entity ID(#257 ONU2-G)
				// 0x0c-0d: field mask, 0x00f8 means from 9th to 13th fields
				// are stored in this message
				// 0x0e-21: Equipment ID
				// 0x22: Card configuration
				// 0x23: Total T-CONT buffer number
				// 0x24: Total priority queue number
				// 0x25: Total traffic scheduler number
				0x00, 0x06, 0x01, 0x01, 0x00, 0xf8, 0x20, 0x20, // 0x08 - 0x0f
				0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, // 0x10 - 0x17
				0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, // 0x18 - 0x1f
				0x20, 0x20, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, // 0x20 - 0x27
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x28 - 0x2f
			},
			{
				// addr: memo
				// 0x08-09: Class ID(#6 Circuit Pack)
				// 0x0a-0b: Managed entity ID(#257 ONU2-G)
				// 0x0c-0d: field mask, 0x0004 means from 14th field
				// is stored in this message
				// 0x0e-0x11: Power shed override
				0x00, 0x06, 0x01, 0x01, 0x00, 0x04, 0x00, 0x00, // 0x08 - 0x0f
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x10 - 0x17
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x18 - 0x1f
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x20 - 0x27
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x28 - 0x2f
			},
			{
				// addr: memo
				// 0x08-09: Class ID(#6 Circuit Pack)
				// 0x0a-0b: Managed entity ID(#384)
				// 0x0c-0d: field mask, 0xf000 means first 4 fields
				// are stored in this message
				// 0x0e: Type, 0xee means XG-PON10G10
				// 0x0f: Number of ports
				// 0x10-17: Serial Number
				// 0x18-25: Version
				0x00, 0x06, 0x01, 0x80, 0xf0, 0x00, 0xee, 0x01,
				0x49, 0x53, 0x4b, 0x54, 0x71, 0xe8, 0x00, 0x80,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Circuit Pack-#384
				0x00, 0x06, 0x01, 0x80, 0x0f, 0x00, 0x42, 0x52,
				0x43, 0x4d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Circuit Pack-#384
				0x00, 0x06, 0x01, 0x80, 0x00, 0xf8, 0x20, 0x20,
				0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
				0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
				0x20, 0x20, 0x00, 0x08, 0x40, 0x10, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Circuit Pack-#384
				0x00, 0x06, 0x01, 0x80, 0x00, 0x04, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Physical Path Termination Point Ethernet UNI(port#1)
				// addr: memo
				// 0x08-09: Class ID(#11 Physical Path Termination Point Ethernet UNI)
				// 0x0a: slot ID
				// 0x0b: port ID
				// 0x0c-0d: field mask, 0xfffe means first 15 fields
				// are stored in this message
				// 0x0e: Expected type
				// 0x0f: Sensed type, 0x2f means 10/100/1000 BASE-T
				// 0x10: Auto detection configuration, 0x00 means Rate/Duplex = Auto/Auto
				// 0x11: Ethernet loopback configulation
				// 0x12: Administrative state(0: unlocks, 1: locks)
				// 0x13: Operational state(0: enabled, 1: disabled)
				// 0x14: Configuration ind, 0x03 means Gigabit Ethernet full duplex
				// 0x15-16: Max frame size
				// 0x17: DTE or DCE ind, 0x00 means DCE or MDI-X
				// 0x18-19: Pause time
				// 0x1a: Bridged or IP ind, 0x02 means Depends on the parent circuit pack
				// 0x1b: ARC
				// 0x1c: ARC interval
				// 0x1d: PPPoE filter
				// 0x1e: Power control
				0x00, 0x0b, 0x01, 0x01, 0xff, 0xfe, 0x00, 0x2f, // 0x08 - 0x0f
				0x00, 0x00, 0x00, 0x00, 0x03, 0x05, 0xee, 0x00, // 0x10 - 0x17
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x18 - 0x1f
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x20 - 0x27
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x28 - 0x2f
			},
			{ // Physical Path Termination Point Ethernet UNI(port#2)
				0x00, 0x0b, 0x01, 0x02, 0xff, 0xfe, 0x00, 0x2f,
				0x00, 0x00, 0x00, 0x00, 0x03, 0x05, 0xee, 0x00,
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Physical Path Termination Point Ethernet UNI(port#3)
				0x00, 0x0b, 0x01, 0x03, 0xff, 0xfe, 0x00, 0x2f,
				0x00, 0x00, 0x00, 0x00, 0x03, 0x05, 0xee, 0x00,
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Physical Path Termination Point Ethernet UNI(port#4)
				0x00, 0x0b, 0x01, 0x04, 0xff, 0xfe, 0x00, 0x2f,
				0x00, 0x00, 0x00, 0x00, 0x03, 0x05, 0xee, 0x00,
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // T-CONT(T-CONT#1)
				// addr: memo
				// 0x08-09: Class ID(#262 T-CONT)
				// 0x0a: slot ID
				// 0x0b: T-CONT ID
				// 0x0c-0d: field mask, 0xe000 means first 3 fields
				// are stored in this message
				// 0x0e-0f: Alloc-ID
				// 0x10: Deprecated
				// 0x11: Policy
				0x01, 0x06, 0x80, 0x01, 0xe0, 0x00, 0xff, 0xff, // 0x08 - 0x0f
				0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x10 - 0x17
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // T-CONT(T-CONT#2)
				0x01, 0x06, 0x80, 0x02, 0xe0, 0x00, 0xff, 0xff,
				0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // T-CONT(T-CONT#3)
				0x01, 0x06, 0x80, 0x03, 0xe0, 0x00, 0xff, 0xff,
				0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // T-CONT(T-CONT#4)
				0x01, 0x06, 0x80, 0x04, 0xe0, 0x00, 0xff, 0xff,
				0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // T-CONT(T-CONT#5)
				0x01, 0x06, 0x80, 0x05, 0xe0, 0x00, 0xff, 0xff,
				0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // T-CONT(T-CONT#6)
				0x01, 0x06, 0x80, 0x06, 0xe0, 0x00, 0xff, 0xff,
				0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // T-CONT(T-CONT#7)
				0x01, 0x06, 0x80, 0x07, 0xe0, 0x00, 0xff, 0xff,
				0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // T-CONT(T-CONT#8)
				0x01, 0x06, 0x80, 0x08, 0xe0, 0x00, 0xff, 0xff,
				0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // ANI-G(port#1)
				// addr: memo
				// 0x08-09: Class ID(#263 ANI-G)
				// 0x0a: slot ID
				// 0x0b: port ID
				// 0x0c-0d: field mask, 0xffff means first 16 fields
				// are stored in this message
				// 0x0e: SR indication
				// 0x0f-10: Total T-CONT number
				// 0x11-12: GEM block length
				// 0x13: Piggyback DBA reporting
				// 0x14: Deprecated
				// 0x15: SF threshold
				// 0x16: SD threshold
				// 0x17: ARC
				// 0x18: ARC interval
				// 0x19-0x1a: Optical signal level
				// 0x1b: Lower optical threshold
				// 0x1c: Upper optical threshold
				// 0x1d-0x1e: ONU response time
				// 0x1f-0x20: Transmit optical level
				// 0x21: Lower transmit power threshold
				// 0x22: Upper transmit power threshold
				0x01, 0x07, 0x80, 0x01, 0xff, 0xff, 0x01, 0x00, // 0x08 - 0x0f
				0x08, 0x00, 0x30, 0x00, 0x00, 0x05, 0x09, 0x00, // 0x10 - 0x17
				0x00, 0xe0, 0x54, 0xff, 0xff, 0x00, 0x00, 0x0c, // 0x18 - 0x1f
				0x63, 0x81, 0x81, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x20 - 0x27
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x28 - 0x2f
			},
			{ // UNI-G(port#1)
				// addr: memo
				// 0x08-09: Class ID(#264 UNI-G)
				// 0x0a-0b: Managed entity ID
				// 0x0c-0d: field mask, 0xf800 means first 5 fields
				// are stored in this message
				// 0x0e-0f: Deprecated
				// 0x10: Administrative state(0: unlocks, 1: locks)
				// 0x11: Management capability
				// 0x12-13: Non-OMCI management identifier
				// 0x14-15: Relay agent options
				0x01, 0x08, 0x01, 0x01, 0xf8, 0x00, 0x00, 0x00, // 0x08 - 0x0f
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x10 - 0x17
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x18 - 0x1f
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x20 - 0x27
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x28 - 0x2f
			},
			/*
				{ // UNI-G(port#2)
					0x01, 0x08, 0x01, 0x02, 0xf8, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				},
				{ // UNI-G(port#3)
					0x01, 0x08, 0x01, 0x03, 0xf8, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				},
				{ // UNI-G(port#4)
					0x01, 0x08, 0x01, 0x04, 0xf8, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				},
			*/
			{ // Priority Queue(downstream-queue#1)
				// addr: memo
				// 0x08-09: Class ID(#277 Priority Queue)
				// 0x0a-0b: Managed entity ID(most significant bit represents the direction (1: upstream, 0: downstream))
				// 0x0c-0d: field mask, 0x000f means from 13th to 16th fields
				// are stored in this message
				// 0x0e-0f: Deprecated
				0x01, 0x15, 0x00, 0x01, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#1)
				0x01, 0x15, 0x00, 0x01, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x01, 0x00, 0x00, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#1)
				0x01, 0x15, 0x80, 0x01, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#1)
				0x01, 0x15, 0x80, 0x01, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x01, 0x00, 0x00, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#2)
				0x01, 0x15, 0x00, 0x02, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#2)
				0x01, 0x15, 0x00, 0x02, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x01, 0x00, 0x01, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#2)
				0x01, 0x15, 0x80, 0x02, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#2)
				0x01, 0x15, 0x80, 0x02, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x01, 0x00, 0x01, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#3)
				0x01, 0x15, 0x00, 0x03, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#3)
				0x01, 0x15, 0x00, 0x03, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x01, 0x00, 0x02, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#3)
				0x01, 0x15, 0x80, 0x03, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#3)
				0x01, 0x15, 0x80, 0x03, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x01, 0x00, 0x02, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#4)
				0x01, 0x15, 0x00, 0x04, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#4)
				0x01, 0x15, 0x00, 0x04, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x01, 0x00, 0x03, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#4)
				0x01, 0x15, 0x80, 0x04, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#4)
				0x01, 0x15, 0x80, 0x04, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x01, 0x00, 0x03, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#5)
				0x01, 0x15, 0x00, 0x05, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#5)
				0x01, 0x15, 0x00, 0x05, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x01, 0x00, 0x04, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#5)
				0x01, 0x15, 0x80, 0x05, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#5)
				0x01, 0x15, 0x80, 0x05, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x01, 0x00, 0x04, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#6)
				0x01, 0x15, 0x00, 0x06, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#6)
				0x01, 0x15, 0x00, 0x06, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x01, 0x00, 0x05, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#6)
				0x01, 0x15, 0x80, 0x06, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#6)
				0x01, 0x15, 0x80, 0x06, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x01, 0x00, 0x05, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#7)
				0x01, 0x15, 0x00, 0x07, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#7)
				0x01, 0x15, 0x00, 0x07, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x01, 0x00, 0x06, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#7)
				0x01, 0x15, 0x80, 0x07, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#7)
				0x01, 0x15, 0x80, 0x07, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x01, 0x00, 0x06, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#8)
				0x01, 0x15, 0x00, 0x08, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#8)
				0x01, 0x15, 0x00, 0x08, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x01, 0x00, 0x07, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#8)
				0x01, 0x15, 0x80, 0x08, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#8)
				0x01, 0x15, 0x80, 0x08, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x01, 0x00, 0x07, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#9)
				0x01, 0x15, 0x00, 0x09, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#9)
				0x01, 0x15, 0x00, 0x09, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x02, 0x00, 0x00, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#9)
				0x01, 0x15, 0x80, 0x09, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#9)
				0x01, 0x15, 0x80, 0x09, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x02, 0x00, 0x00, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#10)
				0x01, 0x15, 0x00, 0x0a, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#10)
				0x01, 0x15, 0x00, 0x0a, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x02, 0x00, 0x01, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#10)
				0x01, 0x15, 0x80, 0x0a, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#10)
				0x01, 0x15, 0x80, 0x0a, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x02, 0x00, 0x01, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#11)
				0x01, 0x15, 0x00, 0x0b, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#11)
				0x01, 0x15, 0x00, 0x0b, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x02, 0x00, 0x02, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#11)
				0x01, 0x15, 0x80, 0x0b, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#11)
				0x01, 0x15, 0x80, 0x0b, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x02, 0x00, 0x02, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#12)
				0x01, 0x15, 0x00, 0x0c, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#12)
				0x01, 0x15, 0x00, 0x0c, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x02, 0x00, 0x03, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#12)
				0x01, 0x15, 0x80, 0x0c, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#12)
				0x01, 0x15, 0x80, 0x0c, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x02, 0x00, 0x03, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#13)
				0x01, 0x15, 0x00, 0x0d, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#13)
				0x01, 0x15, 0x00, 0x0d, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x02, 0x00, 0x04, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#13)
				0x01, 0x15, 0x80, 0x0d, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#13)
				0x01, 0x15, 0x80, 0x0d, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x02, 0x00, 0x04, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#14)
				0x01, 0x15, 0x00, 0x0e, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#14)
				0x01, 0x15, 0x00, 0x0e, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x02, 0x00, 0x05, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#14)
				0x01, 0x15, 0x80, 0x0e, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#14)
				0x01, 0x15, 0x80, 0x0e, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x02, 0x00, 0x05, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#15)
				0x01, 0x15, 0x00, 0x0f, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#15)
				0x01, 0x15, 0x00, 0x0f, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x02, 0x00, 0x06, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#15)
				0x01, 0x15, 0x80, 0x0f, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#15)
				0x01, 0x15, 0x80, 0x0f, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x02, 0x00, 0x06, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#16)
				0x01, 0x15, 0x00, 0x10, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(downstream-queue#16)
				0x01, 0x15, 0x00, 0x10, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x02, 0x00, 0x07, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(upstream-queue#16)
				0x01, 0x15, 0x80, 0x10, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Priority Queue(uptream-queue#16)
				0x01, 0x15, 0x80, 0x10, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x02, 0x00, 0x07, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x11, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x11, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x03, 0x00, 0x00, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x11, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x11, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x03, 0x00, 0x00, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x12, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x12, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x03, 0x00, 0x01, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x12, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x12, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x03, 0x00, 0x01, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x13, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x13, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x03, 0x00, 0x02, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x13, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x13, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x03, 0x00, 0x02, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x14, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x14, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x03, 0x00, 0x03, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x14, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x14, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x03, 0x00, 0x03, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x15, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x15, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x03, 0x00, 0x04, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x15, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x15, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x03, 0x00, 0x04, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x16, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x16, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x03, 0x00, 0x05, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x16, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x16, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x03, 0x00, 0x05, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x17, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x17, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x03, 0x00, 0x06, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x17, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x17, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x03, 0x00, 0x06, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x18, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x18, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x03, 0x00, 0x07, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x18, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x18, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x03, 0x00, 0x07, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x19, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x19, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x04, 0x00, 0x00, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x19, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x19, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x04, 0x00, 0x00, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1a, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1a, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x04, 0x00, 0x01, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1a, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1a, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x04, 0x00, 0x01, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1b, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1b, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x04, 0x00, 0x02, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1b, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1b, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x04, 0x00, 0x02, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1c, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1c, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x04, 0x00, 0x03, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1c, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1c, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x04, 0x00, 0x03, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1d, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1d, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x04, 0x00, 0x04, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1d, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1d, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x04, 0x00, 0x04, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1e, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1e, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x04, 0x00, 0x05, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1e, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1e, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x04, 0x00, 0x05, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1f, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x1f, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x04, 0x00, 0x06, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1f, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x1f, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x04, 0x00, 0x06, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x20, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x20, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x04, 0x00, 0x07, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x20, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x20, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x04, 0x00, 0x07, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x21, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x21, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x05, 0x00, 0x00, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x21, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x21, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x05, 0x00, 0x00, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x22, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x22, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x05, 0x00, 0x01, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x22, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x22, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x05, 0x00, 0x01, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x23, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x23, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x05, 0x00, 0x02, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x23, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x23, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x05, 0x00, 0x02, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x24, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x24, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x05, 0x00, 0x03, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x24, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x24, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x05, 0x00, 0x03, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x25, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x25, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x05, 0x00, 0x04, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x25, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x25, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x05, 0x00, 0x04, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x26, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x26, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x05, 0x00, 0x05, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x26, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x26, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x05, 0x00, 0x05, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x27, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x27, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x05, 0x00, 0x06, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x27, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x27, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x05, 0x00, 0x06, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x28, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x28, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x05, 0x00, 0x07, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x28, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x28, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x05, 0x00, 0x07, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x29, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x29, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x06, 0x00, 0x00, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x29, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x29, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x06, 0x00, 0x00, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2a, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2a, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x06, 0x00, 0x01, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2a, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2a, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x06, 0x00, 0x01, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2b, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2b, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x06, 0x00, 0x02, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2b, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2b, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x06, 0x00, 0x02, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2c, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2c, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x06, 0x00, 0x03, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2c, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2c, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x06, 0x00, 0x03, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2d, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2d, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x06, 0x00, 0x04, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2d, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2d, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x06, 0x00, 0x04, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2e, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2e, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x06, 0x00, 0x05, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2e, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2e, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x06, 0x00, 0x05, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2f, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x2f, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x06, 0x00, 0x06, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2f, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x2f, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x06, 0x00, 0x06, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x30, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x30, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x06, 0x00, 0x07, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x30, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x30, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x06, 0x00, 0x07, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x31, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x31, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x07, 0x00, 0x00, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x31, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x31, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x07, 0x00, 0x00, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x32, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x32, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x07, 0x00, 0x01, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x32, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x32, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x07, 0x00, 0x01, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x33, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x33, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x07, 0x00, 0x02, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x33, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x33, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x07, 0x00, 0x02, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x34, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x34, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x07, 0x00, 0x03, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x34, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x34, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x07, 0x00, 0x03, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x35, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x35, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x07, 0x00, 0x04, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x35, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x35, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x07, 0x00, 0x04, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x36, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x36, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x07, 0x00, 0x05, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x36, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x36, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x07, 0x00, 0x05, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x37, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x37, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x07, 0x00, 0x06, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x37, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x37, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x07, 0x00, 0x06, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x38, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x38, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x07, 0x00, 0x07, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x38, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x38, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x07, 0x00, 0x07, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x39, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x39, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x08, 0x00, 0x00, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x39, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x39, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x08, 0x00, 0x00, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3a, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3a, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x08, 0x00, 0x01, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3a, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3a, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x08, 0x00, 0x01, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3b, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3b, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x08, 0x00, 0x02, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3b, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3b, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x08, 0x00, 0x02, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3c, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3c, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x08, 0x00, 0x03, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3c, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3c, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x08, 0x00, 0x03, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3d, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3d, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x08, 0x00, 0x04, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3d, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3d, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x08, 0x00, 0x04, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3e, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3e, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x08, 0x00, 0x05, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3e, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3e, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x08, 0x00, 0x05, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3f, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x3f, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x08, 0x00, 0x06, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3f, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x3f, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x08, 0x00, 0x06, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x40, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x00, 0x40, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
				0x08, 0x00, 0x07, 0x01, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x40, 0x00, 0x0f, 0xff, 0xff,
				0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
				0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{
				0x01, 0x15, 0x80, 0x40, 0xff, 0xf0, 0x00, 0x01,
				0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
				0x08, 0x00, 0x07, 0x80, 0x08, 0x01, 0x00, 0x01,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Traffic Scheduler
				0x01, 0x16, 0x80, 0x00, 0xf0, 0x00, 0x80, 0x08,
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Traffic Scheduler
				0x01, 0x16, 0x80, 0x00, 0xf0, 0x00, 0x80, 0x09,
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Traffic Scheduler
				0x01, 0x16, 0x80, 0x00, 0xf0, 0x00, 0x80, 0x0a,
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Traffic Scheduler
				0x01, 0x16, 0x80, 0x00, 0xf0, 0x00, 0x80, 0x0b,
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Traffic Scheduler
				0x01, 0x16, 0x80, 0x00, 0xf0, 0x00, 0x80, 0x0c,
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Traffic Scheduler
				0x01, 0x16, 0x80, 0x00, 0xf0, 0x00, 0x80, 0x0d,
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Traffic Scheduler
				0x01, 0x16, 0x80, 0x00, 0xf0, 0x00, 0x80, 0x0e,
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // Traffic Scheduler
				0x01, 0x16, 0x80, 0x00, 0xf0, 0x00, 0x80, 0x0f,
				0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			{ // ONU2-G
				0x01, 0x01, 0x00, 0x00, 0x07, 0xfc, 0x00, 0x40,
				0x08, 0x01, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x7f, 0x00, 0x00, 0x3f, 0x00, 0x01, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
		}[uploadIdx][:])
	case 0x44: // Create
		// meID
		// 0x0110: GAL Ethernet profile
		// 0x002d: MAC bridge service profile
		// 0x002f: MAC brige port configuration data
		// 0x00ab: Extended VLAN tagging operation configuration data
		// 0x0082: IEEE 802.1p mapper service profile
		// 0x010c: GEM port network CTP
		// 0x010a: GEM interworking termination point
		copy(omciInd.Pkt[contentIdx:], []byte{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		})
	case 0x48: // Set
		// meID
		// 0x0101: ONU2-G
		// 0x0100: ONU-G
		// 0x0108: UNI-G
		// 0x0106: T-CONT
		// 0x0115: Priority queue
		// 0x0082: IEEE 802.1p mapper service profile
		copy(omciInd.Pkt[contentIdx:], []byte{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		})
	}
	go func() {
		if err := dh.omciIndication(ctx, omciInd); err != nil {
			return
		}
	}()

	return nil
}

func (d *L2oamOltDevice) setActiveState(ctx context.Context, isActive bool) {
	d.Base.setActiveState(ctx, isActive)
}
func (d *L2oamOltDevice) startMountSequence(ctx context.Context, pkgType string, cmd *L2oamCmd) {
}
func (d *L2oamOltDevice) reboot(ctx context.Context) {
}
func (d *L2oamOltDevice) updateMap() {
	mapMutex.Lock()
	defer mapMutex.Unlock()
	_, ok := deviceMap[d.Base.DstMac]
	if ok {
		deviceMap[d.Base.DstMac] = d
	}
}
func (d *L2oamOltDevice) setL2oamCmd(cmd *L2oamCmd) {
	d.L2oamCmd = cmd
}
func (d *L2oamOltDevice) getL2oamCmd() *L2oamCmd {
	return d.L2oamCmd
}
func (d *L2oamOltDevice) setObjectContext(oc *l2oam.TomiObjectContext) {
	d.ObjectContext = oc
}
func (d *L2oamOltDevice) getObjectContext() *l2oam.TomiObjectContext {
	return d.ObjectContext
}
func (d *L2oamOltDevice) setReferenceTable(tbl *l2oam.GetTrafficControlReferenceTableRes) {
	d.ReferenceTable = tbl
}
func (d *L2oamOltDevice) getReferenceTable() *l2oam.GetTrafficControlReferenceTableRes {
	return d.ReferenceTable
}
func (d *L2oamOltDevice) setProfile(down *l2oam.SetGenericActionCreateRes, up *l2oam.SetGenericActionCreateRes) {
	d.DownProfile = down
	d.UpProfile = up
}
func (d *L2oamOltDevice) getProfile() (*l2oam.SetGenericActionCreateRes, *l2oam.SetGenericActionCreateRes) {
	return d.DownProfile, d.UpProfile
}
func (d *L2oamOltDevice) delete(ctx context.Context) {
	d.Base.delete(ctx)
}
func (d *L2oamOltDevice) setDeleteFlag() {
	d.Base.setDeleteFlag()
}
func (d *L2oamOltDevice) getDeleteFlag() bool {
	return d.Base.getDeleteFlag()
}
func (d *L2oamOltDevice) setAutonomousFlag(flag bool) {
	d.Base.setAutonomousFlag(flag)
	d.updateMap()
}
func (d *L2oamOltDevice) getAutonomousFlag() bool {
	return d.Base.getAutonomousFlag()
}
func (d *L2oamOltDevice) receiveEapol(etherPacket *layers.Ethernet) {
	d.Base.receiveEapol(etherPacket)
}

func (d *L2oamOltDevice) addFlow(ctx context.Context, cmd *L2oamCmd) error {
	d.autonomousExecMutex.Lock()
	defer d.autonomousExecMutex.Unlock()

	onu := FindL2oamDeviceByDeviceID(cmd.OnuDeviceID)
	if onu == nil {
		return fmt.Errorf(fmt.Sprintf("catnnot-get-onu-device. device-id:%s", cmd.OnuDeviceID))
	} else if !(onu.(*L2oamOnuDevice)).Base.isActive() {
		logger.Debug(ctx, fmt.Sprintf("onu is not active. device-id:%s", cmd.OnuDeviceID))
	} else {
		logger.Debug(ctx, fmt.Sprintf("add-flow-for-onu. device-id:%s", cmd.OnuDeviceID))

		oc := onu.getObjectContext()
		if oc == nil {
			logger.Error(ctx, fmt.Sprintf("catnnot-get-onu-object-context. device-id:%s", cmd.OnuDeviceID))
		} else {
			getTrafficControl := onu.getReferenceTable()
			if getTrafficControl == nil {
				getTrafficControl = sendGetTrafficControlReferenceTable(ctx, d, oc)
				if getTrafficControl == nil {
					return errors.New("L2oamAddFlow() sendGetTrafficControlReferenceTable() error. ")
				}
			}
			downProfile, upProfile := onu.getProfile()
			// if profiles are created, remove these profiles.
			if downProfile != nil {
				if err := sendSetActionDelete(ctx, d, downProfile.GetTrafficProfile(), l2oam.ActionTypeTrafficProfile); err != nil {
					logger.Error(ctx,
						fmt.Sprintf("L2oamAddFlow() Generic/Action Delete(downstream) error. %v", err))
				}
			}
			if upProfile != nil {
				if err := sendSetActionDelete(ctx, d, upProfile.GetTrafficProfile(), l2oam.ActionTypeTrafficProfile); err != nil {
					logger.Error(ctx,
						fmt.Sprintf("L2oamAddFlow() Generic/Action Delete(upstream) error. %v", err))
				}
			}
			downProfile = sendSetGenericActionCreate(ctx, d, l2oam.ActionTypeTrafficProfile)
			if downProfile == nil {
				return errors.New("L2oamAddFlow() Generic/Action Create(downstream) error")
			}
			upProfile = sendSetGenericActionCreate(ctx, d, l2oam.ActionTypeTrafficProfile)
			if upProfile == nil {
				return errors.New("L2oamAddFlow() Generic/Action Create(upstream) error")
			}
			onu.setReferenceTable(getTrafficControl)
			onu.setProfile(downProfile, upProfile)
			onu.updateMap()

			err01 := sendSetTrafficControl(ctx, d, getTrafficControl.GetReferenceControlDown(), downProfile.GetTrafficProfile())
			err02 := sendSetTrafficControl(ctx, d, getTrafficControl.GetReferenceControlUp(), upProfile.GetTrafficProfile())
			logger.Debug(ctx, fmt.Sprintf("traffic-control/traffic-profile results: %v, %v", err01, err02))

			err03 := sendSetPriority(ctx, d, downProfile.GetTrafficProfile())
			err04 := sendSetGuranteedRate(ctx, d, cmd.Cir, downProfile.GetTrafficProfile())
			err05 := sendSetGuranteedRate(ctx, d, cmd.Cir, upProfile.GetTrafficProfile())
			err06 := sendSetBestEffortRate(ctx, d, cmd.Pir, downProfile.GetTrafficProfile())
			err07 := sendSetBestEffortRate(ctx, d, cmd.Pir, upProfile.GetTrafficProfile())
			logger.Debug(ctx, fmt.Sprintf("traffic-profile-settings-results: %v, %v, %v, %v, %v",
				err03, err04, err05, err06, err07))
		}
	}

	return nil
}

func (d *L2oamOltDevice) updateFlow(ctx context.Context, cmd *L2oamCmd) error {
	d.autonomousExecMutex.Lock()
	defer d.autonomousExecMutex.Unlock()

	onuDeviceID := cmd.OnuDeviceID
	logger.Debugf(ctx, "updateFlow() Start. onu-device-id:%s, tpid:%x, vid%x",
		onuDeviceID, cmd.Tpid, cmd.Vid)

	onu := FindL2oamDeviceByDeviceID(onuDeviceID)
	if onu == nil {
		return errors.New("L2oamUpdateFlow() FindL2oamDeviceByDeviceId() error. onu not found")
	}
	var action *l2oam.TomiObjectContext
	var err error
	onuID := onu.getObjectContext()
	for _, settings := range d.L2SwitchDomainMap {
		if _, exists := settings.Onus[onuDeviceID]; exists {
			if err = sendSetActionInletEntryUsDel(ctx,
				d, settings.OcAction, settings.Tpid, settings.Vid, onuID); err != nil {
				logger.Errorf(ctx, "error:%v", err)
			}
			delete(settings.Onus, onuDeviceID)
		}
	}
	vidValue := byte2Int(cmd.Vid)
	settings, exists := d.L2SwitchDomainMap[vidValue]
	if !exists {
		action, err = sendSetGenericActionCreateForDS(ctx, d)
		if err != nil {
			logger.Errorf(ctx, "error:%v", err)
		}
		settings = &l2SwitchDomain{
			OcAction: action,
			Tpid:     cmd.Tpid,
			Vid:      cmd.Vid,
			Onus:     map[string]bool{},
		}
		d.L2SwitchDomainMap[vidValue] = settings
		if err := sendSetL2SwitchingDomainForDS(ctx, d, action, cmd.Tpid, cmd.Vid); err != nil {
			logger.Errorf(ctx, "error:%v", err)
		}
	} else {
		action = settings.OcAction
	}
	if err := sendSetDefaultOutlet(ctx, d, action, onuID); err != nil {
		logger.Errorf(ctx, "error:%v", err)
	}
	if err := sendSetL2SwitchingDomainForUS(ctx, d, action, cmd.Tpid, cmd.Vid, onuID); err != nil {
		logger.Errorf(ctx, "error:%v", err)
	}
	settings.Onus[onuDeviceID] = true

	logger.Debugf(ctx, "updateFlow() end. onu-device-id:%s, tpid:%x, vid%x",
		onuDeviceID, cmd.Tpid, cmd.Vid)

	return nil
}

func (d *L2oamOltDevice) removeFlow(ctx context.Context) error {
	logger.Debugf(ctx, "removeFlow() Start. device-id:%s", d.Base.DeviceID)

	for _, settings := range d.L2SwitchDomainMap {
		for onuDeviceID := range settings.Onus {
			onu := FindL2oamDeviceByDeviceID(onuDeviceID)
			if onu == nil {
				logger.Errorf(ctx, "onu not found. device-id:%s", onuDeviceID)
				continue
			}
			onuID := onu.getObjectContext()
			if err := sendSetActionInletEntryUsDel(ctx,
				d, settings.OcAction, settings.Tpid, settings.Vid, onuID); err != nil {
				logger.Errorf(ctx, "error:%v", err)
			}
		}
		settings.Onus = map[string]bool{}

		if err := sendSetActionInletEntryDsDel(ctx, d, settings.OcAction, settings.Tpid, settings.Vid); err != nil {
			logger.Errorf(ctx, "error:%v", err)
		}
		if err := sendSetActionDeleteStream(ctx, d, settings.OcAction); err != nil {
			logger.Errorf(ctx, "error:%v", err)
		}
	}
	d.L2SwitchDomainMap = map[uint32]*l2SwitchDomain{}

	return nil
}

// L2oamOnuDevice =================================
// L2oamOnuDevice provides functions for OLT device
// ================================================
type L2oamOnuDevice struct {
	Base              DeviceBase
	OnuID             uint32
	ObjectContext     *l2oam.TomiObjectContext
	ReferenceTable    *l2oam.GetTrafficControlReferenceTableRes
	DownProfile       *l2oam.SetGenericActionCreateRes
	UpProfile         *l2oam.SetGenericActionCreateRes
	PkgType           string
	keepAliveResponse chan string
	MacAddress        string
}

func (d *L2oamOnuDevice) getDeviceName() string {
	return d.Base.getDeviceName()
}
func (d *L2oamOnuDevice) send(message gopacket.SerializableLayer) error {
	return d.Base.send(message)
}
func (d *L2oamOnuDevice) waitResponse(timeoutSecond int) (*layers.Ethernet, error) {
	return d.Base.waitResponse(timeoutSecond)
}
func (d *L2oamOnuDevice) sendAndWait(message gopacket.SerializableLayer, timeoutSecond int) (*layers.Ethernet, error) {
	return d.Base.sendAndWait(message, timeoutSecond)
}
func (d *L2oamOnuDevice) recieve(etherPacket *layers.Ethernet) {
	d.Base.recieve(etherPacket)
}
func (d *L2oamOnuDevice) recieveKeepAlive(etherPacket *layers.Ethernet) {
	packet := &l2oam.OAMPDUInformation{}
	if err := packet.Decode(etherPacket.Payload); err != nil {
		return
	}
	if (l2oam.OnuPkgType == l2oam.OnuPkgTypeA) && (packet.IsOnuPkgA()) {
		logger.Info(context.Background(), fmt.Sprintf("[%s] recieveKeepAlive() Pkg type A.", d.getDeviceName()))
		if d.keepAliveResponse != nil {
			//d.keepAliveResponse <- "PkgTypeError"
			d.keepAliveResponse <- "success"
		}
		d.Base.recieveKeepAlive(etherPacket)
		d.PkgType = l2oam.OnuPkgTypeA
	} else if (l2oam.OnuPkgType == l2oam.OnuPkgTypeB) && (packet.IsOnuPkgB()) {
		logger.Info(context.Background(), fmt.Sprintf("[%s] recieveKeepAlive() Pkg type B.", d.getDeviceName()))
		if d.keepAliveResponse != nil {
			d.keepAliveResponse <- "success"
		}
		d.Base.recieveKeepAlive(etherPacket)
		d.PkgType = l2oam.OnuPkgTypeB
	} else {
		logger.Error(context.Background(), fmt.Sprintf("[%s] recieveKeepAlive() Pkg type error. support type=%s, isOnuPkgA=%v, isOnuPkgB=%v", d.getDeviceName(), l2oam.OnuPkgType, packet.IsOnuPkgA(), packet.IsOnuPkgB()))
		if d.keepAliveResponse != nil {
			d.keepAliveResponse <- "PkgTypeError"
		}
	}
}

func (d *L2oamOnuDevice) startKeepAlive() {
	d.Base.startKeepAlive()
}
func (d *L2oamOnuDevice) stopKeepAlive() {
	d.Base.stopKeepAlive()
}
func (d *L2oamOnuDevice) setActiveState(ctx context.Context, isActive bool) {
	d.Base.setActiveState(ctx, isActive)
}
func (d *L2oamOnuDevice) update(mac string, deviceID string, onuID uint32) {
	mapMutex.Lock()
	defer mapMutex.Unlock()
	d.OnuID = onuID
	d.Base.DeviceID = deviceID
	deviceMap[mac] = d
	deviceMacMap[deviceID] = mac
}

func (d *L2oamOnuDevice) startMountSequence(ctx context.Context, pkgType string, cmd *L2oamCmd) {
	if pkgType == l2oam.OnuPkgTypeA {
		d.startMountSequencePkgA(ctx, pkgType, cmd)
		return
	}
	go func() {
		logger.Debug(ctx, fmt.Sprintf("[%s] startMountSequence(). MacAddress=%s", d.getDeviceName(), d.Base.DstMac))
		// wait until the onu keep-alive state transits to step3
		for {
			if d.Base.KeepAliveStatus != KeepAliveStep3 {
				time.Sleep(1 * time.Second)
			} else {
				logger.Debug(ctx, fmt.Sprintf("[%s] startMountSequence(). onu active status. start sequence. MacAddress=%s", d.getDeviceName(), d.Base.DstMac))
				break
			}
		}

		// ONU System Info(get request)
		_, err := d.sendAndWait(l2oam.GenerateOnuSystemInfo(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateOnuSystemInfo() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// ONU Config
		if pkgType == l2oam.OnuPkgTypeB {
			_, err = d.sendAndWait(l2oam.GenerateOnuConfig(pkgType), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateOnuConfig() Send Error: %v", d.getDeviceName(), err))
				return
			}
		}
		// FEC Mode
		_, err = d.sendAndWait(l2oam.GenerateFecMode(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateFecMode() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Encryption Mode
		max := 2
		if pkgType == l2oam.OnuPkgTypeB {
			max = 1
		}
		for index := 0; index < max; index++ {
			_, err = d.sendAndWait(l2oam.GenerateEncryptionMode(pkgType, index+1), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateEncryptionMode(%v) Send Error: %v", d.getDeviceName(), index+1, err))
				return
			}
		}

		// Dyn Learning Mode
		max = 3
		if pkgType == l2oam.OnuPkgTypeB {
			max = 1
		}
		for index := 0; index < max; index++ {
			_, err = d.sendAndWait(l2oam.GenerateDynLearningMode(pkgType, index+1), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateDynLearningMode(%v) Send Error: %v", d.getDeviceName(), index+1, err))
				return
			}
		}
		// Ponp Temperature
		_, err = d.sendAndWait(l2oam.GeneratePonpTemperature(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GeneratePonpTemperature() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Ponp Optical TRx Supply Voltage
		_, err = d.sendAndWait(l2oam.GeneratePonpOptTRxSupplyVoltage(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GeneratePonpOptTRxSupplyVoltage() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Ponp Optical Tx Bias Current
		_, err = d.sendAndWait(l2oam.GeneratePonpOptTxBiasCurrent(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GeneratePonpOptTxBiasCurrent() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Ponp Optical Tx Output Power
		_, err = d.sendAndWait(l2oam.GeneratePonpOptTxOutputPower(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GeneratePonpOptTxOutputPower() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Ponp Optical Rx Output Power
		_, err = d.sendAndWait(l2oam.GeneratePonpOptRxInputPower(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GeneratePonpOptRxInputPower() Send Error: %v", d.getDeviceName(), err))
			return
		}

		// Onu Priority Queue Count
		max = 2
		if pkgType == l2oam.OnuPkgTypeB {
			max = 1
		}
		for index := 0; index < max; index++ {
			_, err = d.sendAndWait(l2oam.GenerateOnuPriorityQueueCount(pkgType, index+1), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateOnuPriorityQueueCount(%v) Send Error: %v", d.getDeviceName(), index+1, err))
				return
			}
		}
		// Unip Target Queue Priority
		if pkgType == l2oam.OnuPkgTypeB {
			_, err = d.sendAndWait(l2oam.GenerateUnipTargetQueuePriority(pkgType), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateUnipTargetQueuePriority() Send Error: %v", d.getDeviceName(), err))
				return
			}
		}
		// Vlan Pon VID Value
		max = 2
		if pkgType == l2oam.OnuPkgTypeB {
			max = 1
		}
		for index := 0; index < max; index++ {
			_, err = d.sendAndWait(l2oam.GenerateSetVlanPonVIDValue(pkgType, index+1, cmd.Itpid, cmd.Ivid), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateSetVlanPonVIDValue(%v) Send Error: %v", d.getDeviceName(), index+1, err))
				return
			}
		}
		// Vlan Tag Filter
		max = 0
		if pkgType == l2oam.OnuPkgTypeB {
			max = 2
		}
		for index := 0; index < max; index++ {
			_, err = d.sendAndWait(l2oam.GenerateSetVlanTagFilter(pkgType, index+1, cmd.Itpid), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateSetVlanTagFilter(%v) Send Error: %v", d.getDeviceName(), index+1, err))
				return
			}
		}
		// Frame Processing
		/*if pkgType == ONU_PKG_TYPE_A {
			_, err = d.sendAndWait(l2oam.GeneateSetFrameProcessing(pkgType), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GeneateOnuSysInfo() Send Error: %v", d.getDeviceName(), err))
				return
			}
		}*/
		// Qos Policer Rate
		max = 1
		if pkgType == l2oam.OnuPkgTypeB {
			max = 1
		}
		for index := 0; index < max; index++ {
			_, err = d.sendAndWait(l2oam.GenerateSetQosPolicerRate(pkgType, index+1), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateSetQosPolicerRate(%v) Send Error: %v", d.getDeviceName(), index+1, err))
				return
			}
		}
		// Qos Policer Burst
		if pkgType == l2oam.OnuPkgTypeB {
			_, err = d.sendAndWait(l2oam.GenerateSetQosPolicerBurst(pkgType), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateSetQosPolicerBurst() Send Error: %v", d.getDeviceName(), err))
				return
			}
		}
		// Unip Info
		_, err = d.sendAndWait(l2oam.GenerateGetUnipInfo(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateGetUnipInfo() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Unip Link Mode
		_, err = d.sendAndWait(l2oam.GenerateSetUnipLinkMode(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateSetUnipLinkMode() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Unip TOSCOS Conversion
		_, err = d.sendAndWait(l2oam.GenerateUnipTOSCOSConversion(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateUnipTOSCOSConversion() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// ONU Errored Frame Window
		_, err = d.sendAndWait(l2oam.GenerateOnuErroredFrameWindow(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateOnuErroredFrameWindow() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// ONU Errord Frame Threshold
		_, err = d.sendAndWait(l2oam.GenerateOnuErroredFrameWindow(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateOnuErroredFrameWindow() Send Error: %v", d.getDeviceName(), err))
			return
		}
		logger.Debug(ctx, fmt.Sprintf("[%s] startMountSequence(). Normal End.", d.getDeviceName()))
	}()
}
func (d *L2oamOnuDevice) startMountSequencePkgA(ctx context.Context, pkgType string, cmd *L2oamCmd) {
	go func() {
		logger.Debug(ctx, fmt.Sprintf("[%s] startMountSequence(). MacAddress=%s", d.getDeviceName(), d.Base.DstMac))
		// wait until the onu keep-alive state transits to step3
		for {
			if d.Base.KeepAliveStatus != KeepAliveStep3 {
				time.Sleep(1 * time.Second)
			} else {
				logger.Debug(ctx, fmt.Sprintf("[%s] startMountSequence(). onu active status. start sequence. MacAddress=%s", d.getDeviceName(), d.Base.DstMac))
				break
			}
		}

		// // Encryption Mode
		// _, err := d.sendAndWait(l2oam.GenerateEncryptionMode(pkgType, 1), ResponseTimer)
		// if err != nil {
		// 	logger.Error(ctx, fmt.Sprintf("[%s] GenerateEncryptionMode(%v) Send Error: %v", d.getDeviceName(), 1, err))
		// 	return
		// }

		// ONU System Info(get request)
		_, err := d.sendAndWait(l2oam.GenerateOnuSystemInfo(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateOnuSystemInfo() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// ONU Config
		if pkgType == l2oam.OnuPkgTypeB {
			_, err = d.sendAndWait(l2oam.GenerateOnuConfig(pkgType), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateOnuConfig() Send Error: %v", d.getDeviceName(), err))
				return
			}
		}
		// FEC Mode
		_, err = d.sendAndWait(l2oam.GenerateFecMode(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateFecMode() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Encryption Mode
		_, err = d.sendAndWait(l2oam.GenerateEncryptionMode(pkgType, 2), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateEncryptionMode(%v) Send Error: %v", d.getDeviceName(), 2, err))
			return
		}

		// Dyn Learning Mode
		max := 3
		if pkgType == l2oam.OnuPkgTypeB {
			max = 1
		}
		for index := 0; index < max; index++ {
			_, err = d.sendAndWait(l2oam.GenerateDynLearningMode(pkgType, index+1), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateDynLearningMode(%v) Send Error: %v", d.getDeviceName(), index+1, err))
				return
			}
		}
		// Ponp Temperature
		_, err = d.sendAndWait(l2oam.GeneratePonpTemperature(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GeneratePonpTemperature() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Ponp Optical TRx Supply Voltage
		_, err = d.sendAndWait(l2oam.GeneratePonpOptTRxSupplyVoltage(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GeneratePonpOptTRxSupplyVoltage() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Ponp Optical Tx Bias Current
		_, err = d.sendAndWait(l2oam.GeneratePonpOptTxBiasCurrent(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GeneratePonpOptTxBiasCurrent() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Ponp Optical Tx Output Power
		_, err = d.sendAndWait(l2oam.GeneratePonpOptTxOutputPower(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GeneratePonpOptTxOutputPower() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Ponp Optical Rx Output Power
		_, err = d.sendAndWait(l2oam.GeneratePonpOptRxInputPower(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GeneratePonpOptRxInputPower() Send Error: %v", d.getDeviceName(), err))
			return
		}

		// Onu Priority Queue Count
		max = 2
		if pkgType == l2oam.OnuPkgTypeB {
			max = 1
		}
		for index := 0; index < max; index++ {
			_, err = d.sendAndWait(l2oam.GenerateOnuPriorityQueueCount(pkgType, index+1), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateOnuPriorityQueueCount(%v) Send Error: %v", d.getDeviceName(), index+1, err))
				return
			}
		}
		// Unip Target Queue Priority
		if pkgType == l2oam.OnuPkgTypeB {
			_, err = d.sendAndWait(l2oam.GenerateUnipTargetQueuePriority(pkgType), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateUnipTargetQueuePriority() Send Error: %v", d.getDeviceName(), err))
				return
			}
		}
		// Vlan Pon VID Value
		max = 2
		if pkgType == l2oam.OnuPkgTypeB {
			max = 1
		}
		for index := 0; index < max; index++ {
			_, err = d.sendAndWait(l2oam.GenerateSetVlanPonVIDValue(pkgType, index+1, cmd.Itpid, cmd.Ivid), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateSetVlanPonVIDValue(%v) Send Error: %v", d.getDeviceName(), index+1, err))
				return
			}
		}
		// Vlan Tag Filter
		max = 0
		if pkgType == l2oam.OnuPkgTypeB {
			max = 2
		}
		for index := 0; index < max; index++ {
			_, err = d.sendAndWait(l2oam.GenerateSetVlanTagFilter(pkgType, index+1, cmd.Itpid), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateSetVlanTagFilter(%v) Send Error: %v", d.getDeviceName(), index+1, err))
				return
			}
		}
		// Frame Processing
		/*if pkgType == ONU_PKG_TYPE_A {
			_, err = d.sendAndWait(l2oam.GeneateSetFrameProcessing(pkgType), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GeneateOnuSysInfo() Send Error: %v", d.getDeviceName(), err))
				return
			}
		}*/
		// Qos Policer Rate
		max = 1
		if pkgType == l2oam.OnuPkgTypeB {
			max = 1
		}
		for index := 0; index < max; index++ {
			_, err = d.sendAndWait(l2oam.GenerateSetQosPolicerRate(pkgType, index+1), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateSetQosPolicerRate(%v) Send Error: %v", d.getDeviceName(), index+1, err))
				return
			}
		}
		// Qos Policer Burst
		if pkgType == l2oam.OnuPkgTypeB {
			_, err = d.sendAndWait(l2oam.GenerateSetQosPolicerBurst(pkgType), ResponseTimer)
			if err != nil {
				logger.Error(ctx, fmt.Sprintf("[%s] GenerateSetQosPolicerBurst() Send Error: %v", d.getDeviceName(), err))
				return
			}
		}
		// Unip Info
		_, err = d.sendAndWait(l2oam.GenerateGetUnipInfo(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateGetUnipInfo() Send Error: %v", d.getDeviceName(), err))
			return
		}
		// Traffic Enable
		_, err = d.sendAndWait(l2oam.GenerateSetTrafficEnable(pkgType), ResponseTimer)
		if err != nil {
			logger.Error(ctx, fmt.Sprintf("[%s] GenerateSetTrafficEnable() Send Error: %v", d.getDeviceName(), err))
			return
		}
		logger.Debug(ctx, fmt.Sprintf("[%s] startMountSequence(). Normal End.", d.getDeviceName()))
	}()
}

func (d *L2oamOnuDevice) reboot(ctx context.Context) {
	_, err := d.sendAndWait(l2oam.GenerateSetResetOnu(l2oam.OnuPkgType), ResponseTimer)
	if err != nil {
		logger.Error(ctx, fmt.Sprintf("[%s] reboot() Send Error: %v", d.getDeviceName(), err))
		return
	}
}

func (d *L2oamOnuDevice) updateMap() {
	mapMutex.Lock()
	defer mapMutex.Unlock()

	_, ok := deviceMap[d.Base.DstMac]
	if ok {
		deviceMap[d.Base.DstMac] = d
	}
}
func (d *L2oamOnuDevice) setL2oamCmd(cmd *L2oamCmd) {
}
func (d *L2oamOnuDevice) getL2oamCmd() *L2oamCmd {
	return nil
}
func (d *L2oamOnuDevice) setObjectContext(oc *l2oam.TomiObjectContext) {
	d.ObjectContext = oc
}
func (d *L2oamOnuDevice) getObjectContext() *l2oam.TomiObjectContext {
	return d.ObjectContext
}
func (d *L2oamOnuDevice) setReferenceTable(tbl *l2oam.GetTrafficControlReferenceTableRes) {
	d.ReferenceTable = tbl
}
func (d *L2oamOnuDevice) getReferenceTable() *l2oam.GetTrafficControlReferenceTableRes {
	return d.ReferenceTable
}
func (d *L2oamOnuDevice) setProfile(down *l2oam.SetGenericActionCreateRes, up *l2oam.SetGenericActionCreateRes) {
	d.DownProfile = down
	d.UpProfile = up
}
func (d *L2oamOnuDevice) getProfile() (*l2oam.SetGenericActionCreateRes, *l2oam.SetGenericActionCreateRes) {
	return d.DownProfile, d.UpProfile
}
func (d *L2oamOnuDevice) delete(ctx context.Context) {
	d.Base.delete(ctx)
}
func (d *L2oamOnuDevice) setDeleteFlag() {
	d.Base.setDeleteFlag()
}
func (d *L2oamOnuDevice) getDeleteFlag() bool {
	return d.Base.getDeleteFlag()
}
func (d *L2oamOnuDevice) setAutonomousFlag(flag bool) {
	d.Base.setAutonomousFlag(flag)
	d.updateMap()
}
func (d *L2oamOnuDevice) getAutonomousFlag() bool {
	return d.Base.getAutonomousFlag()
}
func (d *L2oamOnuDevice) receiveEapol(etherPacket *layers.Ethernet) {
	d.Base.receiveEapol(etherPacket)
}
func (d *L2oamOnuDevice) waitKeepALiveResponse() error {
	select {
	case result := <-d.keepAliveResponse:
		if result == "success" {
			return nil
		}
		return fmt.Errorf("waitKeepALiveResponse() error. reason=%s", result)

	case <-time.After(ResponseTimer * time.Second):
		return fmt.Errorf("waitKeepALiveResponse() error. reason=%s", "timeout")
	}
}

func (d *L2oamOnuDevice) addFlow(ctx context.Context, cmd *L2oamCmd) error {
	return errors.New("onu.addFlow: not implemented")
}

func (d *L2oamOnuDevice) updateFlow(ctx context.Context, cmd *L2oamCmd) error {
	return errors.New("onu.updateFlow: not implemented")
}

func (d *L2oamOnuDevice) removeFlow(ctx context.Context) error {
	return errors.New("onu.removeFlow: not implemented")
}

// macAddress must not have colons
/*
func isOnuEnable(ctx context.Context, macAddress string) (bool, string) {
	mac := convertMacString(macAddress)
	onulist, err := l2oam.ReadOnuStatusList()
	if err != nil {
		return false, ""
	}
	for _, v := range onulist {
		if v.MacAddress == mac {
			if v.AdminState == "ENABLED" {
				return true, v.ID
			}
		}
	}

	return false, ""
}
*/

func updateOltActiveStatus(ctx context.Context, dh *DeviceHandler, isActive bool) {
	logger.Debug(ctx, fmt.Sprintf("updateOltActiveStatus() isActive=%v", isActive))
	if dh != nil && dh.coreProxy != nil {
		var err error
		if isActive {
			err = dh.coreProxy.DeviceStateUpdate(ctx, dh.device.Id, voltha.ConnectStatus_REACHABLE, voltha.OperStatus_ACTIVE)
		} else {
			err = dh.coreProxy.DeviceStateUpdate(ctx, dh.device.Id, voltha.ConnectStatus_UNREACHABLE, voltha.OperStatus_ACTIVATING)
		}
		if err != nil {
			logger.Debug(ctx, fmt.Sprintf("updateOltActiveStatus() error=%v", err))
		}
	} else {
		logger.Debug(ctx, fmt.Sprintf("updateOltActiveStatus() deviceId=%s, isActive=%v", dh.device.Id, isActive))
	}
}

// macAddress must not have colons
/*
func updateOnuActiveStatus(ctx context.Context, macAddress string, isActive bool, dh *DeviceHandler) {
	mac := convertMacString(macAddress)
	onulist, err := l2oam.ReadOnuStatusList()
	logger.Debug(ctx, fmt.Sprintf("updateOnuActiveStatus() ReadOnuList() onulist=%v, err=%s, macAddress=%s, mac=%s, isActive=%v", onulist, err, macAddress, mac, isActive))
	if err != nil {
		return
	}
	for i, v := range onulist {
		if v.MacAddress == mac {
			if isActive {
				onulist[i].OpeState = "ACTIVE"
				onulist[i].ConnectState = "REACHABLE"
				dh.coreProxy.DeviceStateUpdate(ctx, v.ID, voltha.ConnectStatus_REACHABLE, voltha.OperStatus_ACTIVE)
			} else {
				onulist[i].OpeState = "FAILED"
				onulist[i].ConnectState = "UNREACHABLE"
				dh.coreProxy.DeviceStateUpdate(ctx, v.ID, voltha.ConnectStatus_UNREACHABLE, voltha.OperStatus_FAILED)
			}
			logger.Debug(ctx, fmt.Sprintf("updateOnuActiveStatus() WriteOnuList() macaddress=%s list=%v", mac, onulist))
			//l2oam.WriteOnuStatusList(onulist)
			break
		}
	}
}
*/

func updateOnuActiveStatus2(ctx context.Context, dh *DeviceHandler, isActive bool, deviceID string) {
	logger.Debug(ctx, fmt.Sprintf("updateOnuActiveStatus2() deviceId=%s, isActive=%v", deviceID, isActive))
	if dh != nil && dh.coreProxy != nil {
		var err error
		if isActive {
			err = dh.coreProxy.DeviceStateUpdate(ctx, deviceID, voltha.ConnectStatus_REACHABLE, voltha.OperStatus_ACTIVE)
		} else {
			err = dh.coreProxy.DeviceStateUpdate(ctx, deviceID, voltha.ConnectStatus_UNREACHABLE, voltha.OperStatus_ACTIVATING)
		}
		if err != nil {
			logger.Debug(ctx, fmt.Sprintf("updateOnuActiveStatus2() error=%v", err))
		}
	} else {
		logger.Debug(ctx, fmt.Sprintf("updateOnuActiveStatus2() deviceId=%s, isActive=%v", deviceID, isActive))
	}
}
func convertMacString(mac string) string {
	macbytes := []byte(mac)
	colonbytes := []byte(":")
	colon := colonbytes[0]

	resbytes := []byte{}
	resbytes = append(resbytes, macbytes[0])
	resbytes = append(resbytes, macbytes[1])
	resbytes = append(resbytes, colon)
	resbytes = append(resbytes, macbytes[2])
	resbytes = append(resbytes, macbytes[3])
	resbytes = append(resbytes, colon)
	resbytes = append(resbytes, macbytes[4])
	resbytes = append(resbytes, macbytes[5])
	resbytes = append(resbytes, colon)
	resbytes = append(resbytes, macbytes[6])
	resbytes = append(resbytes, macbytes[7])
	resbytes = append(resbytes, colon)
	resbytes = append(resbytes, macbytes[8])
	resbytes = append(resbytes, macbytes[9])
	resbytes = append(resbytes, colon)
	resbytes = append(resbytes, macbytes[10])
	resbytes = append(resbytes, macbytes[11])
	return string(resbytes)

}
func monitoringOnuStatus() {
	go WatchStatus(context.Background(), nil)
}
func byte2Int(v []byte) uint32 {
	switch len(v) {
	case 1:
		return uint32(v[0])
	case 2:
		return uint32(binary.BigEndian.Uint16(v[:]))
	case 4:
		return binary.BigEndian.Uint32(v[:])
	default:
		return 0
	}
}
