/*
<:copyright-BRCM:2016:DUAL/GPL:standard

   Broadcom Proprietary and Confidential.(c) 2016 Broadcom
   All Rights Reserved

Unless you and Broadcom execute a separate written software license
agreement governing use of this software, this software is licensed
to you under the terms of the GNU General Public License version 2
(the "GPL"), available at http://www.broadcom.com/licenses/GPLv2.php,
with the following added to such license:

   As a special exception, the copyright holders of this software give
   you permission to link this software with independent modules, and
   to copy and distribute the resulting executable under terms of your
   choice, provided that you also meet, for each linked independent
   module, the terms and conditions of the license of that module.
   An independent module is a module which is not derived from this
   software.  The special exception does not apply to any modifications
   of the software.

Not withstanding the above, under no circumstances may you combine
this software in any way with any other Broadcom software provided
under a license other than the GPL, without Broadcom's express prior
written consent.

:>
 */


/*******************************************************************
 * bcmcli_session.h
 *
 * BCM CLI engine - session management
 *
 *******************************************************************/

#ifndef BCMCLI_SESSION_H

#define BCMCLI_SESSION_H

#include <bcmos_system.h>
#include <stdarg.h>
#include "bcmolt_string.h"

/** \defgroup bcmcli_session Management Session Control
 *
 * APIs in this header file allow to create/destroy management sessions.
 * Management session is characterized by its access level and also
 * input/output functions.
 * Management sessions allow managed entities in the system to communicate
 * with local or remote managers (e.g., local or remote shell or NMS)
 * @{
 */

/** Access rights */
typedef enum
{
    BCMCLI_ACCESS_GUEST,     /**< Guest. Doesn't have access to commands and directories registered with ADMIN rights */
    BCMCLI_ACCESS_ADMIN,     /**< Administrator: full access */
    BCMCLI_ACCESS_DEBUG,     /**< Administrator: full access + extended debug features */
} bcmcli_access_right;

/** Line edit mode */
typedef enum
{
    BCMCLI_LINE_EDIT_DEFAULT,/**< Enable line editing and history if CONFIG_EDITLINE is defined, disable otherwise */
    BCMCLI_LINE_EDIT_ENABLE, /**< Enable line editing. Requires CONFIG_EDITLINE define and libedit-dev library */
    BCMCLI_LINE_EDIT_DISABLE,/**< Disable line editing and history */
} bcmcli_line_edit_mode;


/** Management session handle
 */
typedef struct bcmcli_session bcmcli_session;


/** Session parameters structure.
 * See \ref bcmcli_session_open
 */
typedef struct bcmcli_session_parm
{
    const char *name;       /**< Session name */
    void *user_priv;        /**< Private user's data */

    /** Session's output function. NULL=use write(stdout)
     * returns the number of bytes written or <0 if error
     */
    int (*write)(bcmcli_session *session, const char *buf, uint32_t size);

    /** Session line input function. NULL=use default(stdin[+line edit) */
    char *(*gets)(bcmcli_session *session, char *buf, uint32_t size);

    /** Session char input function. NULL=use bcmos_getchar() */
    int (*get_char)(bcmcli_session *session);

    /** Fill the specified buffer with the prompt for this session (NULL-terminated). NULL = no prompt. */
    void (*get_prompt)(bcmcli_session *session, char *buf, uint32_t max_len);

    /** Access rights */
    bcmcli_access_right access_right;

    /** Line editing mode */
    bcmcli_line_edit_mode line_edit_mode;

    /** Extra data size to be allocated along with session control block.
     * The extra data is accessible using bcmcli_session_data().
     * Please note that if session is created using bcmcli_session_open(),
     * extra_size is reserved.
     * It can only be used for user context allocation if session is created
     * using bcmcli_session_open_user()
     */
    uint32_t extra_size;
} bcmcli_session_parm;


/** Open monitor session
 *
 * Monitor supports multiple simultaneous sessions with different
 * access rights.
 * Note that there already is a default session with full administrative rights,
 * that takes input from stdin and outputs to stdout.
 *
 * Please don't use parm.extra_size. This field is reserved.
 *
 * \param[in]   parm        Session parameters. Must not be allocated on the stack.
 * \param[out]  p_session   Session handle
 * \return
 *      0   =OK\n
 *      <0  =error code
 */
bcmos_errno bcmcli_session_open(const bcmcli_session_parm *parm, bcmcli_session **p_session);


/** Close monitor session.
 * \param[in]   session         Session handle
 */
void bcmcli_session_close(bcmcli_session *session );


/** Write function.
 * Write buffer to the current session.
 * \param[in]   session         Session handle. NULL=use stdout
 * \param[in]   buf             output buffer
 * \param[in]   size            number of bytes to be written
 * \return
 *  >=0 - number of bytes written\n
 *  <0  - output error
 */
int bcmcli_session_write(bcmcli_session *session, const char *buf, uint32_t size);


/** Read line
 * \param[in]       session         Session handle. NULL=use default
 * \param[in,out]   buf             input buffer
 * \param[in]       size            buf size
 * \return
 *      buf if successful
 *      NULL if EOF or error
 */
char *bcmcli_session_gets(bcmcli_session *session, char *buf, uint32_t size);


/** Print function.
 * Prints in the context of current session.
 * \param[in]   session         Session handle. NULL=use stdout
 * \param[in]   format          print format - as in printf
 */
void bcmcli_session_print(bcmcli_session *session, const char *format, ...)
#ifndef BCMCLI_SESSION_DISABLE_FORMAT_CHECK
__attribute__((format(printf, 2, 3)))
#endif
;


/** Print function.
 * Prints in the context of current session.
 * \param[in]   session         Session handle. NULL=use stdout
 * \param[in]   format          print format - as in printf
 * \param[in]   ap              parameters list. Undefined after the call
 */
void bcmcli_session_vprint(bcmcli_session *session, const char *format, va_list ap);

/** Print buffer in hexadecimal format
 * \param[in]   session         Session handle. NULL=use stdout
 * \param[in]   buffer          Buffer address
 * \param[in]   offset          Start offset in the buffer
 * \param[in]   count           Number of bytes to dump
 * \param[in]   indent          Optional indentation string
 */
void bcmcli_session_hexdump(bcmcli_session *session, const void *buffer, uint32_t offset, uint32_t count, const char *indent);

/** Get extra data associated with the session
 * \param[in]       session         Session handle. NULL=default session
 * \return extra_data pointer or NULL if there is no extra data
 */
void *bcmcli_session_data(bcmcli_session *session);


/** Get user_priv provided in session parameters when it was registered
 * \param[in]       session         Session handle. NULL=default session
 * \return usr_priv value
 */
void *bcmcli_session_user_priv(bcmcli_session *session);


/** Get session name
 * \param[in]       session         Session handle. NULL=use stdin
 * \return session name
 */
const char *bcmcli_session_name(bcmcli_session *session);


/** Get session access rights
 * \param[in]       session         Session handle. NULL=default debug session
 * \return session access right
 */
bcmcli_access_right bcmcli_session_access_right(bcmcli_session *session);

/** @} end of bcmcli_session group */

/** Low-level interface for when session is used outside CLI
 *
 * \param[in]   parm        Session parameters. Must not be allocated on the stack.
 * \param[out]  p_session   Session handle
 * \return
 *      0   =OK\n
 *      <0  =error code
 */
int bcmcli_session_open_user(const bcmcli_session_parm *parm, bcmcli_session **p_session);

/** Open a session that prints to the specified string
 */
bcmos_errno bcmcli_session_open_string(bcmcli_session **session, bcmolt_string *str);

/** Configure RAW input mode
 *
 * \param[in]       session         Session handle
 * \param[in]       is_raw          TRUE=enable raw mode, FALSE=disable raw mode
 * \return
 *      =0  - OK \n
 *      BCM_ERR_NOT_SUPPORTED - raw mode is not supported\n
 */
bcmos_errno bcmcli_session_raw_mode_set(bcmcli_session *session, bcmos_bool is_raw);

/** Context extension
 *
 * - if no command - display list of command or extend command
 * - if prev char is " "
 *      - if positional and next parm is enum - show/extends list of matching values
 *      - else - show/extend list of unused parameters
 *   else
 *      - if entering value and enum - show/extends list of matching values
 *      - else - show/extend list of matching unused parameters
 *
 * \param[in]       session         Session handle
 * \param[in]       input_string    String to be parsed
 * \param[out]      insert_str      String to insert at cursor position
 * \param[in]       insert_size     Insert buffer size
 * \return
 *      =0  - OK \n
 *      BCM_ERR_PARM - parsing error\n
 */
bcmos_errno bcmcli_extend(bcmcli_session *session, char *input_string,
    char *insert_str, uint32_t insert_size);


#ifdef BCMCLI_INTERNAL

#define BCMCLI_SESSION_OUTBUF_LEN 2048
#define BCMCLI_MAX_PROMPT_LEN 8

/* editline functionality */
/* If libedit is included - it takes precedence */
#ifdef CONFIG_LIBEDIT
#include <histedit.h>
#undef CONFIG_LINENOISE
#endif /* #ifdef CONFIG_LIBEDIT */

#ifdef CONFIG_LINENOISE
#include <linenoise.h>
#endif

/* Management session structure */
struct bcmcli_session
{
    bcmcli_session *next;
    bcmcli_session_parm parms;
    uint32_t magic;
#define BCMCLI_SESSION_MAGIC            (('s'<<24)|('e'<<16)|('s'<<8)|'s')
#define BCMCLI_SESSION_MAGIC_DEL        (('s'<<24)|('e'<<16)|('s'<<8)|'~')

    /* Line editing and history support */
#ifdef CONFIG_LIBEDIT
    EditLine *el;
    History *history;
    HistEvent histevent;
#endif
#ifdef CONFIG_LINENOISE
    linenoiseSession *ln_session;
#endif
    char outbuf[BCMCLI_SESSION_OUTBUF_LEN];
    char prompt_buf[BCMCLI_MAX_PROMPT_LEN];

    /* Followed by session data */
};
#endif

#endif /* #ifndef BCMCLI_SESSION_H */
