/*
<:copyright-BRCM:2016:DUAL/GPL:standard

   Broadcom Proprietary and Confidential.(c) 2016 Broadcom
   All Rights Reserved

Unless you and Broadcom execute a separate written software license
agreement governing use of this software, this software is licensed
to you under the terms of the GNU General Public License version 2
(the "GPL"), available at http://www.broadcom.com/licenses/GPLv2.php,
with the following added to such license:

   As a special exception, the copyright holders of this software give
   you permission to link this software with independent modules, and
   to copy and distribute the resulting executable under terms of your
   choice, provided that you also meet, for each linked independent
   module, the terms and conditions of the license of that module.
   An independent module is a module which is not derived from this
   software.  The special exception does not apply to any modifications
   of the software.

Not withstanding the above, under no circumstances may you combine
this software in any way with any other Broadcom software provided
under a license other than the GPL, without Broadcom's express prior
written consent.

:>
 */


/*******************************************************************
 * bcmcli.h
 *
 * CLI engine
 *
 *******************************************************************/

#ifndef BCMCLI_H

#define BCMCLI_H

#include <bcmos_system.h>
#include <bcmcli_session.h>
#include <bcmolt_buf.h>

/** \defgroup bcm_cli Broadcom CLI Engine
 * Broadcom CLI engine is used for all configuration and status monitoring.\n
 * It doesn't have built-in scripting capabilities (logical expressions, loops),
 * but can be used in combination with any available scripting language.\n
 * Broadcom CLI supports the following features:\n
 * - parameter number and type validation (simplifies command handlers development)
 * - parameter value range checking
 * - mandatory and optional parameters
 * - positional and named parameters
 * - parameters with default values
 * - enum parameters can have arbitrary values
 * - automatic command help generation
 * - automatic or user-defined command shortcuts
 * - command handlers return completion status to enable scripting
 * - multiple sessions
 * - session access rights
 * - extendible. Supports user-defined parameter types
 * - relatively low stack usage
 * @{
 */

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#define BCMCLI_MAX_SEARCH_SUBSTR_LENGTH 80

#define BCMCLI_ARRAY_EMPTY        "-"
#define BCMCLI_PARM_NO_VALUE      "_"
#define BCMCLI_ENUM_MASK_DEL_STR  "+"

/** Monitor entry handle
 */
typedef struct bcmcli_entry bcmcli_entry;

/* if BCMCLI_PARM_USERIO flag is set:
   low_val: t_userscanf_f function
   high_val: t_userprintf_f function
*/

/** Function parameter structure */
typedef struct bcmcli_cmd_parm bcmcli_cmd_parm;

/** Parameter type */
typedef enum
{
    BCMCLI_PARM_NONE,
    BCMCLI_PARM_DECIMAL,         /**< Decimal number */
    BCMCLI_PARM_DECIMAL64,       /**< Signed 64-bit decimal */
    BCMCLI_PARM_UDECIMAL,        /**< Unsigned decimal number */
    BCMCLI_PARM_UDECIMAL64,      /**< Unsigned 64-bit decimal number */
    BCMCLI_PARM_HEX,             /**< Hexadecimal number */
    BCMCLI_PARM_HEX64,           /**< 64-bit hexadecimal number */
    BCMCLI_PARM_NUMBER,          /**< Decimal number or hex number prefixed by 0x */
    BCMCLI_PARM_NUMBER64,        /**< 64bit decimal number or hex number prefixed by 0x */
    BCMCLI_PARM_UNUMBER,         /**< Unsigned decimal number or hex number prefixed by 0x */
    BCMCLI_PARM_UNUMBER64,       /**< Unsigned 64bit decimal number or hex number prefixed by 0x */
    BCMCLI_PARM_FLOAT,           /**< IEEE 32-bit floating-point number */
    BCMCLI_PARM_DOUBLE,          /**< IEEE 64-bit floating point number */
    BCMCLI_PARM_STRING,          /**< String */
    BCMCLI_PARM_ENUM,            /**< Enumeration */
    BCMCLI_PARM_ENUM_MASK,       /**< Bitmask created from enumeration values */
    BCMCLI_PARM_IP,              /**< IP address n.n.n.n */
    BCMCLI_PARM_IPV6,            /**< IPv6 address */
    BCMCLI_PARM_MAC,             /**< MAC address xx:xx:xx:xx:xx:xx */
    BCMCLI_PARM_BUFFER,          /**< Byte array */

    BCMCLI_PARM_USERDEF          /**< User-defined parameter. User must provide scan_cb */
} bcmcli_parm_type;

/** Numeric type used for storing enum values. */
typedef long bcmcli_enum_number;

/** Enum attribute value.
 *
 *  Enum values is an array of bcmcli_enum_val terminated by element with name==NULL
 *
 */
typedef struct bcmcli_enum_val
{
    const char *name;           /**< Enum symbolic name */
    bcmcli_enum_number val;     /**< Enum internal value */
    bcmcli_cmd_parm *parms;     /**< Extension parameter table for enum-selector */
} bcmcli_enum_val;
#define BCMCLI_MAX_ENUM_VALUES   128     /**< Max number of enum values */
#define BCMCLI_ENUM_LAST     { NULL, 0}  /**< Last entry in enum table */

/** Boolean values (true/false, yes/no, on/off)
 *
 */
extern bcmcli_enum_val bcmcli_enum_bool_table[];

/* Monitor data types */
typedef long bcmcli_number;      /**< Type underlying BCMCLI_PARM_NUMBER, BCMCLI_PARM_DECIMAL */
typedef long bcmcli_unumber;     /**< Type underlying BCMCLI_PARM_HEX, BCMCLI_PARM_UDECIMAL */
typedef long bcmcli_number64;    /**< Type underlying BCMCLI_PARM_NUMBER64, BCMCLI_PARM_DECIMAL64 */
typedef long bcmcli_unumber64;   /**< Type underlying BCMCLI_PARM_HEX64, BCMCLI_PARM_UDECIMAL64 */

/** Parameter value */
typedef union bcmcli_parm_value
{
    long number;                    /**< Signed number */
    unsigned long unumber;          /**< Unsigned number */
    long long number64;             /**< Signed 64-bit number */
    unsigned long long unumber64;   /**< Unsigned 64-bit number */
    const char *string;             /**< 0-terminated string */
    double d;                       /**< Double-precision floating point number */
    bcmos_mac_address mac;          /**< MAC address */
    bcmolt_buf buffer;              /**< Buffer: used for BCMCLI_PARM_BUFFER */
    bcmcli_enum_number enum_val;    /**< Enum value (number) */
}  bcmcli_parm_value;

/** User-defined scan function.
 * The function is used for parsing user-defined parameter types
 * Returns: 0-ok, <=error
 *
 */
typedef bcmos_errno (*bcmcli_scan_cb)(const bcmcli_cmd_parm *parm, bcmcli_parm_value *value,
    const char *string_val);

/** User-defined print function.
 * The function is used for printing user-defined parameter types
 *
 */
typedef void (*bcmcli_format_cb)(const bcmcli_cmd_parm *parm, bcmcli_parm_value value,
    char *buffer, int size);


/** Function parameter structure */
struct bcmcli_cmd_parm
{
   const char *name;            /**< Parameter name. Shouldn't be allocated on stack! */
   const char *description;     /**< Parameter description. Shouldn't be allocated on stack! */
   bcmcli_parm_type type;       /**< Parameter type */
   uint8_t flags;               /**< Combination of BCMCLI_PARM_xx flags */
#define BCMCLI_PARM_FLAG_NONE       0x00 /**< For use instead of magic number 0 when no flags apply */
#define BCMCLI_PARM_FLAG_OPTIONAL   0x01 /**< Parameter is optional */
#define BCMCLI_PARM_FLAG_DEFVAL     0x02 /**< Default value is set */
#define BCMCLI_PARM_FLAG_RANGE      0x04 /**< Range is set */
#define BCMCLI_PARM_FLAG_EOL        0x20 /**< String from the current parser position till EOL */
#define BCMCLI_PARM_FLAG_SELECTOR   0x40 /**< Parameter selects other parameters */
#define BCMCLI_PARM_FLAG_ASSIGNED   0x80 /**< Internal flag: parameter is assigned */

   bcmcli_number low_val;       /**< Low val for range checking */
   bcmcli_number hi_val;        /**< Hi val for range checking */
   bcmcli_parm_value value;     /**< Value */
   bcmcli_enum_val *enum_table; /**< Table containing { enum_name, enum_value } pairs */
   bcmcli_scan_cb scan_cb;      /**< User-defined scan function for BCMCLI_PARM_USERDEF parameter type */
   bcmcli_format_cb format_cb;  /**< User-defined format function for BCMCLI_PARM_USERDEF parameter type */
   uint32_t max_array_size;     /**< Max array size for array-parameter */
   uint32_t array_size;         /**< Actual array size for array-parameter */
   bcmcli_parm_value *values;   /**< Array values */
   void *user_data;             /**< User data - passed transparently to command handler */
};

/** Command parameter list terminator */
#define BCMCLI_PARM_LIST_TERMINATOR  { .name=NULL, .type=BCMCLI_PARM_NONE }

/** Helper macro: make simple parameter
 * \param[in] _name     Parameter name
 * \param[in] _descr    Parameter description
 * \param[in] _type     Parameter type
 * \param[in] _flags    Parameter flags
 */
#define BCMCLI_MAKE_PARM(_name, _descr, _type, _flags) \
    { .name=(_name), .description=(_descr), .type=(_type), .flags=(_flags) }

/** Helper macro: make simple parameter
 * \param[in] _name       Parameter name
 * \param[in] _descr      Parameter description
 * \param[in] _type       Parameter type
 * \param[in] _flags      Parameter flags
 * \param[in] _size       Max array size
 * \param[in] _values     Array values buffer
 */
#define BCMCLI_MAKE_PARM_ARRAY(_name, _descr, _type, _flags, _size, _values) \
    { .name=(_name), .description=(_descr), .type=(_type), .flags=(_flags),\
        .max_array_size=(_size), .values=(_values) }

/** Helper macro: make simple parameter for arrays of enums
 * \param[in] _name       Parameter name
 * \param[in] _descr      Parameter description
 * \param[in] _type       Parameter type
 * \param[in] _flags      Parameter flags
 * \param[in] _size       Max array size
 * \param[in] _values     Array values buffer
 * \param[in] _enum_table An array of enums that may be in the array
 */
#define BCMCLI_MAKE_PARM_ENUM_ARRAY(_name, _descr, _type, _flags, _size, _values, _enum_table) \
    { .name=(_name), .description=(_descr), .type=(_type), .flags=(_flags),\
        .max_array_size=(_size), .values=(_values), .enum_table=(_enum_table) }

/** Helper macro: make range parameter
 * \param[in] _name     Parameter name
 * \param[in] _descr    Parameter description
 * \param[in] _type     Parameter type
 * \param[in] _flags    Parameter flags
 * \param[in] _min      Min value
 * \param[in] _max      Max value
 */
#define BCMCLI_MAKE_PARM_RANGE(_name, _descr, _type, _flags, _min, _max) \
    { .name=(_name), .description=(_descr), .type=(_type), .flags=(_flags) | BCMCLI_PARM_FLAG_RANGE, \
        .low_val=(_min), .hi_val=(_max) }

/** Helper macro: make range parameter for arrays with range
 * \param[in] _name     Parameter name
 * \param[in] _descr    Parameter description
 * \param[in] _type     Parameter type
 * \param[in] _flags    Parameter flags
 * \param[in] _size     Max array size
 * \param[in] _values   Array values buffer
 * \param[in] _min      Min value
 * \param[in] _max      Max value
 */
#define BCMCLI_MAKE_PARM_ARRAY_RANGE(_name, _descr, _type, _flags, _size, _values, _min, _max) \
    { .name=(_name), .description=(_descr), .type=(_type), .flags=(_flags) | BCMCLI_PARM_FLAG_RANGE,\
         .max_array_size=(_size), .values=(_values), .low_val=(_min), .hi_val=(_max) }

/** Helper macro: make parameter with default value
 * \param[in] _name     Parameter name
 * \param[in] _descr    Parameter description
 * \param[in] _type     Parameter type
 * \param[in] _flags    Parameter flags
 * \param[in] _dft      Default value
 */
#define BCMCLI_MAKE_PARM_DEFVAL(_name, _descr, _type, _flags, _dft) \
    { .name=(_name), .description=(_descr), .type=(_type), .flags=(_flags) | BCMCLI_PARM_FLAG_DEFVAL, \
        .value = {_dft} }

/** Helper macro: make range parameter with default value
 * \param[in] _name     Parameter name
 * \param[in] _descr    Parameter description
 * \param[in] _type     Parameter type
 * \param[in] _flags    Parameter flags
 * \param[in] _min      Min value
 * \param[in] _max      Max value
 * \param[in] _dft      Default value
 */
#define BCMCLI_MAKE_PARM_RANGE_DEFVAL(_name, _descr, _type, _flags, _min, _max, _dft) \
    { .name=(_name), .description=(_descr), .type=(_type), \
        .flags=(_flags) | BCMCLI_PARM_FLAG_RANGE | BCMCLI_PARM_FLAG_DEFVAL, \
        .low_val=(_min), .hi_val=(_max), .value = {_dft} }

/** Helper macro: make enum parameter
 * \param[in] _name     Parameter name
 * \param[in] _descr    Parameter description
 * \param[in] _values   Enum values table
 * \param[in] _flags    Parameter flags
 */
#define BCMCLI_MAKE_PARM_ENUM(_name, _descr, _values, _flags) \
    { .name=(_name), .description=(_descr), .type=BCMCLI_PARM_ENUM, .flags=(_flags), .enum_table=(_values)}

/** Helper macro: make enum parameter with default value
 * \param[in] _name     Parameter name
 * \param[in] _descr    Parameter description
 * \param[in] _values   Enum values table
 * \param[in] _flags    Parameter flags
 * \param[in] _dft      Default value
 */
#define BCMCLI_MAKE_PARM_ENUM_DEFVAL(_name, _descr, _values, _flags, _dft) \
    { .name=(_name), .description=(_descr), .type=BCMCLI_PARM_ENUM, .flags=(_flags) | BCMCLI_PARM_FLAG_DEFVAL,\
        .enum_table=(_values), .value={.string=_dft} }

/** Helper macro: make enum mask parameter
 * \param[in] _name     Parameter name
 * \param[in] _descr    Parameter description
 * \param[in] _values   Enum values table
 * \param[in] _flags    Parameter flags
 */
#define BCMCLI_MAKE_PARM_ENUM_MASK(_name, _descr, _values, _flags) \
    { .name=(_name), .description=(_descr), .type=BCMCLI_PARM_ENUM_MASK, .flags=(_flags), .enum_table=(_values)}

/** Helper macro: make enum_mask parameter with default value
 * \param[in] _name     Parameter name
 * \param[in] _descr    Parameter description
 * \param[in] _values   Enum values table
 * \param[in] _flags    Parameter flags
 * \param[in] _dft      Default value
 */
#define BCMCLI_MAKE_PARM_ENUM_MASK_DEFVAL(_name, _descr, _values, _flags, _dft) \
    { .name=(_name), .description=(_descr), .type=BCMCLI_PARM_ENUM_MASK, .flags=(_flags) | BCMCLI_PARM_FLAG_DEFVAL,\
        .enum_table=(_values), .value={.string=_dft} }

/** Helper macro: make enum-selector parameter
 * \param[in] _name     Parameter name
 * \param[in] _descr    Parameter description
 * \param[in] _values   Selector values table
 * \param[in] _flags    Parameter flags
 */
#define BCMCLI_MAKE_PARM_SELECTOR(_name, _descr, _values, _flags) \
    { .name=(_name), .description=(_descr), .type=BCMCLI_PARM_ENUM, .flags=(_flags) | BCMCLI_PARM_FLAG_SELECTOR,\
        .enum_table=(_values) }

/** Helper macro: make buffer parameter
 * \param[in] _name     Parameter name
 * \param[in] _descr    Parameter description
 * \param[in] _flags    Parameter flags
 * \param[in] _buf      Memory buffer associated with the parameter
 * \param[in] _size     Buffer size
 */
#define BCMCLI_MAKE_PARM_BUFFER(_name, _descr, _flags, _buf, _size) \
    { .name=(_name), .description=(_descr), .type=BCMCLI_PARM_BUFFER, \
        .flags=(_flags), .value.buffer = {.start = _buf, .curr = _buf, .len = _size} }

/** Register command without parameters helper */
#define BCMCLI_MAKE_CMD_NOPARM(dir, cmd, help, cb) \
{\
    bcmos_errno bcmcli_cmd_add_err = bcmcli_cmd_add(dir, cmd, cb, help, BCMCLI_ACCESS_ADMIN, NULL, NULL);\
    BUG_ON(BCM_ERR_OK != bcmcli_cmd_add_err);\
}

/** Register command helper */
#define BCMCLI_MAKE_CMD(dir, cmd, help, cb, parms...)                           \
{                                                                               \
    static bcmcli_cmd_parm cmd_parms[]={                                        \
        parms,                                                                  \
        BCMCLI_PARM_LIST_TERMINATOR                                             \
    };                                                                          \
    bcmos_errno bcmcli_cmd_add_err = bcmcli_cmd_add(dir, cmd, cb, help, BCMCLI_ACCESS_ADMIN, NULL, cmd_parms);   \
    BUG_ON(BCM_ERR_OK != bcmcli_cmd_add_err);\
}

/** Optional custom directory handlers */
typedef void (*bcmcli_dir_enter_leave_cb)(bcmcli_session *session, bcmcli_entry *dir, int is_enter);

/** Optional command or directory help callback
 * \param[in]   session     Session handle
 * \param[in]   h           Command or directory handle
 * \param[in]   parms       Parameter(s) - the rest of the command string.
 *                          Can be used for example to get help on individual parameters
 */
typedef void (*bcmcli_help_cb)(bcmcli_session *session, bcmcli_entry *h, const char *parms);


/** Extra parameters of monitor directory.
 * See \ref bcmcli_dir_add
 *
 */
typedef struct bcmcli_dir_extra_parm
{
    void *user_priv;            /**< private data passed to enter_leave_cb */
    bcmcli_dir_enter_leave_cb enter_leave_cb; /**< callback function to be called when session enters/leavs the directory */
    bcmcli_help_cb help_cb;     /**< Help function called to print directory help instead of the automatic help */
} bcmcli_dir_extra_parm;


/** Extra parameters of monitor command.
 * See \ref bcmcli_cmd_add
 *
 */
typedef struct bcmcli_cmd_extra_parm
{
    bcmcli_help_cb help_cb;     /**< Optional help callback. Can be used for more sophisticated help, e.g., help for specific parameters */
    uint32_t flags;             /**< Command flags */
#define BCMCLI_CMD_FLAG_NO_NAME_PARMS   0x00000001 /**< No named parms. Positional only. Can be useful if parameter value can contain ',' */
    void (*free_parms)(bcmcli_cmd_parm *parms);    /* Optional user-defined free */
} bcmcli_cmd_extra_parm;


/** Monitor command handler prototype */
typedef bcmos_errno (*bcmcli_cmd_cb)(bcmcli_session *session, const bcmcli_cmd_parm parm[], uint16_t n_parms);

/** CLI command logging mode */
typedef enum
{
    BCMCLI_LOG_NONE,                /**< Disable logging */
    BCMCLI_LOG_CLI,                 /**< Log commands as is and rc as CLI comment*/
    BCMCLI_LOG_C_COMMENT            /**< Log as C comments */
} bcmcli_log_mode;

/** Add subdirectory to the parent directory
 *
 * \param[in]   parent          Parent directory handle. NULL=root
 * \param[in]   name            Directory name
 * \param[in]   help            Help string
 * \param[in]   access_right    Access rights
 * \param[in]   extras          Optional directory descriptor. Mustn't be allocated on the stack.
 * \return      new directory handle or NULL in case of failure
 */
bcmcli_entry *bcmcli_dir_add(bcmcli_entry *parent, const char *name,
                             const char *help, bcmcli_access_right access_right,
                             const bcmcli_dir_extra_parm *extras);


/** Scan directory tree and look for directory named "name".
 *
 * \param[in]   parent          Directory sub-tree root. NULL=root
 * \param[in]   name            Name of directory to be found
 * \return      directory handle if found or NULL if not found
 */
bcmcli_entry *bcmcli_dir_find(bcmcli_entry *parent, const char *name );


/** Scan directory tree and look for command named "name".
 *
 * \param[in]   parent          Directory sub-tree root. NULL=root
 * \param[in]   name            Name of command to be found
 * \return      command handle if found or NULL if not found
 */
bcmcli_entry *bcmcli_cmd_find(bcmcli_entry *parent, const char *name );


/** Get token name
 * \param[in]   token           Directory or command token
 * \return      directory token name
 */
const char *bcmcli_token_name(bcmcli_entry *token);

/** Find the CLI parameter with the specified name (case insensitive).
 * \param[in]   session  CLI session
 * \param[in]   name     Parameter name
 * \return      The CLI parameter that was found, or NULL if not found
 */
bcmcli_cmd_parm *bcmcli_find_named_parm(bcmcli_session *session, const char *name);

/** Find the first CLI parameter whose name starts with the specified string (case insensitive).
 * \param[in]   session  CLI session
 * \param[in]   prefix   Parameter name prefix
 * \return      The CLI parameter that was found, or NULL if not found
 */
bcmcli_cmd_parm *bcmcli_find_parm_by_prefix(bcmcli_session *session, const char *prefix);

/** Query parameter value status.
 * The function can be used for scalar and array parameters
 * \param[in]   session         CLI session
 * \param[in]   parm            Parameter from the array passed to the CLI command handler
 *                              or returned by bcmcli_find_named_parm()
 * \param[in]   value_index     value_index - for array parameters
 * \return      BCMOS_TRUE if the parameter value is set, BCMOS_FALSE otherwise
 */
bcmos_bool bcmcli_parm_value_is_set(bcmcli_session *session, bcmcli_cmd_parm *parm, uint32_t value_index);

/** Add CLI command
 *
 * \param[in]   dir             Handle of directory to add command to. NULL=root
 * \param[in]   name            Command name
 * \param[in]   cmd_cb          Command handler
 * \param[in]   help            Help string
 * \param[in]   access_right    Access rights
 * \param[in]   extras          Optional extras
 * \param[in]   parms           Optional parameters array. Must not be allocated on the stack!
 *                              If parms!=NULL, the last parameter in the array must have name==NULL.
 * \return
 *      0   =OK\n
 *      <0  =error code
 */
bcmos_errno bcmcli_cmd_add(bcmcli_entry *dir, const char *name, bcmcli_cmd_cb cmd_cb,
                  const char *help, bcmcli_access_right access_right,
                  const bcmcli_cmd_extra_parm *extras, bcmcli_cmd_parm parms[]);


/** Destroy token (command or directory)
 * \param[in]   token           Directory or command token. NULL=root
 */
void bcmcli_token_destroy(bcmcli_entry *token);

/** Parse and execute input string.
 * input_string can contain multiple commands delimited by ';'
 *
 * \param[in]   session         Session handle
 * \param[in]   input_string    String to be parsed
 * \return
 *      =0  - OK \n
 *      -EINVAL - parsing error\n
 *      other - return code - as returned from command handler.
 *            It is recommended to return -EINTR to interrupt monitor loop.
 */
bcmos_errno bcmcli_parse(bcmcli_session *session, char *input_string);

/** Read input and parse iteratively until EOF or bcmcli_is_stopped()
 *
 * \param[in]   session         Session handle
 * \return
 *      =0  - OK \n
 */
bcmos_errno bcmcli_driver(bcmcli_session *session);

/** Stop monitor driver.
 * The function stops \ref bcmcli_driver
 * \param[in]   session         Session handle
 */
void bcmcli_stop(bcmcli_session *session);

/** Returns 1 if monitor session is stopped
 * \param[in]   session         Session handle
 * \returns 1 if monitor session stopped by bcmcli_stop()\n
 * 0 otherwise
 */
bcmos_bool bcmcli_is_stopped(bcmcli_session *session);

/** Get current directory for the session,
 * \param[in]   session         Session handle
 * \return      The current directory handle
 */
bcmcli_entry *bcmcli_dir_get(bcmcli_session *session );

/** Set current directory for the session.
 * \param[in]   session         Session handle
 * \param[in]   dir             Directory that should become current
 * \return
 *      =0  - OK
 *      <0  - error
 */
bcmos_errno bcmcli_dir_set(bcmcli_session *session, bcmcli_entry *dir);

/** Get parameter number given its name.
 * The function is intended for use by command handlers
 * \param[in]       session         Session handle
 * \param[in]       parm_name       Parameter name
 * \return
 *  >=0 - parameter number\n
 *  <0  - parameter with this name doesn't exist
 */
int bcmcli_parm_number(bcmcli_session *session, const char *parm_name);

/** Get parameter by name
 * The function is intended for use by command handlers
 * \param[in]       session         Session handle
 * \param[in]       parm_name       Parameter name
 * \return
 * parameter pointer or NULL if not found
 */
bcmcli_cmd_parm *bcmcli_parm_get(bcmcli_session *session, const char *parm_name);

/** Check if parameter is set
 * The function is intended for use by command handlers
 * \param[in]       session         Session handle
 * \param[in]       parm            Parameter name
 * \return
 * TRUE if parameter is set, FALSE otherwise
 */
static inline bcmos_bool bcmcli_parm_is_set(bcmcli_session *session, const bcmcli_cmd_parm *parm)
{
    return (parm->flags & BCMCLI_PARM_FLAG_ASSIGNED) ? BCMOS_TRUE : BCMOS_FALSE;
}

/** Check if parameter is set
 * \param[in]       session         Session handle
 * \param[in]       parm_number     Parameter number
 * \return
 *  0 if parameter is set\n
 *  BCM_ERR_NOENT if parameter is not set
 *  BCM_ERR_PARM if parm_number is invalid
 */
bcmos_errno bcmcli_parm_check(bcmcli_session *session, int parm_number);


/** Get enum's string value given its internal value
 * \param[in]       table           Enum table
 * \param[in]       value           Internal value
 * \return
 *      enum string value or NULL if internal value is invalid
 */
static inline const char *bcmcli_enum_stringval(const bcmcli_enum_val table[], long value)
{
    while(table->name)
    {
        if (table->val==value)
            return table->name;
        ++table;
    }
    return NULL;
}


/** Get enum's parameter string value given its internal value
 * \param[in]       session         Session handle
 * \param[in]       parm_number     Parameter number
 * \param[in]       value           Internal value
 * \return
 *      enum string value or NULL if parameter is not enum or
 *      internal value is invalid
 */
const char *bcmcli_enum_parm_stringval(bcmcli_session *session, int parm_number, long value);


/** Print CLI parameter value
 * \param[in]       session         Session handle
 * \param[in]       parm            Parameter
 */
void bcmcli_parm_print(bcmcli_session *session, const bcmcli_cmd_parm *parm);


/** strncpy flavour that always add 0 terminator
 * \param[in]       dst             Destination string
 * \param[in]       src             Source string
 * \param[in]       dst_size        Destination buffer size
 * \return dst
 */
static inline char *bcmcli_strncpy(char *dst, const char *src, uint32_t dst_size)
{
    strncpy(dst, src, dst_size-1);
    dst[dst_size-1] = 0;
    return dst;
}


/** strncat flavour that limits size of destination buffer
 * \param[in]       dst             Destination string
 * \param[in]       src             Source string
 * \param[in]       dst_size        Destination buffer size
 * \return dst
 */
static inline char *bcmcli_strncat(char *dst, const char *src, uint32_t dst_size)
{
    uint32_t dst_len = strlen(dst);
    return strncat(dst, src, dst_size-dst_len-1);
}

/* Redefine bcmcli_session_print --> bcmcli_print */
#define bcmcli_print bcmcli_session_print

/** Enable / disable CLI command logging
 * \param[in]   mode    Logging flags
 * \param[in]   log     Log session. Must be set if mode != BCMCLI_CMD_LOG_NONE
 * \return 0=OK or error <0
 */
bcmos_errno bcmcli_log_set(bcmcli_log_mode mode, bcmcli_session *log);

/** Write string to CLI log.
 * The function is ignored if CLI logging is not enabled using bcmcli_log_set()
 * \param[in]   format  printf-like format followed by arguments
 */
void bcmcli_log(const char *format, ...);

#ifdef __cplusplus
}
#endif

/** @} end bcm_cli group */

#endif /* #ifndef BCM_CLI_H */
