/*
<:copyright-BRCM:2016:DUAL/GPL:standard

   Broadcom Proprietary and Confidential.(c) 2016 Broadcom
   All Rights Reserved

Unless you and Broadcom execute a separate written software license
agreement governing use of this software, this software is licensed
to you under the terms of the GNU General Public License version 2
(the "GPL"), available at http://www.broadcom.com/licenses/GPLv2.php,
with the following added to such license:

   As a special exception, the copyright holders of this software give
   you permission to link this software with independent modules, and
   to copy and distribute the resulting executable under terms of your
   choice, provided that you also meet, for each linked independent
   module, the terms and conditions of the license of that module.
   An independent module is a module which is not derived from this
   software.  The special exception does not apply to any modifications
   of the software.

Not withstanding the above, under no circumstances may you combine
this software in any way with any other Broadcom software provided
under a license other than the GPL, without Broadcom's express prior
written consent.

:>
 */

#ifndef BCMOLT_FLD_H
#define BCMOLT_FLD_H

#include "bcmos_system.h"
#include "bcmolt_model_types.h"
#include "bcmolt_model_ids.h"
#include "bcm_fld_common.h"

/** \defgroup \fld_data_types Data Types
 * \ingroup fld
 * @{
 */

#define PCIE_REVISION_REGISTER_OFFSET       0x6406CU /* PCIE_PCIE_PCIE_0_PCIE_PCIE_0_MISC_REVISION    */

typedef struct
{
    unsigned long   soc_regs_base;
    unsigned long   soc_sram_base;
    unsigned long   soc_ddr_base;
    uint32_t        soc_ddr_length;     /* ddr window length */
    uint32_t        soc_crt_length;     /* ddr used for load length */
} bcm_fld_device_info;

typedef struct
{
    uint32_t    os_entry_offset;
    uint32_t    protocol_version;
    uint32_t    state;
    uint32_t    reason;
} bcm_fld_device_stat;

/**
* \brief Initialize internal data base
*
*  allocate the array of pointers to info per device allocate
*  and clear the info structure for every possible device
*
* \param[in] max_devices How many devices might be defined
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_init(uint32_t max_devices);

/**
* \brief Initialize communication area
*
*  clear the communication area
*
* \param[in] device Identifies the device
*
*/
void bcm_fld_clear_comm_area(uint32_t device);

/**
* \brief Release internal data base
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_exit(void);

/**
* \brief Register a device to driver
*
*  store user's information about the device
*
* \param[in] device Identifies the device
* \param[in] info Information
*   - regs base address
*   - sram base address
*   - ddr base address
*   - ddr window size
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_register(uint32_t device, bcm_fld_device_info *info);

/**
* \brief Unregister a device from driver
*
*  remove the device from the internal db
*
* \param[in] device Identifies the device
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_unregister(uint32_t device);

/**
* \brief Read from communication SoC debug information
*
* \param[in] device Identifies the device
* \param[in] debug_state Information to be filled pointer
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_get_device_status(uint32_t device, bcm_fld_device_stat *debug_state);

/**
* \brief Read from communication SoC status
*
* \param[in] device Identifies the device
* \param[in] cpu_state Information to be filled pointer
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_get_device_loading_state(uint32_t device, BCM_FLD_CPU_STATE *cpu_state);

/**
* \brief Write in communication area the finish writing DDR
*        indicator
*
* \param[in] device Identifies the device
* \param[in] os_entry_address Entry point for embedded
*       application
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_host_finish_write_ddr(uint32_t device, uint32_t os_entry_address);

/**
* \brief Read from communication the logs area
*
* \param[in] device Identifies the device
* \param[out] log_area Pointer logger area
* \param[out] length Length logger area
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_get_logs(uint32_t device, char **log_area, int *length);

/**
* \brief Write the received buffer to memory according to offset
*        and type
*
* \param[in] device Identifies the device
* \param[in] buff Pointer to the buffer to be written
* \param[in] buff_size Buffer size
* \param[in] offset_in_image offset to write the buffer
* \param[in] image_type Loading type, according to destination :
*       SRAM or DDR
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_write(uint32_t device, char *buff, uint32_t buff_size, uint32_t offset_in_image, bcmolt_device_image_type image_type);

/**
* \brief Read into received buffer from memory according to
*        offset and type
*
* \param[in] device Identifies the device
* \param[in] buff Pointer to the buffer to receive the content
* \param[in] buff_size Buffer size
* \param[in] offset_in_image offset to read into the buffer
* \param[in] image_type Loading type, according to destination :
*       SRAM or DDR
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_read(uint32_t device, char *buff, uint32_t *buff_size, uint32_t offset_in_image, bcmolt_device_image_type image_type);

/**
* \brief Take soc cpu out of reset
*
* \param[in] device Identifies the device
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_start_bootloader(uint32_t device, uint32_t test_ddr);

/**
* \brief Check if bootloader finished to run
*
* \param[in] device Identifies the device
*
* \return
* BCMOS_TRUE if successful, BCMOS_FALSE if failed
*/
bcmos_bool bcm_fld_is_bootloader_done(uint32_t device);

bcmos_bool bcm_fld_is_ddr_test_done(uint32_t device);

/**
* \brief Check if SoC finished synchronization process
*
* \param[in] device Identifies the device
*
* \return
*/
bcmos_bool bcm_fld_test_device_bootrecord(uint32_t device);

/**
* \brief Returns the bootrecord of the SoC
*
* \param[in] device Identifies the device
* \param[in] opaque_data Pointer to information
*   - physical address of the buffer descriptors ring
*   - physical address of shadow buffer descriptors ring
*   info_length MUST be modulo 4
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_get_device_bootrecord(uint32_t device, uint32_t *opaque_data);

/**
* \brief Set bootrecord valid bit to be checked by the device
*
* \param[in] device Identifies the device
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_set_host_bootrecord_flag(uint32_t device);

/**
* \brief Check bootrecord valid bit set previously by the device
*
* \param[in] device Identifies the device
* \return
* 0 if successful, <0 if failed
*/
bcmos_bool bcm_fld_test_host_bootrecord_flag(uint32_t device);

/**
* \brief Clear bootrecord valid bit to be checked by the device
*
* \param[in] device Identifies the device
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_clear_device_bootrecord_flag(uint32_t device);

/**
* \brief Check bootrecord valid bit set previously by the device
*
* \param[in] device Identifies the device
* \return
* 0 if successful, <0 if failed
*/
bcmos_bool bcm_fld_test_device_bootrecord_flag(uint32_t device);

/**
* \brief Write the tx and rx queue sizes to communication area
*   set queue validity bit for the device
*
* \param[in] device Identifies the device
* \param[in] host_tx_size Queue size : is also the device rx
*       queue size
* \param[in] host_rx_size Queue size : is also the device tx
*       queue size
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_set_rings_size(uint32_t device, uint32_t host_tx_size, uint32_t host_rx_size);

/**
* \brief set DDR RASconfiguration properties
*
* \param[in] device Identifies the device
*
* \param[in] ras  Identifies the ras number , can be 0 or 1
*
* \param[in] mode Identifies mode - disable 0, gpon 1, xgpon 2,
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_set_ras_mode_set(uint32_t device, uint32_t ras, uint32_t mode);

/**
* \brief Notify device about an event that occurred on the host side
*
* \param[in] device Identifies the device
*
* \param[in] event  User defined event
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_set_host_event(uint32_t device, uint32_t event);

/**
* \brief Check queues valid bit set previously by Host
*
* \param[in] device Identifies the device
*
* \return
* FALSE if PRM_VALID bit is not set, TRUE if PRM_VALID bit is
* set
*/
bcmos_bool bcm_fld_test_queues_flag(uint32_t device);

/**
* \brief Get the exception area state, set by the device exception
*        handler
*
* \param[in] device Identifies the device
* \param[out] state Pointer to exception state on cpu0 value
* \param[out] state Pointer to exception state on cpu1 value
*
* \return
* 0 if successful, <0 if failed
*/

bcmos_errno bcm_fld_get_exception_state(uint32_t device, uint32_t *state0, uint32_t *state1);

/**
* \brief Clear the exception area state
*
* \param[in] device Identifies the device
* \param[in] cpuid Identifies the the device cpu
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_clear_exception_state(uint32_t device, uint32_t cpuid);

/**
* \brief Fill buffer with the exception area content, set by
*        the device exception handler
*
* \param[in] device Identifies the device
* \param[in] cpuid Identifies the the device cpu
* \param[out] buffer Pointer to be filled with the exception
*       area content
* \param[out] length Length of the buffer
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_copy_exception_log(uint32_t device, uint32_t cpuid, char *buffer, int *length);

#ifdef TX_ENABLE_EVENT_TRACE
/**
* \brief Set event trace flag to be read by embedded and start
*        traceing
*
* \param[in] device Identifies the device
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_set_event_trace(uint32_t device);

/**
* \brief Clear event trace flag to be read by embedded and stop
*        traceing
*
* \param[in] device Identifies the device
*
* \return
* 0 if successful, <0 if failed
*/
bcmos_errno bcm_fld_clear_event_trace(uint32_t device);
#endif

/**
* \Debug states and functions
*/
typedef struct
{
    uint32_t    write_sram;
    uint32_t    start_device;
    uint32_t    write_ddr;
    uint32_t    finish_ddr;
} bcm_fld_host_debug_state;

typedef enum
{
    BCM_FLD_HOST_NONE,
    BCM_FLD_HOST_WRITE_SRAM,    /* set before load  image to SRAM */
    BCM_FLD_HOST_START_CPU,     /* set before resume ARM */
    BCM_FLD_HOST_WRITE_DDR      /* set before load  image to DDR */
} BCM_FLD_HOST_DEBUG_VALUES;

typedef struct
{
    uint32_t boot_from_sram_states;
    uint32_t boot_from_sram_errors;
    uint32_t boot_from_sram_exception;
    uint32_t run_from_ddr_state;
} bcm_fld_device_debug_state;

/**
* \brief Set and get debugging traceing from communication area
*
* \param[in] device Identifies the device
* \param[in] info Pointer to information
*
* \return
* 0 if successful, <0 if failed
*/
/* returns soc debug states */
bcmos_errno bcm_fld_get_device_debug_status(uint32_t device, bcm_fld_device_debug_state *info);
/* deal with host states - functional and debug */
bcmos_errno bcm_fld_get_host_debug_status(uint32_t device, bcm_fld_host_debug_state *info);
bcmos_errno bcm_fld_set_host_debug_status(uint32_t device, BCM_FLD_HOST_DEBUG_VALUES stat);
/**
* \brief Check if swap needed for the respective memory
*
* \param[in] device Identifies the device
*
* \return
* true if swap needed, false if swap not needed
*/
bcmos_bool bcm_fld_regs_swap_needed(uint32_t device);
bcmos_bool bcm_fld_sram_swap_needed(uint32_t device);
bcmos_bool bcm_fld_ddr_swap_needed(uint32_t device);

bcmos_errno bcm_fld_set_avs_cont(uint32_t device, uint32_t value);

void *bcm_fld_get_sw_error_table(uint32_t device, uint32_t *size);
uint32_t bcm_fld_ddr_test_result_get(uint32_t device, uint32_t word);

#endif

