/******************************************************************************
 *
 *  <:copyright-BRCM:2016:DUAL/GPL:standard
 *
 *     Copyright (c) 2016 Broadcom
 *     All Rights Reserved
 *
 *  Unless you and Broadcom execute a separate written software license
 *  agreement governing use of this software, this software is licensed
 *  to you under the terms of the GNU General Public License version 2
 *  (the "GPL"), available at http://www.broadcom.com/licenses/GPLv2.php,
 *  with the following added to such license:
 *
 *     As a special exception, the copyright holders of this software give
 *     you permission to link this software with independent modules, and
 *     to copy and distribute the resulting executable under terms of your
 *     choice, provided that you also meet, for each linked independent
 *     module, the terms and conditions of the license of that module.
 *     An independent module is a module which is not derived from this
 *     software.  The special exception does not apply to any modifications
 *     of the software.
 *
 *  Not withstanding the above, under no circumstances may you combine
 *  this software in any way with any other Broadcom software provided
 *  under a license other than the GPL, without Broadcom's express prior
 *  written consent.
 *
 *  :>
 *
 *****************************************************************************/
/*
 * cmdline.c - Command line argument helper
 */

#include <cmdline.h>

/* Print usage information */
bcmos_errno cl_print_usage(const char *app, const char *arg,
    const cl_argument supported_args[], int num_supported_args,
    cl_argument_usage_flags flags)
{
    const char *owner = "";
    int i;

    if (arg)
        bcmos_printf("*** Error in parameter %s\n", arg);
    if (app)
    {
        const char *p, *fname;
        p = strrchr(app, '/');
        fname = p ? p + 1 : app;
        bcmos_printf("Usage: %s arguments\n", fname);
    }
    for (i = 0; i < num_supported_args; i++)
    {
        const cl_argument *opt = &supported_args[i];
        if ((flags & CL_ARGUMENT_USAGE_FLAG_OWNER) != 0 &&
            opt->owner && strcmp(opt->owner, owner))
        {
            owner = opt->owner;
            bcmos_printf("\n%s options:\n", owner);
        }
        if (opt->short_name)
            bcmos_printf("%-3s%c", opt->short_name, opt->long_name ? ',' : ' ');
        else
            bcmos_printf("    ");
        bcmos_printf("%-20s %20s  %s\n",
            opt->long_name ? opt->long_name : "",
            opt->extra_arg ? opt->extra_arg : "",
            opt->description ? opt->description : "");
    }
    return BCM_ERR_PARM;
}

/* Get parameter by name */
const cl_argument *cl_parm_get(const char *name, const cl_argument supported_args[], int num_supported_args)
{
    const cl_argument *opt = NULL;
    int i;

    /* Make sure that all mandatory parameters are set */
    for (i = 0; i < num_supported_args; i++)
    {
        opt = &supported_args[i];
        if ((opt->short_name && !strcmp(opt->short_name, name)) ||
            (opt->long_name && !strcmp(opt->long_name, name)))
            break;
    }
    return (i < num_supported_args) ? opt : NULL;
}

/* Validate command line parameters */
bcmos_errno cl_validate(int argc, char *argv[],
    const cl_argument supported_args[], int num_supported_args)
{
    bcmos_bool parm_set[num_supported_args];
    bcmos_errno rc = BCM_ERR_OK;
    int i;

    memset(parm_set, 0, sizeof(parm_set));
    for (i = 1; i < argc; i++)
    {
        const cl_argument *opt = cl_parm_get(argv[i], supported_args, num_supported_args);
        if (opt == NULL)
        {
            bcmos_printf("*** Invalid parameter: %s\n", argv[i]);
            rc = BCM_ERR_PARM;
            break;
        }
        if (opt->extra_arg)
        {
            ++i;
            if (i >= argc)
            {
                bcmos_printf("*** Argument is missing after %s\n", argv[i-1]);
                rc = BCM_ERR_PARM;
                break;
            }
        }
        parm_set[opt - supported_args] = BCMOS_TRUE;
    }

    /* Make sure that all mandatory parameters are set */
    for (i = 0; i < num_supported_args; i++)
    {
        const cl_argument *opt = &supported_args[i];
        if ((opt->flags & CL_ARGUMENT_FLAG_MANDATORY) && !parm_set[i])
        {
            bcmos_printf("*** Mandatory parameter %s is missing\n", opt->long_name ? opt->long_name : opt->short_name);
            rc = BCM_ERR_PARM;
            break;
        }
    }
    return rc;
}
