/******************************************************************************
 *
 *  <:copyright-BRCM:2016:DUAL/GPL:standard
 *
 *     Copyright (c) 2016 Broadcom
 *     All Rights Reserved
 *
 *  Unless you and Broadcom execute a separate written software license
 *  agreement governing use of this software, this software is licensed
 *  to you under the terms of the GNU General Public License version 2
 *  (the "GPL"), available at http://www.broadcom.com/licenses/GPLv2.php,
 *  with the following added to such license:
 *
 *     As a special exception, the copyright holders of this software give
 *     you permission to link this software with independent modules, and
 *     to copy and distribute the resulting executable under terms of your
 *     choice, provided that you also meet, for each linked independent
 *     module, the terms and conditions of the license of that module.
 *     An independent module is a module which is not derived from this
 *     software.  The special exception does not apply to any modifications
 *     of the software.
 *
 *  Not withstanding the above, under no circumstances may you combine
 *  this software in any way with any other Broadcom software provided
 *  under a license other than the GPL, without Broadcom's express prior
 *  written consent.
 *
 *  :>
 *
 *****************************************************************************/

#ifndef _BCM_TOPO_H_
#define _BCM_TOPO_H_

#include <bcmolt_conv.h>
#include <bcmos_system.h>
#include <bcmolt_model_types.h>
#include <bal_model_types.h>

#define BCM_TOPO_DEV_INVALID UINT8_MAX
#define BCM_TOPO_PON_INVALID UINT32_MAX
#define BCM_TOPO_ERR_INVALID UINT32_MAX
#define BCM_TOPO_MAX_NUM_OF_DEVS 8 /* Maxmimum of 8 devices per vOLT. */
#define BCM_TOPO_MAX_NUM_OF_PONS_PER_DEV 16
#define BCM_TOPO_MAX_NUM_OF_LOGICAL_PONS (BCM_TOPO_MAX_NUM_OF_DEVS * BCM_TOPO_MAX_NUM_OF_PONS_PER_DEV)

#define BCM_TOPO_FOR_EACH_DEV(device_id) \
    for (device_id = bcm_topo_dev_get_next(BCM_TOPO_DEV_INVALID); device_id != BCM_TOPO_DEV_INVALID; device_id = bcm_topo_dev_get_next(device_id))

#define BCM_TOPO_DEV_FOR_EACH_PON(device_id, pon) \
    for (pon = bcm_topo_pon_get_next(device_id, BCM_TOPO_PON_INVALID); pon != BCM_TOPO_PON_INVALID; pon = bcm_topo_pon_get_next(device_id, pon))

#define BCM_TOPO_FOR_EACH_PON(device_id, pon) \
    for (device_id = bcm_topo_dev_get_next(BCM_TOPO_DEV_INVALID); device_id != BCM_TOPO_DEV_INVALID; device_id = bcm_topo_dev_get_next(device_id)) \
        BCM_TOPO_DEV_FOR_EACH_PON(device_id, pon)

#define BCM_TOPO_FOR_EACH_ONU(pon, onu) \
    for (onu = 0; onu < bcm_topo_pon_get_max_num_of_onus(pon); onu++)

typedef enum
{
    BCM_TOPO_PON_MODE_GPON,
    BCM_TOPO_PON_MODE_XGPON,
    BCM_TOPO_PON_MODE_XGS,
    BCM_TOPO_PON_MODE_EPON_TDMA,
    BCM_TOPO_PON_MODE_EPON_1G,
    BCM_TOPO_PON_MODE_EPON_10G,
    BCM_TOPO_PON_MODE_INVALID,
    BCM_TOPO_PON_MODE__NUM_OF,
} bcm_topo_pon_mode;

typedef enum
{
    BCM_TOPO_PON_FAMILY_GPON, /* GPON, XGPON, XGS, NGPON2 */
    BCM_TOPO_PON_FAMILY_EPON,
    BCM_TOPO_PON_FAMILY_INVALID,
    BCM_TOPO_PON_FAMILY__NUM_OF,
} bcm_topo_pon_family;

typedef enum
{
    BCM_TOPO_PON_SUB_FAMILY_GPON, /* GPON only */
    BCM_TOPO_PON_SUB_FAMILY_XGPON, /* XGPON, XGS, NGPON2 */
    BCM_TOPO_PON_SUB_FAMILY_EPON,
    BCM_TOPO_PON_SUB_FAMILY_INVALID,
    BCM_TOPO_PON_SUB_FAMILY__NUM_OF,
} bcm_topo_pon_sub_family;

/* User context identifier - can be extended if more modules in the system use the bcm_topo.context. */
typedef enum
{
    BCM_TOPO_PON_CONTEXT_ID_MAC_UTIL,
    BCM_TOPO_PON_CONTEXT_ID_OMCI_SVC,
    BCM_TOPO_PON_CONTEXT_ID_RSC_MGR,
    BCM_TOPO_PON_CONTEXT_ID__NUM_OF,
} bcm_topo_pon_context_id;

typedef struct
{
    uint32_t num_of_devs;
    uint32_t num_of_pons_per_dev;
    bcm_topo_pon_mode pon_mode;
} bcm_topo_params;

/**
 * @brief Get the PON mode of a given device ID.
 * @param device_id            device ID
 *
 * @returns PON mode or Null on error
 */
const char *bcm_topo_dev_get_pon_mode_str(bcmolt_devid device_id);


/**
 * @brief Get the max number of pons of a given device ID.
 * @param device_id            device ID
 *
 * @returns the max number of pons or -1 on error
 */
int bcm_topo_dev_get_max_pon(bcmolt_devid device_id);

/**
 * @brief Get the PON mode of a given logical PON ID.
 * @param pon                 Logical PON ID
 *
 * @returns PON mode or BCM_TOPO_PON_MODE_INVALID on error
 */
bcm_topo_pon_mode bcm_topo_pon_get_pon_mode(uint32_t pon);

/**
 * @brief Get the PON family of a given logical PON ID.
 * @param pon                 Logical PON ID
 *
 * @returns PON mode or BCM_TOPO_PON_FAMILY_INVALID on error
 */
bcm_topo_pon_family bcm_topo_pon_get_pon_family(uint32_t pon);

/**
 * @brief Get the PON sub-family of a given logical PON ID.
 * @param pon                 Logical PON ID
 *
 * @returns PON mode or BCM_TOPO_PON_SUB_FAMILY_INVALID on error
 */
bcm_topo_pon_sub_family bcm_topo_pon_get_pon_sub_family(uint32_t pon);

/**
 * @brief Get the number of ONUs of a given logical PON ID.
 * @param pon                 Logical PON ID
 *
 * @returns Number of ONUs or BCM_TOPO_ERR_INVALID on error
 */
uint32_t bcm_topo_pon_get_max_num_of_onus(uint32_t pon);

/**
 * @brief Return whether a given logical PON is in the valid range.
 *
 * @returns BCMOS_TRUE is the given logical PON is in the valid range, BCMOS_FALSE otherwise
 */
bcmos_bool bcm_topo_pon_is_valid(uint32_t pon);

/**
 * @brief Traverse devices
 * @param device_id         Device iterator. Should be BCM_TOPO_DEV_INVALID at the beginning.
 *
 * @returns Next device, or BCM_TOPO_DEV_INVALID to mark that no more devices are available.
 */
bcmolt_devid bcm_topo_dev_get_next(bcmolt_devid device_id);

/**
 * @brief Traverse logical PONs within a given device.
 * @param device_id         Device id
 * @param pon               Logical PON iterator. Should be BCM_TOPO_PON_INVALID at the beginning.
 *
 * @returns Next logical PON on this device, or BCM_TOPO_PON_INVALID to mark that no more PONs are available.
 */
uint32_t bcm_topo_pon_get_next(bcmolt_devid device_id, uint32_t pon);

/**
 * @brief Get device ID and physical PON ID from logical PON ID.
 * @param logical_pon            logical PON ID
 * @param *device_id         Pointer to device id
 * @param *physical_pon           Pointer to physical PON ID
 *
 * @returns bcmos_errno
 * @note In general, the physical PON ID is used in the hardware directed function calls.
 */
bcmos_errno bcm_topo_pon_get_logical2physical(uint32_t logical_pon, bcmolt_devid *device_id, uint32_t *physical_pon);

/**
 * @brief Get logical PON ID from device ID and physical PON ID.
 * @param device_id           Device id
 * @param physical_pon             Physical PON ID
 * @param *logical_pon            Pointer to logical PON ID
 *
 * @returns bcmos_errno
 * @note In general, the logical PON ID is used in the BAL core directed function calls.
 */
bcmos_errno bcm_topo_pon_get_physical2logical(bcmolt_devid device_id, uint32_t physical_pon, uint32_t *logical_pon);

/**
 * @brief Set user context for a given logical PON ID.
 * @param pon                 Logical PON ID
 * @param pon_context_id      The identity of the module using the context
 * @param context             Pointer to user context
 *
 * @returns bcmos_errno
 */
bcmos_errno bcm_topo_pon_set_context(uint32_t pon, bcm_topo_pon_context_id pon_context_id, void *context);

/**
 * @brief Get user context for a given logical PON ID.
 * @param pon                 Logical PON ID
 * @param pon_context_id      The identity of the module using the context
 *
 * @returns User context or NULL if there's an error
 */
void *bcm_topo_pon_get_context(uint32_t pon, bcm_topo_pon_context_id pon_context_id);

/**
 * @brief Initialize topology module, either by arguments (probably received from CLI) or by a topology file. The rule is at least one of them must be specified and exist. If both are
 *        specified and exist, then we rely on topology file.
 *        Pay attention that when not using a file, the user have less freedom. For example:
 *        - The user won't be able to have devices with different PON modes.
 *        - The user won't be able to have devices with different number of PONs.
 *        - The user won't be able to map logical PON to (device, physical PON). We assume that logical_pon = (device * num_of_pons_per_dev) + physical_pon.
 * @param params              Topology parameters. If NULL, we rely on topology file.
 * @param topo_filename       File containing bcm_topo.configuration in a .csv format
 *                            The columns are:
 *                            Logical PON ID, PON Mode, Physical Device ID, Physical PON ID
 *
 * @returns bcmos_errno
 */
bcmos_errno bcm_topo_init(bcm_topo_params *params, const char *topo_filename);

/**
 * @brief Returns whether the topology module has been initialized.
 *
 * @returns BCMOS_TRUE if topology module has been initialized
 */
bcmos_bool bcm_topo_is_initialized(void);

#define BCM_TOPO_MAX_NNI_PORTS 16

/**
 * @brief Set the max number of nnis of a given device ID.
 * @param device_id            device ID
 * @param num_nni_ports        nni ports on the device
 *
 * @returns BCMOS_TRUE 
 */
bcmos_bool bcm_topo_dev_set_max_nni(bcmolt_devid device_id, uint32_t num_nni_ports);

/**
 * @brief Get the max number of nnis of a given device ID.
 * @param device_id            device ID
 * @param p_num_nni_ports      pointer for the retrieved nni ports on the device
 *
 * @returns BCMOS_TRUE 
 */
bcmos_bool bcm_topo_dev_get_max_nni(bcmolt_devid device_id, uint32_t *p_num_nni_ports);

/**
 * @brief Return whether a given logical NNI is in the valid range.
 *
 * @returns BCMOS_TRUE is the given logical NNI is in the valid range, BCMOS_FALSE otherwise
 */
bcmos_bool bcm_topo_nni_is_valid(uint32_t nni);

#endif

