#include <bcmos_system.h>
#include <bal_api.h>
#include <bcmcli.h>
#include "bal_api_cli_helpers.h"
#include "bal_api_cli_handlers.h"

bcmcli_session *bcmbal_apicli_log_session = NULL;

typedef struct
{
    uint8_t *start;
    uint32_t used;
} bcmbal_apicli_byte_pool;

static bcmos_errno bcmbal_apicli_byte_pool_create(bcmbal_apicli_byte_pool *buf)
{
    buf->used = 0;
    buf->start = bcmos_calloc(BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);
    return (buf->start == NULL) ? BCM_ERR_NOMEM : BCM_ERR_OK;
}

static void bcmbal_apicli_byte_pool_destroy(bcmbal_apicli_byte_pool *buf)
{
    bcmos_free(buf->start);
}

static void *bcmbal_apicli_byte_pool_calloc(bcmbal_apicli_byte_pool *buf, uint32_t num_bytes)
{
    void *ret;
    if (buf->used + num_bytes > BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE)
    {
        return NULL;
    }

    ret = buf->start + buf->used;
    buf->used += num_bytes;
    memset(ret, 0, num_bytes);
    return ret;
}

/*
 * Start/end banners - these are specially formatted so listening apps can easily tell where API handling starts/ends.
 */
static void bcmbal_apicli_print_start(bcmcli_session *session, const char *api_name)
{
    bcmcli_print(session, "[-- API Start: %s --]\n", api_name);
}

static void bcmbal_apicli_print_data_start(bcmcli_session *session)
{
    bcmcli_print(session, "[-- API Message Data --]\n");
}

static void bcmbal_apicli_print_complete(bcmcli_session *session, bcmos_errno err, const char *err_text)
{
    if (err != BCM_ERR_OK && err_text != NULL && err_text[0] != '\0')
    {
        bcmcli_print(session, "ERROR: %s", err_text);
    }

    bcmcli_print(session, "[-- API Complete: %d (%s) --]\n", err, bcmos_strerror(err));
}

static int bcmbal_apicli_session_write_cb(bcmcli_session *cli_session, const char *buf, uint32_t size)
{
    bcmcli_log(buf, "%.*s", size, buf);
    return (int)size;
}

/* Logs a property value to the CLI log in such a way that it is a valid RHS in an initializer.  For a primitve, this
 * will just print the value (e.g. "42").  For a struct, it will emit all members in between curly braces. */
static void bcmbal_apicli_log_prop_val(bcmolt_obj_id obj, bcmolt_mgt_group group, uint16_t subgroup, uint16_t prop, void *value)
{
    bcmos_errno err;
    const bcmbal_apicli_prop_descr *prop_descr;

    if (bcmbal_apicli_log_session == NULL)
    {
        static bcmcli_session_parm session_params = { .write = bcmbal_apicli_session_write_cb };

        err = bcmcli_session_open(&session_params, &bcmbal_apicli_log_session);
        if (err != BCM_ERR_OK)
        {
            bcmos_trace(BCMOS_TRACE_LEVEL_ERROR, "Error opening session: %s", bcmos_strerror(err));
            return;
        }
    }

    err = bcmbal_apicli_object_property(obj, group, subgroup, prop, &prop_descr);
    if (err != BCM_ERR_OK)
    {
        bcmos_trace(BCMOS_TRACE_LEVEL_ERROR, "Error getting info for property: %s", bcmos_strerror(err));
        return;
    }

    err = bcmbal_apicli_dump_prop_initializer(bcmbal_apicli_log_session, prop_descr, value);
    if (err != BCM_ERR_OK)
    {
        bcmos_trace(BCMOS_TRACE_LEVEL_ERROR, "Error printing property: %s", bcmos_strerror(err));
    }
}

static inline bcmos_ipv4_address bcmbal_apicli_unumber_to_ipv4(uint32_t num)
{
    bcmos_ipv4_address ip = { .u32 = num };
    return ip;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_access_terminal_cfg_get(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_access_terminal_cfg cfg;         /**< declare main API struct */
    bcmbal_access_terminal_key key = { };   /**< declare key */
    bcmcli_log("bcmbal_access_terminal_cfg cfg;\n");
    bcmcli_log("bcmbal_access_terminal_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_get");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, access_terminal, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, access_terminal, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "admin_state");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, access_terminal, admin_state);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, access_terminal, admin_state);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "oper_status");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, access_terminal, oper_status);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, access_terminal, oper_status);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "iwf_mode");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, access_terminal, iwf_mode);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, access_terminal, iwf_mode);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_CFG_PROP_IS_SET(&cfg, access_terminal, admin_state) && !BCMBAL_CFG_PROP_IS_SET(&cfg, access_terminal, oper_status) && !BCMBAL_CFG_PROP_IS_SET(&cfg, access_terminal, iwf_mode))
    {
        BCMBAL_CFG_PROP_GET(&cfg, access_terminal, all_properties);
        bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, access_terminal, all_properties);\n");
    }

    /* call API */
    err = bcmbal_cfg_get(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_get(&cfg.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &cfg.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_access_terminal_cfg_set(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_access_terminal_cfg cfg;         /**< declare main API struct */
    bcmbal_access_terminal_key key = { };   /**< declare key */
    bcmcli_log("bcmbal_access_terminal_cfg cfg;\n");
    bcmcli_log("bcmbal_access_terminal_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_set");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, access_terminal, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, access_terminal, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "admin_state");
    if (cli_parm != NULL)
    {
        bcmbal_state val;
        val = (bcmbal_state) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, access_terminal, admin_state, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, access_terminal, admin_state, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_ACCESS_TERMINAL, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_ACCESS_TERMINAL_CFG_ID_ADMIN_STATE, &val);
        bcmcli_log(");\n");
    }

    /* call API */
    err = bcmbal_cfg_set(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_set(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_access_terminal_cfg_clear(bcmcli_session *session)
{
    bcmos_errno err;
    bcmbal_access_terminal_cfg cfg;         /**< declare main API struct */
    bcmbal_access_terminal_key key = { };   /**< declare key */
    bcmcli_log("bcmbal_access_terminal_cfg cfg;\n");
    bcmcli_log("bcmbal_access_terminal_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_clear");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, access_terminal, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, access_terminal, key);\n");

    /* call API */
    err = bcmbal_cfg_clear(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_clear(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_flow_cfg_get(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_flow_cfg cfg;        /**< declare main API struct */
    bcmbal_flow_key key = { };  /**< declare key */
    bcmcli_log("bcmbal_flow_cfg cfg;\n");
    bcmcli_log("bcmbal_flow_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_get");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "flow_id");
    if (cli_parm != NULL)
    {
        key.flow_id = (bcmbal_flow_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "flow_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.flow_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_FLOW_KEY_ID_FLOW_ID, &key.flow_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "flow_type");
    if (cli_parm != NULL)
    {
        key.flow_type = (bcmbal_flow_type) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "flow_type is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.flow_type = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_FLOW_KEY_ID_FLOW_TYPE, &key.flow_type);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, flow, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, flow, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "admin_state");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, admin_state);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, admin_state);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "oper_status");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, oper_status);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, oper_status);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "access_int_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, access_int_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, access_int_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "network_int_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, network_int_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, network_int_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "sub_term_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, sub_term_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, sub_term_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "sub_term_uni_idx");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, sub_term_uni_idx);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, sub_term_uni_idx);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "svc_port_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, svc_port_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, svc_port_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "agg_port_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, agg_port_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, agg_port_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "resolve_mac");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, resolve_mac);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, resolve_mac);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "classifier");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, classifier);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, classifier);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "action");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, action);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, action);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "sla");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, sla);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, sla);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "cookie");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, cookie);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, cookie);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "priority");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, priority);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, priority);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "group_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, group_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, group_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "queue");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, flow, queue);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, queue);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_CFG_PROP_IS_SET(&cfg, flow, admin_state) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, oper_status) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, access_int_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, network_int_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, sub_term_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, sub_term_uni_idx) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, svc_port_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, agg_port_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, resolve_mac) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, classifier) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, action) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, sla) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, cookie) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, priority) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, group_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, flow, queue))
    {
        BCMBAL_CFG_PROP_GET(&cfg, flow, all_properties);
        bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, flow, all_properties);\n");
    }

    /* call API */
    err = bcmbal_cfg_get(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_get(&cfg.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &cfg.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_flow_cfg_set(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_flow_cfg cfg;        /**< declare main API struct */
    bcmbal_flow_key key = { };  /**< declare key */
    bcmcli_log("bcmbal_flow_cfg cfg;\n");
    bcmcli_log("bcmbal_flow_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_set");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "flow_id");
    if (cli_parm != NULL)
    {
        key.flow_id = (bcmbal_flow_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "flow_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.flow_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_FLOW_KEY_ID_FLOW_ID, &key.flow_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "flow_type");
    if (cli_parm != NULL)
    {
        key.flow_type = (bcmbal_flow_type) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "flow_type is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.flow_type = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_FLOW_KEY_ID_FLOW_TYPE, &key.flow_type);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, flow, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, flow, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "admin_state");
    if (cli_parm != NULL)
    {
        bcmbal_state val;
        val = (bcmbal_state) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, flow, admin_state, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, admin_state, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_ADMIN_STATE, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "access_int_id");
    if (cli_parm != NULL)
    {
        bcmbal_intf_id val;
        val = (bcmbal_intf_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, flow, access_int_id, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, access_int_id, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_ACCESS_INT_ID, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "network_int_id");
    if (cli_parm != NULL)
    {
        bcmbal_intf_id val;
        val = (bcmbal_intf_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, flow, network_int_id, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, network_int_id, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_NETWORK_INT_ID, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "sub_term_id");
    if (cli_parm != NULL)
    {
        bcmbal_sub_id val;
        val = (bcmbal_sub_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, flow, sub_term_id, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, sub_term_id, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_SUB_TERM_ID, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "sub_term_uni_idx");
    if (cli_parm != NULL)
    {
        uint8_t val;
        val = cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, flow, sub_term_uni_idx, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, sub_term_uni_idx, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_SUB_TERM_UNI_IDX, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "svc_port_id");
    if (cli_parm != NULL)
    {
        bcmbal_service_port_id val;
        val = (bcmbal_service_port_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, flow, svc_port_id, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, svc_port_id, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_SVC_PORT_ID, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "agg_port_id");
    if (cli_parm != NULL)
    {
        bcmbal_aggregation_port_id val;
        val = (bcmbal_aggregation_port_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, flow, agg_port_id, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, agg_port_id, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_AGG_PORT_ID, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "resolve_mac");
    if (cli_parm != NULL)
    {
        bcmos_bool val;
        val = cli_parm->value.number;
        BCMBAL_CFG_PROP_SET(&cfg, flow, resolve_mac, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, resolve_mac, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_RESOLVE_MAC, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "classifier.");
    if (cli_parm != NULL)
    {
        bcmbal_classifier val = { };
        cli_parm = bcmcli_find_named_parm(session, "classifier.o_tpid");
        if (cli_parm != NULL)
        {
            val.o_tpid = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_O_TPID;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.o_vid");
        if (cli_parm != NULL)
        {
            val.o_vid = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_O_VID;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.i_tpid");
        if (cli_parm != NULL)
        {
            val.i_tpid = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_I_TPID;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.i_vid");
        if (cli_parm != NULL)
        {
            val.i_vid = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_I_VID;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.o_pbits");
        if (cli_parm != NULL)
        {
            val.o_pbits = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_O_PBITS;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.i_pbits");
        if (cli_parm != NULL)
        {
            val.i_pbits = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_I_PBITS;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.ether_type");
        if (cli_parm != NULL)
        {
            val.ether_type = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_ETHER_TYPE;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.dst_mac");
        if (cli_parm != NULL)
        {
            val.dst_mac = cli_parm->value.mac;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_DST_MAC;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.src_mac");
        if (cli_parm != NULL)
        {
            val.src_mac = cli_parm->value.mac;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_SRC_MAC;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.ip_proto");
        if (cli_parm != NULL)
        {
            val.ip_proto = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_IP_PROTO;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.dst_ip");
        if (cli_parm != NULL)
        {
            val.dst_ip = bcmbal_apicli_unumber_to_ipv4(cli_parm->value.unumber);
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_DST_IP;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.src_ip");
        if (cli_parm != NULL)
        {
            val.src_ip = bcmbal_apicli_unumber_to_ipv4(cli_parm->value.unumber);
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_SRC_IP;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.src_port");
        if (cli_parm != NULL)
        {
            val.src_port = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_SRC_PORT;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.dst_port");
        if (cli_parm != NULL)
        {
            val.dst_port = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_DST_PORT;
        }

        cli_parm = bcmcli_find_named_parm(session, "classifier.pkt_tag_type");
        if (cli_parm != NULL)
        {
            val.pkt_tag_type = (bcmbal_pkt_tag_type) cli_parm->value.enum_val;
            val.presence_mask = val.presence_mask | BCMBAL_CLASSIFIER_ID_PKT_TAG_TYPE;
        }

        BCMBAL_CFG_PROP_SET(&cfg, flow, classifier, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_classifier val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_CLASSIFIER, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, classifier, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "action.");
    if (cli_parm != NULL)
    {
        bcmbal_action val = { };
        cli_parm = bcmcli_find_named_parm(session, "action.cmds_bitmask");
        if (cli_parm != NULL)
        {
            val.cmds_bitmask = (bcmbal_action_cmd_id) cli_parm->value.enum_val;
            val.presence_mask = val.presence_mask | BCMBAL_ACTION_ID_CMDS_BITMASK;
        }

        cli_parm = bcmcli_find_named_parm(session, "action.o_vid");
        if (cli_parm != NULL)
        {
            val.o_vid = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_ACTION_ID_O_VID;
        }

        cli_parm = bcmcli_find_named_parm(session, "action.o_pbits");
        if (cli_parm != NULL)
        {
            val.o_pbits = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_ACTION_ID_O_PBITS;
        }

        cli_parm = bcmcli_find_named_parm(session, "action.o_tpid");
        if (cli_parm != NULL)
        {
            val.o_tpid = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_ACTION_ID_O_TPID;
        }

        cli_parm = bcmcli_find_named_parm(session, "action.i_vid");
        if (cli_parm != NULL)
        {
            val.i_vid = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_ACTION_ID_I_VID;
        }

        cli_parm = bcmcli_find_named_parm(session, "action.i_pbits");
        if (cli_parm != NULL)
        {
            val.i_pbits = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_ACTION_ID_I_PBITS;
        }

        cli_parm = bcmcli_find_named_parm(session, "action.i_tpid");
        if (cli_parm != NULL)
        {
            val.i_tpid = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_ACTION_ID_I_TPID;
        }

        BCMBAL_CFG_PROP_SET(&cfg, flow, action, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_action val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_ACTION, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, action, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "sla.");
    if (cli_parm != NULL)
    {
        bcmbal_sla val = { };
        cli_parm = bcmcli_find_named_parm(session, "sla.min_rate");
        if (cli_parm != NULL)
        {
            val.min_rate = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_SLA_ID_MIN_RATE;
        }

        cli_parm = bcmcli_find_named_parm(session, "sla.max_rate");
        if (cli_parm != NULL)
        {
            val.max_rate = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_SLA_ID_MAX_RATE;
        }

        BCMBAL_CFG_PROP_SET(&cfg, flow, sla, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_sla val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_SLA, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, sla, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "cookie");
    if (cli_parm != NULL)
    {
        bcmbal_cookie val;
        val = (bcmbal_cookie) cli_parm->value.unumber64;
        BCMBAL_CFG_PROP_SET(&cfg, flow, cookie, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, cookie, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_COOKIE, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "priority");
    if (cli_parm != NULL)
    {
        uint16_t val;
        val = cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, flow, priority, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, priority, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_PRIORITY, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "group_id");
    if (cli_parm != NULL)
    {
        bcmbal_group_id val;
        val = (bcmbal_group_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, flow, group_id, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, group_id, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_GROUP_ID, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "queue.");
    if (cli_parm != NULL)
    {
        bcmbal_tm_queue_ref val = { };
        cli_parm = bcmcli_find_named_parm(session, "queue.sched_id");
        if (cli_parm != NULL)
        {
            val.sched_id = (bcmbal_tm_sched_id) cli_parm->value.unumber;
        }
        else
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "queue.sched_id is not set\n");
            return BCM_ERR_PARM;
        }

        cli_parm = bcmcli_find_named_parm(session, "queue.queue_id");
        if (cli_parm != NULL)
        {
            val.queue_id = (bcmbal_tm_queue_id) cli_parm->value.unumber;
        }
        else
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "queue.queue_id is not set\n");
            return BCM_ERR_PARM;
        }

        BCMBAL_CFG_PROP_SET(&cfg, flow, queue, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_tm_queue_ref val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_FLOW_CFG_ID_QUEUE, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, flow, queue, val);\n");
        bcmcli_log("}\n");
    }

    /* call API */
    err = bcmbal_cfg_set(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_set(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_flow_cfg_clear(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_flow_cfg cfg;        /**< declare main API struct */
    bcmbal_flow_key key = { };  /**< declare key */
    bcmcli_log("bcmbal_flow_cfg cfg;\n");
    bcmcli_log("bcmbal_flow_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_clear");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "flow_id");
    if (cli_parm != NULL)
    {
        key.flow_id = (bcmbal_flow_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "flow_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.flow_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_FLOW_KEY_ID_FLOW_ID, &key.flow_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "flow_type");
    if (cli_parm != NULL)
    {
        key.flow_type = (bcmbal_flow_type) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "flow_type is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.flow_type = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_FLOW_KEY_ID_FLOW_TYPE, &key.flow_type);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, flow, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, flow, key);\n");

    /* call API */
    err = bcmbal_cfg_clear(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_clear(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_flow_stat_get(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_flow_stat stat;      /**< declare main API struct */
    bcmbal_flow_key key = { };  /**< declare key */
    bcmcli_log("bcmbal_flow_stat stat;\n");
    bcmcli_log("bcmbal_flow_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_stat_get");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "flow_id");
    if (cli_parm != NULL)
    {
        key.flow_id = (bcmbal_flow_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "flow_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.flow_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_FLOW_KEY_ID_FLOW_ID, &key.flow_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "flow_type");
    if (cli_parm != NULL)
    {
        key.flow_type = (bcmbal_flow_type) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "flow_type is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.flow_type = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_FLOW, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_FLOW_KEY_ID_FLOW_TYPE, &key.flow_type);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_STAT_INIT(&stat, flow, key);
    bcmcli_log("BCMBAL_STAT_INIT(&stat, flow, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "rx_packets");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, flow, rx_packets);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, flow, rx_packets);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "rx_bytes");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, flow, rx_bytes);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, flow, rx_bytes);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "tx_packets");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, flow, tx_packets);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, flow, tx_packets);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "tx_bytes");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, flow, tx_bytes);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, flow, tx_bytes);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_STAT_PROP_IS_SET(&stat, flow, rx_packets) && !BCMBAL_STAT_PROP_IS_SET(&stat, flow, rx_bytes) && !BCMBAL_STAT_PROP_IS_SET(&stat, flow, tx_packets) && !BCMBAL_STAT_PROP_IS_SET(&stat, flow, tx_bytes))
    {
        BCMBAL_STAT_PROP_GET(&stat, flow, all_properties);
        bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, flow, all_properties);\n");
    }

    /* call API */
    err = bcmbal_stat_get(&stat.hdr);
    bcmcli_log("bcmbal_stat_get(&stat.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &stat.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_group_cfg_get(bcmcli_session *session, bcmbal_apicli_byte_pool *byte_pool)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_group_cfg cfg;       /**< declare main API struct */
    bcmbal_group_key key = { }; /**< declare key */
    uint8_t *list_mem;          /**< declare memory buffer for variable-sized lists */
    bcmcli_log("bcmbal_group_cfg cfg;\n");
    bcmcli_log("bcmbal_group_key key = {  };\n");
    bcmcli_log("uint8_t* list_mem;\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_get");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "group_id");
    if (cli_parm != NULL)
    {
        key.group_id = (bcmbal_group_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "group_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.group_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_GROUP, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_GROUP_KEY_ID_GROUP_ID, &key.group_id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, group, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, group, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "members_cmd");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, group, members_cmd);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, group, members_cmd);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "members");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, group, members);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, group, members);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "cookie");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, group, cookie);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, group, cookie);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "flows");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, group, flows);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, group, flows);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "owner");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, group, owner);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, group, owner);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_CFG_PROP_IS_SET(&cfg, group, members_cmd) && !BCMBAL_CFG_PROP_IS_SET(&cfg, group, members) && !BCMBAL_CFG_PROP_IS_SET(&cfg, group, cookie) && !BCMBAL_CFG_PROP_IS_SET(&cfg, group, flows) && !BCMBAL_CFG_PROP_IS_SET(&cfg, group, owner))
    {
        BCMBAL_CFG_PROP_GET(&cfg, group, all_properties);
        bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, group, all_properties);\n");
    }

    /* set memory to use for variable-sized lists */
    list_mem = bcmbal_apicli_byte_pool_calloc(byte_pool, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);
    if (list_mem == NULL)
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_NOMEM, "\n");
        return BCM_ERR_NOMEM;
    }

    bcmcli_log("list_mem = bcmos_calloc(BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);\n");
    BCMBAL_CFG_LIST_BUF_SET(&cfg, group, list_mem, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);
    bcmcli_log("BCMBAL_CFG_LIST_BUF_SET(&cfg, group, list_mem, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);\n");

    /* call API */
    err = bcmbal_cfg_get(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_get(&cfg.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &cfg.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_group_cfg_set(bcmcli_session *session, bcmbal_apicli_byte_pool *byte_pool)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_group_cfg cfg;       /**< declare main API struct */
    bcmbal_group_key key = { }; /**< declare key */
    bcmcli_log("bcmbal_group_cfg cfg;\n");
    bcmcli_log("bcmbal_group_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_set");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "group_id");
    if (cli_parm != NULL)
    {
        key.group_id = (bcmbal_group_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "group_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.group_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_GROUP, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_GROUP_KEY_ID_GROUP_ID, &key.group_id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, group, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, group, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "members_cmd");
    if (cli_parm != NULL)
    {
        bcmbal_group_member_cmd val;
        val = (bcmbal_group_member_cmd) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, group, members_cmd, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, group, members_cmd, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_GROUP, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_GROUP_CFG_ID_MEMBERS_CMD, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "members.");
    if (cli_parm != NULL)
    {
        bcmbal_group_member_info_list_u16 val = { };
        int32_t i0;
        val.val = bcmbal_apicli_byte_pool_calloc(byte_pool, cli_parm->array_size * sizeof(*val.val));
        if (val.val == NULL)
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_NOMEM, "\n");
            return BCM_ERR_NOMEM;
        }

        val.len = cli_parm->array_size;
        cli_parm = bcmcli_find_named_parm(session, "members.intf_id");
        if (cli_parm != NULL)
        {
            if (cli_parm->array_size != val.len)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "members.intf_id is a different size than other arrays in the struct\n");
                return BCM_ERR_PARM;
            }

            for (i0 = 0; i0 < val.len; i0++)
            {
                val.val[i0].intf_id = (bcmbal_intf_id) cli_parm->values[i0].unumber;
            }
        }

        cli_parm = bcmcli_find_named_parm(session, "members.svc_port_id");
        if (cli_parm != NULL)
        {
            if (cli_parm->array_size != val.len)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "members.svc_port_id is a different size than other arrays in the struct\n");
                return BCM_ERR_PARM;
            }

            for (i0 = 0; i0 < val.len; i0++)
            {
                val.val[i0].svc_port_id = (bcmbal_service_port_id) cli_parm->values[i0].unumber;
            }
        }

        cli_parm = bcmcli_find_named_parm(session, "members.action.cmds_bitmask");
        if (cli_parm != NULL)
        {
            if (cli_parm->array_size != val.len)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "members.action.cmds_bitmask is a different size than other arrays in the struct\n");
                return BCM_ERR_PARM;
            }

            for (i0 = 0; i0 < val.len; i0++)
            {
                if (bcmcli_parm_value_is_set(session, cli_parm, i0))
                {
                    val.val[i0].action.cmds_bitmask = (bcmbal_action_cmd_id) cli_parm->values[i0].enum_val;
                    val.val[i0].action.presence_mask = val.val[i0].action.presence_mask | BCMBAL_ACTION_ID_CMDS_BITMASK;
                }
            }
        }

        cli_parm = bcmcli_find_named_parm(session, "members.action.o_vid");
        if (cli_parm != NULL)
        {
            if (cli_parm->array_size != val.len)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "members.action.o_vid is a different size than other arrays in the struct\n");
                return BCM_ERR_PARM;
            }

            for (i0 = 0; i0 < val.len; i0++)
            {
                if (bcmcli_parm_value_is_set(session, cli_parm, i0))
                {
                    val.val[i0].action.o_vid = cli_parm->values[i0].unumber;
                    val.val[i0].action.presence_mask = val.val[i0].action.presence_mask | BCMBAL_ACTION_ID_O_VID;
                }
            }
        }

        cli_parm = bcmcli_find_named_parm(session, "members.action.o_pbits");
        if (cli_parm != NULL)
        {
            if (cli_parm->array_size != val.len)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "members.action.o_pbits is a different size than other arrays in the struct\n");
                return BCM_ERR_PARM;
            }

            for (i0 = 0; i0 < val.len; i0++)
            {
                if (bcmcli_parm_value_is_set(session, cli_parm, i0))
                {
                    val.val[i0].action.o_pbits = cli_parm->values[i0].unumber;
                    val.val[i0].action.presence_mask = val.val[i0].action.presence_mask | BCMBAL_ACTION_ID_O_PBITS;
                }
            }
        }

        cli_parm = bcmcli_find_named_parm(session, "members.action.o_tpid");
        if (cli_parm != NULL)
        {
            if (cli_parm->array_size != val.len)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "members.action.o_tpid is a different size than other arrays in the struct\n");
                return BCM_ERR_PARM;
            }

            for (i0 = 0; i0 < val.len; i0++)
            {
                if (bcmcli_parm_value_is_set(session, cli_parm, i0))
                {
                    val.val[i0].action.o_tpid = cli_parm->values[i0].unumber;
                    val.val[i0].action.presence_mask = val.val[i0].action.presence_mask | BCMBAL_ACTION_ID_O_TPID;
                }
            }
        }

        cli_parm = bcmcli_find_named_parm(session, "members.action.i_vid");
        if (cli_parm != NULL)
        {
            if (cli_parm->array_size != val.len)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "members.action.i_vid is a different size than other arrays in the struct\n");
                return BCM_ERR_PARM;
            }

            for (i0 = 0; i0 < val.len; i0++)
            {
                if (bcmcli_parm_value_is_set(session, cli_parm, i0))
                {
                    val.val[i0].action.i_vid = cli_parm->values[i0].unumber;
                    val.val[i0].action.presence_mask = val.val[i0].action.presence_mask | BCMBAL_ACTION_ID_I_VID;
                }
            }
        }

        cli_parm = bcmcli_find_named_parm(session, "members.action.i_pbits");
        if (cli_parm != NULL)
        {
            if (cli_parm->array_size != val.len)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "members.action.i_pbits is a different size than other arrays in the struct\n");
                return BCM_ERR_PARM;
            }

            for (i0 = 0; i0 < val.len; i0++)
            {
                if (bcmcli_parm_value_is_set(session, cli_parm, i0))
                {
                    val.val[i0].action.i_pbits = cli_parm->values[i0].unumber;
                    val.val[i0].action.presence_mask = val.val[i0].action.presence_mask | BCMBAL_ACTION_ID_I_PBITS;
                }
            }
        }

        cli_parm = bcmcli_find_named_parm(session, "members.action.i_tpid");
        if (cli_parm != NULL)
        {
            if (cli_parm->array_size != val.len)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "members.action.i_tpid is a different size than other arrays in the struct\n");
                return BCM_ERR_PARM;
            }

            for (i0 = 0; i0 < val.len; i0++)
            {
                if (bcmcli_parm_value_is_set(session, cli_parm, i0))
                {
                    val.val[i0].action.i_tpid = cli_parm->values[i0].unumber;
                    val.val[i0].action.presence_mask = val.val[i0].action.presence_mask | BCMBAL_ACTION_ID_I_TPID;
                }
            }
        }

        cli_parm = bcmcli_find_named_parm(session, "members.queue.sched_id");
        if (cli_parm != NULL)
        {
            if (cli_parm->array_size != val.len)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "members.queue.sched_id is a different size than other arrays in the struct\n");
                return BCM_ERR_PARM;
            }

            for (i0 = 0; i0 < val.len; i0++)
            {
                val.val[i0].queue.sched_id = (bcmbal_tm_sched_id) cli_parm->values[i0].unumber;
            }
        }

        cli_parm = bcmcli_find_named_parm(session, "members.queue.queue_id");
        if (cli_parm != NULL)
        {
            if (cli_parm->array_size != val.len)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "members.queue.queue_id is a different size than other arrays in the struct\n");
                return BCM_ERR_PARM;
            }

            for (i0 = 0; i0 < val.len; i0++)
            {
                val.val[i0].queue.queue_id = (bcmbal_tm_queue_id) cli_parm->values[i0].unumber;
            }
        }

        BCMBAL_CFG_PROP_SET(&cfg, group, members, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_group_member_info_list_u16 val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_GROUP, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_GROUP_CFG_ID_MEMBERS, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, group, members, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "cookie");
    if (cli_parm != NULL)
    {
        bcmbal_cookie val;
        val = (bcmbal_cookie) cli_parm->value.unumber64;
        BCMBAL_CFG_PROP_SET(&cfg, group, cookie, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, group, cookie, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_GROUP, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_GROUP_CFG_ID_COOKIE, &val);
        bcmcli_log(");\n");
    }

    /* call API */
    err = bcmbal_cfg_set(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_set(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_group_cfg_clear(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_group_cfg cfg;       /**< declare main API struct */
    bcmbal_group_key key = { }; /**< declare key */
    bcmcli_log("bcmbal_group_cfg cfg;\n");
    bcmcli_log("bcmbal_group_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_clear");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "group_id");
    if (cli_parm != NULL)
    {
        key.group_id = (bcmbal_group_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "group_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.group_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_GROUP, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_GROUP_KEY_ID_GROUP_ID, &key.group_id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, group, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, group, key);\n");

    /* call API */
    err = bcmbal_cfg_clear(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_clear(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_interface_cfg_get(bcmcli_session *session, bcmbal_apicli_byte_pool *byte_pool)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_interface_cfg cfg;       /**< declare main API struct */
    bcmbal_interface_key key = { }; /**< declare key */
    uint8_t *list_mem;              /**< declare memory buffer for variable-sized lists */
    bcmcli_log("bcmbal_interface_cfg cfg;\n");
    bcmcli_log("bcmbal_interface_key key = {  };\n");
    bcmcli_log("uint8_t* list_mem;\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_get");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "intf_id");
    if (cli_parm != NULL)
    {
        key.intf_id = cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_INTERFACE_KEY_ID_INTF_ID, &key.intf_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "intf_type");
    if (cli_parm != NULL)
    {
        key.intf_type = (bcmbal_intf_type) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_type is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_type = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_INTERFACE_KEY_ID_INTF_TYPE, &key.intf_type);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, interface, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, interface, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "admin_state");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, interface, admin_state);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, admin_state);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "oper_status");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, interface, oper_status);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, oper_status);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "min_data_agg_port_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, interface, min_data_agg_port_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, min_data_agg_port_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "min_data_svc_port_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, interface, min_data_svc_port_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, min_data_svc_port_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "transceiver_type");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, interface, transceiver_type);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, transceiver_type);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "ds_miss_mode");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, interface, ds_miss_mode);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, ds_miss_mode);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "mtu");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, interface, mtu);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, mtu);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "flow_control");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, interface, flow_control);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, flow_control);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "ds_tm");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, interface, ds_tm);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, ds_tm);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "us_tm");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, interface, us_tm);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, us_tm);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "sub_term_id_list");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, interface, sub_term_id_list);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, sub_term_id_list);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_CFG_PROP_IS_SET(&cfg, interface, admin_state) && !BCMBAL_CFG_PROP_IS_SET(&cfg, interface, oper_status) && !BCMBAL_CFG_PROP_IS_SET(&cfg, interface, min_data_agg_port_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, interface, min_data_svc_port_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, interface, transceiver_type) && !BCMBAL_CFG_PROP_IS_SET(&cfg, interface, ds_miss_mode) && !BCMBAL_CFG_PROP_IS_SET(&cfg, interface, mtu) && !BCMBAL_CFG_PROP_IS_SET(&cfg, interface, flow_control) && !BCMBAL_CFG_PROP_IS_SET(&cfg, interface, ds_tm) && !BCMBAL_CFG_PROP_IS_SET(&cfg, interface, us_tm) && !BCMBAL_CFG_PROP_IS_SET(&cfg, interface, sub_term_id_list))
    {
        BCMBAL_CFG_PROP_GET(&cfg, interface, all_properties);
        bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, interface, all_properties);\n");
    }

    /* set memory to use for variable-sized lists */
    list_mem = bcmbal_apicli_byte_pool_calloc(byte_pool, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);
    if (list_mem == NULL)
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_NOMEM, "\n");
        return BCM_ERR_NOMEM;
    }

    bcmcli_log("list_mem = bcmos_calloc(BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);\n");
    BCMBAL_CFG_LIST_BUF_SET(&cfg, interface, list_mem, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);
    bcmcli_log("BCMBAL_CFG_LIST_BUF_SET(&cfg, interface, list_mem, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);\n");

    /* call API */
    err = bcmbal_cfg_get(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_get(&cfg.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &cfg.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_interface_cfg_set(bcmcli_session *session, bcmbal_apicli_byte_pool *byte_pool)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_interface_cfg cfg;       /**< declare main API struct */
    bcmbal_interface_key key = { }; /**< declare key */
    bcmcli_log("bcmbal_interface_cfg cfg;\n");
    bcmcli_log("bcmbal_interface_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_set");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "intf_id");
    if (cli_parm != NULL)
    {
        key.intf_id = cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_INTERFACE_KEY_ID_INTF_ID, &key.intf_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "intf_type");
    if (cli_parm != NULL)
    {
        key.intf_type = (bcmbal_intf_type) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_type is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_type = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_INTERFACE_KEY_ID_INTF_TYPE, &key.intf_type);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, interface, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, interface, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "admin_state");
    if (cli_parm != NULL)
    {
        bcmbal_state val;
        val = (bcmbal_state) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, interface, admin_state, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, interface, admin_state, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_INTERFACE_CFG_ID_ADMIN_STATE, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "min_data_agg_port_id");
    if (cli_parm != NULL)
    {
        bcmbal_aggregation_port_id val;
        val = (bcmbal_aggregation_port_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, interface, min_data_agg_port_id, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, interface, min_data_agg_port_id, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_INTERFACE_CFG_ID_MIN_DATA_AGG_PORT_ID, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "min_data_svc_port_id");
    if (cli_parm != NULL)
    {
        bcmbal_service_port_id val;
        val = (bcmbal_service_port_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, interface, min_data_svc_port_id, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, interface, min_data_svc_port_id, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_INTERFACE_CFG_ID_MIN_DATA_SVC_PORT_ID, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "transceiver_type");
    if (cli_parm != NULL)
    {
        bcmbal_trx_type val;
        val = (bcmbal_trx_type) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, interface, transceiver_type, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, interface, transceiver_type, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_INTERFACE_CFG_ID_TRANSCEIVER_TYPE, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "ds_miss_mode");
    if (cli_parm != NULL)
    {
        bcmbal_ds_miss_mode val;
        val = (bcmbal_ds_miss_mode) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, interface, ds_miss_mode, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, interface, ds_miss_mode, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_INTERFACE_CFG_ID_DS_MISS_MODE, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "mtu");
    if (cli_parm != NULL)
    {
        uint16_t val;
        val = cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, interface, mtu, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, interface, mtu, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_INTERFACE_CFG_ID_MTU, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "flow_control");
    if (cli_parm != NULL)
    {
        bcmbal_control val;
        val = (bcmbal_control) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, interface, flow_control, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, interface, flow_control, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_INTERFACE_CFG_ID_FLOW_CONTROL, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "ds_tm");
    if (cli_parm != NULL)
    {
        bcmbal_tm_sched_id val;
        val = (bcmbal_tm_sched_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, interface, ds_tm, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, interface, ds_tm, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_INTERFACE_CFG_ID_DS_TM, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "us_tm");
    if (cli_parm != NULL)
    {
        bcmbal_tm_sched_id val;
        val = (bcmbal_tm_sched_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, interface, us_tm, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, interface, us_tm, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_INTERFACE_CFG_ID_US_TM, &val);
        bcmcli_log(");\n");
    }

    /* call API */
    err = bcmbal_cfg_set(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_set(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_interface_cfg_clear(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_interface_cfg cfg;       /**< declare main API struct */
    bcmbal_interface_key key = { }; /**< declare key */
    bcmcli_log("bcmbal_interface_cfg cfg;\n");
    bcmcli_log("bcmbal_interface_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_clear");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "intf_id");
    if (cli_parm != NULL)
    {
        key.intf_id = cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_INTERFACE_KEY_ID_INTF_ID, &key.intf_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "intf_type");
    if (cli_parm != NULL)
    {
        key.intf_type = (bcmbal_intf_type) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_type is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_type = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_INTERFACE_KEY_ID_INTF_TYPE, &key.intf_type);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, interface, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, interface, key);\n");

    /* call API */
    err = bcmbal_cfg_clear(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_clear(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_interface_stat_get(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_interface_stat stat;     /**< declare main API struct */
    bcmbal_interface_key key = { }; /**< declare key */
    bcmcli_log("bcmbal_interface_stat stat;\n");
    bcmcli_log("bcmbal_interface_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_stat_get");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "intf_id");
    if (cli_parm != NULL)
    {
        key.intf_id = cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_INTERFACE_KEY_ID_INTF_ID, &key.intf_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "intf_type");
    if (cli_parm != NULL)
    {
        key.intf_type = (bcmbal_intf_type) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_type is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_type = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_INTERFACE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_INTERFACE_KEY_ID_INTF_TYPE, &key.intf_type);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_STAT_INIT(&stat, interface, key);
    bcmcli_log("BCMBAL_STAT_INIT(&stat, interface, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "rx_packets");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, interface, rx_packets);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, interface, rx_packets);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "rx_bytes");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, interface, rx_bytes);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, interface, rx_bytes);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "tx_packets");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, interface, tx_packets);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, interface, tx_packets);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "tx_bytes");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, interface, tx_bytes);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, interface, tx_bytes);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_STAT_PROP_IS_SET(&stat, interface, rx_packets) && !BCMBAL_STAT_PROP_IS_SET(&stat, interface, rx_bytes) && !BCMBAL_STAT_PROP_IS_SET(&stat, interface, tx_packets) && !BCMBAL_STAT_PROP_IS_SET(&stat, interface, tx_bytes))
    {
        BCMBAL_STAT_PROP_GET(&stat, interface, all_properties);
        bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, interface, all_properties);\n");
    }

    /* call API */
    err = bcmbal_stat_get(&stat.hdr);
    bcmcli_log("bcmbal_stat_get(&stat.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &stat.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_packet_cfg_get(bcmcli_session *session, bcmbal_apicli_byte_pool *byte_pool)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_packet_cfg cfg;          /**< declare main API struct */
    bcmbal_packet_key key = { };    /**< declare key */
    uint8_t *list_mem;              /**< declare memory buffer for variable-sized lists */
    bcmcli_log("bcmbal_packet_cfg cfg;\n");
    bcmcli_log("bcmbal_packet_key key = {  };\n");
    bcmcli_log("uint8_t* list_mem;\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_get");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "reserved");
    if (cli_parm != NULL)
    {
        key.reserved = cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "reserved is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.reserved = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_PACKET_KEY_ID_RESERVED, &key.reserved);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_parm_by_prefix(session, "packet_send_dest.");
    if (cli_parm != NULL)
    {
        cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.type");
        if (cli_parm != NULL)
        {
            key.packet_send_dest.type = (bcmbal_dest_type) cli_parm->value.enum_val;
        }
        else
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.type is not set\n");
            return BCM_ERR_PARM;
        }

        switch (key.packet_send_dest.type)
        {
            case BCMBAL_DEST_TYPE_NNI:
                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.int_id");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.nni.int_id = (bcmbal_intf_id) cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.int_id is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            case BCMBAL_DEST_TYPE_SUB_TERM:
                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.sub_term_id");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.sub_term.sub_term_id = (bcmbal_sub_id) cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.sub_term_id is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.sub_term_uni");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.sub_term.sub_term_uni = cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.sub_term_uni is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.int_id");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.sub_term.int_id = cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.int_id is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            case BCMBAL_DEST_TYPE_HOST:
                break;
            default:
                bcmbal_apicli_print_complete(session, BCM_ERR_RANGE, "\n");
                return BCM_ERR_RANGE;
        }
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.packet_send_dest = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_PACKET_KEY_ID_PACKET_SEND_DEST, &key.packet_send_dest);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, packet, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, packet, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "flow_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, packet, flow_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, packet, flow_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "flow_type");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, packet, flow_type);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, packet, flow_type);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "intf_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, packet, intf_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, packet, intf_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "intf_type");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, packet, intf_type);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, packet, intf_type);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "svc_port");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, packet, svc_port);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, packet, svc_port);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "flow_cookie");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, packet, flow_cookie);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, packet, flow_cookie);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "pkt");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, packet, pkt);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, packet, pkt);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_CFG_PROP_IS_SET(&cfg, packet, flow_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, packet, flow_type) && !BCMBAL_CFG_PROP_IS_SET(&cfg, packet, intf_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, packet, intf_type) && !BCMBAL_CFG_PROP_IS_SET(&cfg, packet, svc_port) && !BCMBAL_CFG_PROP_IS_SET(&cfg, packet, flow_cookie) && !BCMBAL_CFG_PROP_IS_SET(&cfg, packet, pkt))
    {
        BCMBAL_CFG_PROP_GET(&cfg, packet, all_properties);
        bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, packet, all_properties);\n");
    }

    /* set memory to use for variable-sized lists */
    list_mem = bcmbal_apicli_byte_pool_calloc(byte_pool, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);
    if (list_mem == NULL)
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_NOMEM, "\n");
        return BCM_ERR_NOMEM;
    }

    bcmcli_log("list_mem = bcmos_calloc(BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);\n");
    BCMBAL_CFG_LIST_BUF_SET(&cfg, packet, list_mem, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);
    bcmcli_log("BCMBAL_CFG_LIST_BUF_SET(&cfg, packet, list_mem, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);\n");

    /* call API */
    err = bcmbal_cfg_get(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_get(&cfg.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &cfg.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_packet_cfg_set(bcmcli_session *session, bcmbal_apicli_byte_pool *byte_pool)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_packet_cfg cfg;          /**< declare main API struct */
    bcmbal_packet_key key = { };    /**< declare key */
    bcmcli_log("bcmbal_packet_cfg cfg;\n");
    bcmcli_log("bcmbal_packet_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_set");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "reserved");
    if (cli_parm != NULL)
    {
        key.reserved = cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "reserved is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.reserved = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_PACKET_KEY_ID_RESERVED, &key.reserved);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_parm_by_prefix(session, "packet_send_dest.");
    if (cli_parm != NULL)
    {
        cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.type");
        if (cli_parm != NULL)
        {
            key.packet_send_dest.type = (bcmbal_dest_type) cli_parm->value.enum_val;
        }
        else
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.type is not set\n");
            return BCM_ERR_PARM;
        }

        switch (key.packet_send_dest.type)
        {
            case BCMBAL_DEST_TYPE_NNI:
                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.int_id");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.nni.int_id = (bcmbal_intf_id) cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.int_id is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            case BCMBAL_DEST_TYPE_SUB_TERM:
                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.sub_term_id");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.sub_term.sub_term_id = (bcmbal_sub_id) cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.sub_term_id is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.sub_term_uni");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.sub_term.sub_term_uni = cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.sub_term_uni is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.int_id");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.sub_term.int_id = cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.int_id is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            case BCMBAL_DEST_TYPE_HOST:
                break;
            default:
                bcmbal_apicli_print_complete(session, BCM_ERR_RANGE, "\n");
                return BCM_ERR_RANGE;
        }
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.packet_send_dest = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_PACKET_KEY_ID_PACKET_SEND_DEST, &key.packet_send_dest);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, packet, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, packet, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "flow_id");
    if (cli_parm != NULL)
    {
        bcmbal_flow_id val;
        val = (bcmbal_flow_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, packet, flow_id, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, packet, flow_id, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_PACKET_CFG_ID_FLOW_ID, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "flow_type");
    if (cli_parm != NULL)
    {
        bcmbal_flow_type val;
        val = (bcmbal_flow_type) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, packet, flow_type, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, packet, flow_type, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_PACKET_CFG_ID_FLOW_TYPE, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "intf_id");
    if (cli_parm != NULL)
    {
        bcmbal_intf_id val;
        val = (bcmbal_intf_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, packet, intf_id, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, packet, intf_id, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_PACKET_CFG_ID_INTF_ID, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "intf_type");
    if (cli_parm != NULL)
    {
        bcmbal_intf_type val;
        val = (bcmbal_intf_type) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, packet, intf_type, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, packet, intf_type, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_PACKET_CFG_ID_INTF_TYPE, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "svc_port");
    if (cli_parm != NULL)
    {
        bcmbal_service_port_id val;
        val = (bcmbal_service_port_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, packet, svc_port, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, packet, svc_port, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_PACKET_CFG_ID_SVC_PORT, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "flow_cookie");
    if (cli_parm != NULL)
    {
        bcmbal_cookie val;
        val = (bcmbal_cookie) cli_parm->value.unumber64;
        BCMBAL_CFG_PROP_SET(&cfg, packet, flow_cookie, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, packet, flow_cookie, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_PACKET_CFG_ID_FLOW_COOKIE, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "pkt");
    if (cli_parm != NULL)
    {
        bcmbal_u8_list_u32 val = { };
        val.len = bcmbal_buf_get_used(&cli_parm->value.buffer);
        val.val = bcmbal_apicli_byte_pool_calloc(byte_pool, val.len);
        if (val.val == NULL)
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_NOMEM, "\n");
            return BCM_ERR_NOMEM;
        }

        bcmbal_buf_set_pos(&cli_parm->value.buffer, 0);
        bcmbal_buf_read(&cli_parm->value.buffer, val.val, val.len);
        BCMBAL_CFG_PROP_SET(&cfg, packet, pkt, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_u8_list_u32 val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_PACKET_CFG_ID_PKT, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, packet, pkt, val);\n");
        bcmcli_log("}\n");
    }

    /* call API */
    err = bcmbal_cfg_set(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_set(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_packet_cfg_clear(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_packet_cfg cfg;          /**< declare main API struct */
    bcmbal_packet_key key = { };    /**< declare key */
    bcmcli_log("bcmbal_packet_cfg cfg;\n");
    bcmcli_log("bcmbal_packet_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_clear");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "reserved");
    if (cli_parm != NULL)
    {
        key.reserved = cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "reserved is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.reserved = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_PACKET_KEY_ID_RESERVED, &key.reserved);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_parm_by_prefix(session, "packet_send_dest.");
    if (cli_parm != NULL)
    {
        cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.type");
        if (cli_parm != NULL)
        {
            key.packet_send_dest.type = (bcmbal_dest_type) cli_parm->value.enum_val;
        }
        else
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.type is not set\n");
            return BCM_ERR_PARM;
        }

        switch (key.packet_send_dest.type)
        {
            case BCMBAL_DEST_TYPE_NNI:
                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.int_id");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.nni.int_id = (bcmbal_intf_id) cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.int_id is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            case BCMBAL_DEST_TYPE_SUB_TERM:
                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.sub_term_id");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.sub_term.sub_term_id = (bcmbal_sub_id) cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.sub_term_id is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.sub_term_uni");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.sub_term.sub_term_uni = cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.sub_term_uni is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_named_parm(session, "packet_send_dest.int_id");
                if (cli_parm != NULL)
                {
                    key.packet_send_dest.u.sub_term.int_id = cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest.int_id is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            case BCMBAL_DEST_TYPE_HOST:
                break;
            default:
                bcmbal_apicli_print_complete(session, BCM_ERR_RANGE, "\n");
                return BCM_ERR_RANGE;
        }
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "packet_send_dest is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.packet_send_dest = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_PACKET, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_PACKET_KEY_ID_PACKET_SEND_DEST, &key.packet_send_dest);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, packet, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, packet, key);\n");

    /* call API */
    err = bcmbal_cfg_clear(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_clear(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_subscriber_terminal_cfg_get(bcmcli_session *session, bcmbal_apicli_byte_pool *byte_pool)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_subscriber_terminal_cfg cfg;         /**< declare main API struct */
    bcmbal_subscriber_terminal_key key = { };   /**< declare key */
    uint8_t *list_mem;  /**< declare memory buffer for variable-sized lists */
    bcmcli_log("bcmbal_subscriber_terminal_cfg cfg;\n");
    bcmcli_log("bcmbal_subscriber_terminal_key key = {  };\n");
    bcmcli_log("uint8_t* list_mem;\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_get");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "sub_term_id");
    if (cli_parm != NULL)
    {
        key.sub_term_id = (bcmbal_sub_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sub_term_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sub_term_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_SUBSCRIBER_TERMINAL_KEY_ID_SUB_TERM_ID, &key.sub_term_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "intf_id");
    if (cli_parm != NULL)
    {
        key.intf_id = (bcmbal_intf_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_SUBSCRIBER_TERMINAL_KEY_ID_INTF_ID, &key.intf_id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, subscriber_terminal, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, subscriber_terminal, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "admin_state");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, admin_state);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, admin_state);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "oper_status");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, oper_status);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, oper_status);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "serial_number");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, serial_number);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, serial_number);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "password");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, password);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, password);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "registration_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, registration_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, registration_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "svc_port_id");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, svc_port_id);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, svc_port_id);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "mac_address");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, mac_address);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, mac_address);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "ds_tm");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, ds_tm);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, ds_tm);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "us_tm");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, us_tm);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, us_tm);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "svc_port_id_list");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, svc_port_id_list);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, svc_port_id_list);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "agg_port_id_list");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, agg_port_id_list);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, agg_port_id_list);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_CFG_PROP_IS_SET(&cfg, subscriber_terminal, admin_state) && !BCMBAL_CFG_PROP_IS_SET(&cfg, subscriber_terminal, oper_status) && !BCMBAL_CFG_PROP_IS_SET(&cfg, subscriber_terminal, serial_number) && !BCMBAL_CFG_PROP_IS_SET(&cfg, subscriber_terminal, password) && !BCMBAL_CFG_PROP_IS_SET(&cfg, subscriber_terminal, registration_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, subscriber_terminal, svc_port_id) && !BCMBAL_CFG_PROP_IS_SET(&cfg, subscriber_terminal, mac_address) && !BCMBAL_CFG_PROP_IS_SET(&cfg, subscriber_terminal, ds_tm) && !BCMBAL_CFG_PROP_IS_SET(&cfg, subscriber_terminal, us_tm) && !BCMBAL_CFG_PROP_IS_SET(&cfg, subscriber_terminal, svc_port_id_list) && !BCMBAL_CFG_PROP_IS_SET(&cfg, subscriber_terminal, agg_port_id_list))
    {
        BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, all_properties);
        bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, subscriber_terminal, all_properties);\n");
    }

    /* set memory to use for variable-sized lists */
    list_mem = bcmbal_apicli_byte_pool_calloc(byte_pool, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);
    if (list_mem == NULL)
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_NOMEM, "\n");
        return BCM_ERR_NOMEM;
    }

    bcmcli_log("list_mem = bcmos_calloc(BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);\n");
    BCMBAL_CFG_LIST_BUF_SET(&cfg, subscriber_terminal, list_mem, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);
    bcmcli_log("BCMBAL_CFG_LIST_BUF_SET(&cfg, subscriber_terminal, list_mem, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);\n");

    /* call API */
    err = bcmbal_cfg_get(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_get(&cfg.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &cfg.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_subscriber_terminal_cfg_set(bcmcli_session *session, bcmbal_apicli_byte_pool *byte_pool)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_subscriber_terminal_cfg cfg; /**< declare main API struct */
    bcmbal_subscriber_terminal_key key = { };   /**< declare key */
    bcmcli_log("bcmbal_subscriber_terminal_cfg cfg;\n");
    bcmcli_log("bcmbal_subscriber_terminal_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_set");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "sub_term_id");
    if (cli_parm != NULL)
    {
        key.sub_term_id = (bcmbal_sub_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sub_term_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sub_term_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_SUBSCRIBER_TERMINAL_KEY_ID_SUB_TERM_ID, &key.sub_term_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "intf_id");
    if (cli_parm != NULL)
    {
        key.intf_id = (bcmbal_intf_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_SUBSCRIBER_TERMINAL_KEY_ID_INTF_ID, &key.intf_id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, subscriber_terminal, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, subscriber_terminal, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "admin_state");
    if (cli_parm != NULL)
    {
        bcmbal_state val;
        val = (bcmbal_state) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, admin_state, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, admin_state, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_SUBSCRIBER_TERMINAL_CFG_ID_ADMIN_STATE, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "serial_number.");
    if (cli_parm != NULL)
    {
        bcmbal_serial_number val = { };
        cli_parm = bcmcli_find_named_parm(session, "serial_number.vendor_id");
        if (cli_parm != NULL)
        {
            if (bcmbal_buf_get_used(&cli_parm->value.buffer) != 4)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "buffer serial_number.vendor_id must have 4 bytes\n");
                return BCM_ERR_PARM;
            }

            bcmbal_buf_set_pos(&cli_parm->value.buffer, 0);
            bcmbal_buf_read(&cli_parm->value.buffer, val.vendor_id, 4);
        }
        else
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "serial_number.vendor_id is not set\n");
            return BCM_ERR_PARM;
        }

        cli_parm = bcmcli_find_named_parm(session, "serial_number.vendor_specific");
        if (cli_parm != NULL)
        {
            if (bcmbal_buf_get_used(&cli_parm->value.buffer) != 4)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "buffer serial_number.vendor_specific must have 4 bytes\n");
                return BCM_ERR_PARM;
            }

            bcmbal_buf_set_pos(&cli_parm->value.buffer, 0);
            bcmbal_buf_read(&cli_parm->value.buffer, val.vendor_specific, 4);
        }
        else
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "serial_number.vendor_specific is not set\n");
            return BCM_ERR_PARM;
        }

        BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, serial_number, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_serial_number val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_SUBSCRIBER_TERMINAL_CFG_ID_SERIAL_NUMBER, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, serial_number, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "password.");
    if (cli_parm != NULL)
    {
        bcmbal_password val = { };
        cli_parm = bcmcli_find_named_parm(session, "password.arr");
        if (cli_parm != NULL)
        {
            if (bcmbal_buf_get_used(&cli_parm->value.buffer) != 10)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "buffer password.arr must have 10 bytes\n");
                return BCM_ERR_PARM;
            }

            bcmbal_buf_set_pos(&cli_parm->value.buffer, 0);
            bcmbal_buf_read(&cli_parm->value.buffer, val.arr, 10);
        }
        else
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "password.arr is not set\n");
            return BCM_ERR_PARM;
        }

        BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, password, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_password val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_SUBSCRIBER_TERMINAL_CFG_ID_PASSWORD, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, password, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "registration_id.");
    if (cli_parm != NULL)
    {
        bcmbal_registration_id val = { };
        cli_parm = bcmcli_find_named_parm(session, "registration_id.arr");
        if (cli_parm != NULL)
        {
            if (bcmbal_buf_get_used(&cli_parm->value.buffer) != 36)
            {
                bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "buffer registration_id.arr must have 36 bytes\n");
                return BCM_ERR_PARM;
            }

            bcmbal_buf_set_pos(&cli_parm->value.buffer, 0);
            bcmbal_buf_read(&cli_parm->value.buffer, val.arr, 36);
        }
        else
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "registration_id.arr is not set\n");
            return BCM_ERR_PARM;
        }

        BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, registration_id, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_registration_id val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_SUBSCRIBER_TERMINAL_CFG_ID_REGISTRATION_ID, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, registration_id, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "mac_address");
    if (cli_parm != NULL)
    {
        bcmos_mac_address val;
        val = cli_parm->value.mac;
        BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, mac_address, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, mac_address, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_SUBSCRIBER_TERMINAL_CFG_ID_MAC_ADDRESS, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "ds_tm");
    if (cli_parm != NULL)
    {
        bcmbal_tm_sched_id val;
        val = (bcmbal_tm_sched_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, ds_tm, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, ds_tm, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_SUBSCRIBER_TERMINAL_CFG_ID_DS_TM, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "us_tm");
    if (cli_parm != NULL)
    {
        bcmbal_tm_sched_id val;
        val = (bcmbal_tm_sched_id) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, us_tm, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, subscriber_terminal, us_tm, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_SUBSCRIBER_TERMINAL_CFG_ID_US_TM, &val);
        bcmcli_log(");\n");
    }

    /* call API */
    err = bcmbal_cfg_set(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_set(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_subscriber_terminal_cfg_clear(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_subscriber_terminal_cfg cfg;         /**< declare main API struct */
    bcmbal_subscriber_terminal_key key = { };   /**< declare key */
    bcmcli_log("bcmbal_subscriber_terminal_cfg cfg;\n");
    bcmcli_log("bcmbal_subscriber_terminal_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_clear");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "sub_term_id");
    if (cli_parm != NULL)
    {
        key.sub_term_id = (bcmbal_sub_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sub_term_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sub_term_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_SUBSCRIBER_TERMINAL_KEY_ID_SUB_TERM_ID, &key.sub_term_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "intf_id");
    if (cli_parm != NULL)
    {
        key.intf_id = (bcmbal_intf_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_SUBSCRIBER_TERMINAL_KEY_ID_INTF_ID, &key.intf_id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, subscriber_terminal, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, subscriber_terminal, key);\n");

    /* call API */
    err = bcmbal_cfg_clear(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_clear(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_subscriber_terminal_stat_get(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_subscriber_terminal_stat stat;       /**< declare main API struct */
    bcmbal_subscriber_terminal_key key = { };   /**< declare key */
    bcmcli_log("bcmbal_subscriber_terminal_stat stat;\n");
    bcmcli_log("bcmbal_subscriber_terminal_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_stat_get");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "sub_term_id");
    if (cli_parm != NULL)
    {
        key.sub_term_id = (bcmbal_sub_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sub_term_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sub_term_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_SUBSCRIBER_TERMINAL_KEY_ID_SUB_TERM_ID, &key.sub_term_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "intf_id");
    if (cli_parm != NULL)
    {
        key.intf_id = (bcmbal_intf_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "intf_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.intf_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_SUBSCRIBER_TERMINAL_KEY_ID_INTF_ID, &key.intf_id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_STAT_INIT(&stat, subscriber_terminal, key);
    bcmcli_log("BCMBAL_STAT_INIT(&stat, subscriber_terminal, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "rx_packets");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, subscriber_terminal, rx_packets);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, subscriber_terminal, rx_packets);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "rx_bytes");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, subscriber_terminal, rx_bytes);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, subscriber_terminal, rx_bytes);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "tx_packets");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, subscriber_terminal, tx_packets);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, subscriber_terminal, tx_packets);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "tx_bytes");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, subscriber_terminal, tx_bytes);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, subscriber_terminal, tx_bytes);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_STAT_PROP_IS_SET(&stat, subscriber_terminal, rx_packets) && !BCMBAL_STAT_PROP_IS_SET(&stat, subscriber_terminal, rx_bytes) && !BCMBAL_STAT_PROP_IS_SET(&stat, subscriber_terminal, tx_packets) && !BCMBAL_STAT_PROP_IS_SET(&stat, subscriber_terminal, tx_bytes))
    {
        BCMBAL_STAT_PROP_GET(&stat, subscriber_terminal, all_properties);
        bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, subscriber_terminal, all_properties);\n");
    }

    /* call API */
    err = bcmbal_stat_get(&stat.hdr);
    bcmcli_log("bcmbal_stat_get(&stat.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &stat.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_tm_queue_cfg_get(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_tm_queue_cfg cfg;        /**< declare main API struct */
    bcmbal_tm_queue_key key = { };  /**< declare key */
    bcmcli_log("bcmbal_tm_queue_cfg cfg;\n");
    bcmcli_log("bcmbal_tm_queue_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_get");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "sched_id");
    if (cli_parm != NULL)
    {
        key.sched_id = (bcmbal_tm_sched_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sched_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sched_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_SCHED_ID, &key.sched_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "sched_dir");
    if (cli_parm != NULL)
    {
        key.sched_dir = (bcmbal_tm_sched_dir) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sched_dir is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sched_dir = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_SCHED_DIR, &key.sched_dir);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "id");
    if (cli_parm != NULL)
    {
        key.id = (bcmbal_tm_queue_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_ID, &key.id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, tm_queue, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, tm_queue, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "priority");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_queue, priority);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_queue, priority);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "weight");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_queue, weight);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_queue, weight);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "rate");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_queue, rate);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_queue, rate);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "bac");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_queue, bac);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_queue, bac);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "creation_mode");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_queue, creation_mode);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_queue, creation_mode);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "ref_count");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_queue, ref_count);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_queue, ref_count);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_CFG_PROP_IS_SET(&cfg, tm_queue, priority) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_queue, weight) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_queue, rate) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_queue, bac) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_queue, creation_mode) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_queue, ref_count))
    {
        BCMBAL_CFG_PROP_GET(&cfg, tm_queue, all_properties);
        bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_queue, all_properties);\n");
    }

    /* call API */
    err = bcmbal_cfg_get(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_get(&cfg.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &cfg.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_tm_queue_cfg_set(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_tm_queue_cfg cfg;        /**< declare main API struct */
    bcmbal_tm_queue_key key = { };  /**< declare key */
    bcmcli_log("bcmbal_tm_queue_cfg cfg;\n");
    bcmcli_log("bcmbal_tm_queue_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_set");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "sched_id");
    if (cli_parm != NULL)
    {
        key.sched_id = (bcmbal_tm_sched_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sched_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sched_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_SCHED_ID, &key.sched_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "sched_dir");
    if (cli_parm != NULL)
    {
        key.sched_dir = (bcmbal_tm_sched_dir) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sched_dir is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sched_dir = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_SCHED_DIR, &key.sched_dir);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "id");
    if (cli_parm != NULL)
    {
        key.id = (bcmbal_tm_queue_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_ID, &key.id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, tm_queue, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, tm_queue, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "priority");
    if (cli_parm != NULL)
    {
        bcmbal_tm_priority val;
        val = (bcmbal_tm_priority) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, tm_queue, priority, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, tm_queue, priority, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_TM_QUEUE_CFG_ID_PRIORITY, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "weight");
    if (cli_parm != NULL)
    {
        bcmbal_tm_weight val;
        val = (bcmbal_tm_weight) cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, tm_queue, weight, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, tm_queue, weight, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_TM_QUEUE_CFG_ID_WEIGHT, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "rate.");
    if (cli_parm != NULL)
    {
        bcmbal_tm_shaping val = { };
        cli_parm = bcmcli_find_named_parm(session, "rate.sbr");
        if (cli_parm != NULL)
        {
            val.sbr = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_SHAPING_ID_SBR;
        }

        cli_parm = bcmcli_find_named_parm(session, "rate.pbr");
        if (cli_parm != NULL)
        {
            val.pbr = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_SHAPING_ID_PBR;
        }

        cli_parm = bcmcli_find_named_parm(session, "rate.burst");
        if (cli_parm != NULL)
        {
            val.burst = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_SHAPING_ID_BURST;
        }

        BCMBAL_CFG_PROP_SET(&cfg, tm_queue, rate, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_tm_shaping val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_TM_QUEUE_CFG_ID_RATE, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, tm_queue, rate, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "bac.");
    if (cli_parm != NULL)
    {
        bcmbal_tm_bac val = { };
        cli_parm = bcmcli_find_named_parm(session, "bac.type");
        if (cli_parm != NULL)
        {
            val.type = (bcmbal_tm_bac_type) cli_parm->value.enum_val;
        }
        else
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.type is not set\n");
            return BCM_ERR_PARM;
        }

        switch (val.type)
        {
            case BCMBAL_TM_BAC_TYPE_TAILDROP:
                cli_parm = bcmcli_find_named_parm(session, "bac.max_size");
                if (cli_parm != NULL)
                {
                    val.u.taildrop.max_size = cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.max_size is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            case BCMBAL_TM_BAC_TYPE_RED:
                cli_parm = bcmcli_find_parm_by_prefix(session, "bac.red.");
                if (cli_parm != NULL)
                {
                    cli_parm = bcmcli_find_named_parm(session, "bac.red.min_threshold");
                    if (cli_parm != NULL)
                    {
                        val.u.red.red.min_threshold = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.red.min_threshold is not set\n");
                        return BCM_ERR_PARM;
                    }

                    cli_parm = bcmcli_find_named_parm(session, "bac.red.max_threshold");
                    if (cli_parm != NULL)
                    {
                        val.u.red.red.max_threshold = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.red.max_threshold is not set\n");
                        return BCM_ERR_PARM;
                    }

                    cli_parm = bcmcli_find_named_parm(session, "bac.red.max_probability");
                    if (cli_parm != NULL)
                    {
                        val.u.red.red.max_probability = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.red.max_probability is not set\n");
                        return BCM_ERR_PARM;
                    }
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.red is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            case BCMBAL_TM_BAC_TYPE_WRED:
                cli_parm = bcmcli_find_parm_by_prefix(session, "bac.green.");
                if (cli_parm != NULL)
                {
                    cli_parm = bcmcli_find_named_parm(session, "bac.green.min_threshold");
                    if (cli_parm != NULL)
                    {
                        val.u.wred.green.min_threshold = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.green.min_threshold is not set\n");
                        return BCM_ERR_PARM;
                    }

                    cli_parm = bcmcli_find_named_parm(session, "bac.green.max_threshold");
                    if (cli_parm != NULL)
                    {
                        val.u.wred.green.max_threshold = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.green.max_threshold is not set\n");
                        return BCM_ERR_PARM;
                    }

                    cli_parm = bcmcli_find_named_parm(session, "bac.green.max_probability");
                    if (cli_parm != NULL)
                    {
                        val.u.wred.green.max_probability = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.green.max_probability is not set\n");
                        return BCM_ERR_PARM;
                    }
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.green is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_parm_by_prefix(session, "bac.yellow.");
                if (cli_parm != NULL)
                {
                    cli_parm = bcmcli_find_named_parm(session, "bac.yellow.min_threshold");
                    if (cli_parm != NULL)
                    {
                        val.u.wred.yellow.min_threshold = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.yellow.min_threshold is not set\n");
                        return BCM_ERR_PARM;
                    }

                    cli_parm = bcmcli_find_named_parm(session, "bac.yellow.max_threshold");
                    if (cli_parm != NULL)
                    {
                        val.u.wred.yellow.max_threshold = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.yellow.max_threshold is not set\n");
                        return BCM_ERR_PARM;
                    }

                    cli_parm = bcmcli_find_named_parm(session, "bac.yellow.max_probability");
                    if (cli_parm != NULL)
                    {
                        val.u.wred.yellow.max_probability = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.yellow.max_probability is not set\n");
                        return BCM_ERR_PARM;
                    }
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.yellow is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_parm_by_prefix(session, "bac.red.");
                if (cli_parm != NULL)
                {
                    cli_parm = bcmcli_find_named_parm(session, "bac.red.min_threshold");
                    if (cli_parm != NULL)
                    {
                        val.u.wred.red.min_threshold = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.red.min_threshold is not set\n");
                        return BCM_ERR_PARM;
                    }

                    cli_parm = bcmcli_find_named_parm(session, "bac.red.max_threshold");
                    if (cli_parm != NULL)
                    {
                        val.u.wred.red.max_threshold = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.red.max_threshold is not set\n");
                        return BCM_ERR_PARM;
                    }

                    cli_parm = bcmcli_find_named_parm(session, "bac.red.max_probability");
                    if (cli_parm != NULL)
                    {
                        val.u.wred.red.max_probability = (bcmbal_percent) cli_parm->value.unumber;
                    }
                    else
                    {
                        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.red.max_probability is not set\n");
                        return BCM_ERR_PARM;
                    }
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "bac.red is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            default:
                bcmbal_apicli_print_complete(session, BCM_ERR_RANGE, "\n");
                return BCM_ERR_RANGE;
        }

        BCMBAL_CFG_PROP_SET(&cfg, tm_queue, bac, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_tm_bac val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_TM_QUEUE_CFG_ID_BAC, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, tm_queue, bac, val);\n");
        bcmcli_log("}\n");
    }

    /* call API */
    err = bcmbal_cfg_set(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_set(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_tm_queue_cfg_clear(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_tm_queue_cfg cfg;        /**< declare main API struct */
    bcmbal_tm_queue_key key = { };  /**< declare key */
    bcmcli_log("bcmbal_tm_queue_cfg cfg;\n");
    bcmcli_log("bcmbal_tm_queue_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_clear");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "sched_id");
    if (cli_parm != NULL)
    {
        key.sched_id = (bcmbal_tm_sched_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sched_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sched_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_SCHED_ID, &key.sched_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "sched_dir");
    if (cli_parm != NULL)
    {
        key.sched_dir = (bcmbal_tm_sched_dir) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sched_dir is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sched_dir = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_SCHED_DIR, &key.sched_dir);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "id");
    if (cli_parm != NULL)
    {
        key.id = (bcmbal_tm_queue_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_ID, &key.id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, tm_queue, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, tm_queue, key);\n");

    /* call API */
    err = bcmbal_cfg_clear(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_clear(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_tm_queue_stat_get(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_tm_queue_stat stat;      /**< declare main API struct */
    bcmbal_tm_queue_key key = { };  /**< declare key */
    bcmcli_log("bcmbal_tm_queue_stat stat;\n");
    bcmcli_log("bcmbal_tm_queue_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_stat_get");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "sched_id");
    if (cli_parm != NULL)
    {
        key.sched_id = (bcmbal_tm_sched_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sched_id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sched_id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_SCHED_ID, &key.sched_id);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "sched_dir");
    if (cli_parm != NULL)
    {
        key.sched_dir = (bcmbal_tm_sched_dir) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "sched_dir is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.sched_dir = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_SCHED_DIR, &key.sched_dir);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "id");
    if (cli_parm != NULL)
    {
        key.id = (bcmbal_tm_queue_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_QUEUE, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_QUEUE_KEY_ID_ID, &key.id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_STAT_INIT(&stat, tm_queue, key);
    bcmcli_log("BCMBAL_STAT_INIT(&stat, tm_queue, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "packets_ok");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, tm_queue, packets_ok);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, tm_queue, packets_ok);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "bytes_ok");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, tm_queue, bytes_ok);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, tm_queue, bytes_ok);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "packets_discarded");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, tm_queue, packets_discarded);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, tm_queue, packets_discarded);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "bytes_discarded");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_STAT_PROP_GET(&stat, tm_queue, bytes_discarded);
            bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, tm_queue, bytes_discarded);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_STAT_PROP_IS_SET(&stat, tm_queue, packets_ok) && !BCMBAL_STAT_PROP_IS_SET(&stat, tm_queue, bytes_ok) && !BCMBAL_STAT_PROP_IS_SET(&stat, tm_queue, packets_discarded) && !BCMBAL_STAT_PROP_IS_SET(&stat, tm_queue, bytes_discarded))
    {
        BCMBAL_STAT_PROP_GET(&stat, tm_queue, all_properties);
        bcmcli_log("BCMBAL_STAT_PROP_GET(&stat, tm_queue, all_properties);\n");
    }

    /* call API */
    err = bcmbal_stat_get(&stat.hdr);
    bcmcli_log("bcmbal_stat_get(&stat.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &stat.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_tm_sched_cfg_get(bcmcli_session *session, bcmbal_apicli_byte_pool *byte_pool)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_tm_sched_cfg cfg;        /**< declare main API struct */
    bcmbal_tm_sched_key key = { };  /**< declare key */
    uint8_t *list_mem;              /**< declare memory buffer for variable-sized lists */
    bcmcli_log("bcmbal_tm_sched_cfg cfg;\n");
    bcmcli_log("bcmbal_tm_sched_key key = {  };\n");
    bcmcli_log("uint8_t* list_mem;\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_get");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "dir");
    if (cli_parm != NULL)
    {
        key.dir = (bcmbal_tm_sched_dir) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "dir is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.dir = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_SCHED_KEY_ID_DIR, &key.dir);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "id");
    if (cli_parm != NULL)
    {
        key.id = (bcmbal_tm_sched_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_SCHED_KEY_ID_ID, &key.id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, tm_sched, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, tm_sched, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_named_parm(session, "owner");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_sched, owner);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_sched, owner);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "sched_type");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_sched, sched_type);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_sched, sched_type);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "sched_parent");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_sched, sched_parent);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_sched, sched_parent);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "sched_child_type");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_sched, sched_child_type);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_sched, sched_child_type);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "rate");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_sched, rate);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_sched, rate);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "tcont_sla");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_sched, tcont_sla);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_sched, tcont_sla);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "creation_mode");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_sched, creation_mode);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_sched, creation_mode);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "queues");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_sched, queues);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_sched, queues);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "sub_scheds");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_sched, sub_scheds);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_sched, sub_scheds);\n");
        }
    }

    cli_parm = bcmcli_find_named_parm(session, "num_priorities");
    if (cli_parm != NULL)
    {
        if (cli_parm->value.number)
        {
            BCMBAL_CFG_PROP_GET(&cfg, tm_sched, num_priorities);
            bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_sched, num_priorities);\n");
        }
    }

    /* if no properties were requested, include everything */
    if (!BCMBAL_CFG_PROP_IS_SET(&cfg, tm_sched, owner) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_sched, sched_type) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_sched, sched_parent) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_sched, sched_child_type) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_sched, rate) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_sched, tcont_sla) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_sched, creation_mode) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_sched, queues) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_sched, sub_scheds) && !BCMBAL_CFG_PROP_IS_SET(&cfg, tm_sched, num_priorities))
    {
        BCMBAL_CFG_PROP_GET(&cfg, tm_sched, all_properties);
        bcmcli_log("BCMBAL_CFG_PROP_GET(&cfg, tm_sched, all_properties);\n");
    }

    /* set memory to use for variable-sized lists */
    list_mem = bcmbal_apicli_byte_pool_calloc(byte_pool, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);
    if (list_mem == NULL)
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_NOMEM, "\n");
        return BCM_ERR_NOMEM;
    }

    bcmcli_log("list_mem = bcmos_calloc(BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);\n");
    BCMBAL_CFG_LIST_BUF_SET(&cfg, tm_sched, list_mem, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);
    bcmcli_log("BCMBAL_CFG_LIST_BUF_SET(&cfg, tm_sched, list_mem, BCMBAL_APICLI_DYNAMIC_LIST_BUFFER_SIZE);\n");

    /* call API */
    err = bcmbal_cfg_get(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_get(&cfg.hdr);\n");
    if (err == BCM_ERR_OK)
    {
        /* print API contents to the CLI */
        bcmbal_apicli_print_data_start(session);
        err = bcmbal_apicli_msg_dump(session, &cfg.hdr.hdr);
    }

    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_tm_sched_cfg_set(bcmcli_session *session, bcmbal_apicli_byte_pool *byte_pool)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_tm_sched_cfg cfg;        /**< declare main API struct */
    bcmbal_tm_sched_key key = { };  /**< declare key */
    bcmcli_log("bcmbal_tm_sched_cfg cfg;\n");
    bcmcli_log("bcmbal_tm_sched_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_set");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "dir");
    if (cli_parm != NULL)
    {
        key.dir = (bcmbal_tm_sched_dir) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "dir is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.dir = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_SCHED_KEY_ID_DIR, &key.dir);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "id");
    if (cli_parm != NULL)
    {
        key.id = (bcmbal_tm_sched_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_SCHED_KEY_ID_ID, &key.id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, tm_sched, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, tm_sched, key);\n");

    /* decode API parameters from CLI */
    cli_parm = bcmcli_find_parm_by_prefix(session, "owner.");
    if (cli_parm != NULL)
    {
        bcmbal_tm_sched_owner val = { };
        cli_parm = bcmcli_find_named_parm(session, "owner.type");
        if (cli_parm != NULL)
        {
            val.type = (bcmbal_tm_sched_owner_type) cli_parm->value.enum_val;
        }
        else
        {
            bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "owner.type is not set\n");
            return BCM_ERR_PARM;
        }

        switch (val.type)
        {
            case BCMBAL_TM_SCHED_OWNER_TYPE_INTERFACE:
                cli_parm = bcmcli_find_named_parm(session, "owner.intf_type");
                if (cli_parm != NULL)
                {
                    val.u.interface.intf_type = (bcmbal_intf_type) cli_parm->value.enum_val;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "owner.intf_type is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_named_parm(session, "owner.intf_id");
                if (cli_parm != NULL)
                {
                    val.u.interface.intf_id = (bcmbal_intf_id) cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "owner.intf_id is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            case BCMBAL_TM_SCHED_OWNER_TYPE_SUB_TERM:
                cli_parm = bcmcli_find_named_parm(session, "owner.intf_id");
                if (cli_parm != NULL)
                {
                    val.u.sub_term.intf_id = (bcmbal_intf_id) cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "owner.intf_id is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_named_parm(session, "owner.sub_term_id");
                if (cli_parm != NULL)
                {
                    val.u.sub_term.sub_term_id = (bcmbal_sub_id) cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "owner.sub_term_id is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            case BCMBAL_TM_SCHED_OWNER_TYPE_AGG_PORT:
                cli_parm = bcmcli_find_named_parm(session, "owner.intf_id");
                if (cli_parm != NULL)
                {
                    val.u.agg_port.intf_id = cli_parm->value.unumber;
                    val.u.agg_port.presence_mask = val.u.agg_port.presence_mask | BCMBAL_TM_SCHED_OWNER_AGG_PORT_ID_INTF_ID;
                }

                cli_parm = bcmcli_find_named_parm(session, "owner.sub_term_id");
                if (cli_parm != NULL)
                {
                    val.u.agg_port.sub_term_id = (bcmbal_sub_id) cli_parm->value.unumber;
                    val.u.agg_port.presence_mask = val.u.agg_port.presence_mask | BCMBAL_TM_SCHED_OWNER_AGG_PORT_ID_SUB_TERM_ID;
                }

                cli_parm = bcmcli_find_named_parm(session, "owner.agg_port_id");
                if (cli_parm != NULL)
                {
                    val.u.agg_port.agg_port_id = (bcmbal_aggregation_port_id) cli_parm->value.unumber;
                    val.u.agg_port.presence_mask = val.u.agg_port.presence_mask | BCMBAL_TM_SCHED_OWNER_AGG_PORT_ID_AGG_PORT_ID;
                }
                break;
            case BCMBAL_TM_SCHED_OWNER_TYPE_UNI:
                cli_parm = bcmcli_find_named_parm(session, "owner.intf_id");
                if (cli_parm != NULL)
                {
                    val.u.uni.intf_id = cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "owner.intf_id is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_named_parm(session, "owner.sub_term_id");
                if (cli_parm != NULL)
                {
                    val.u.uni.sub_term_id = (bcmbal_sub_id) cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "owner.sub_term_id is not set\n");
                    return BCM_ERR_PARM;
                }

                cli_parm = bcmcli_find_named_parm(session, "owner.idx");
                if (cli_parm != NULL)
                {
                    val.u.uni.idx = cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "owner.idx is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            case BCMBAL_TM_SCHED_OWNER_TYPE_VIRTUAL:
                cli_parm = bcmcli_find_named_parm(session, "owner.idx");
                if (cli_parm != NULL)
                {
                    val.u.virtual.idx = cli_parm->value.unumber;
                }
                else
                {
                    bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "owner.idx is not set\n");
                    return BCM_ERR_PARM;
                }
                break;
            default:
                bcmbal_apicli_print_complete(session, BCM_ERR_RANGE, "\n");
                return BCM_ERR_RANGE;
        }

        BCMBAL_CFG_PROP_SET(&cfg, tm_sched, owner, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_tm_sched_owner val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_TM_SCHED_CFG_ID_OWNER, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, tm_sched, owner, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "sched_type");
    if (cli_parm != NULL)
    {
        bcmbal_tm_sched_type val;
        val = (bcmbal_tm_sched_type) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, tm_sched, sched_type, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, tm_sched, sched_type, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_TM_SCHED_CFG_ID_SCHED_TYPE, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "sched_parent.");
    if (cli_parm != NULL)
    {
        bcmbal_tm_sched_parent val = { };
        cli_parm = bcmcli_find_named_parm(session, "sched_parent.sched_id");
        if (cli_parm != NULL)
        {
            val.sched_id = (bcmbal_tm_sched_id) cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_SCHED_PARENT_ID_SCHED_ID;
        }

        cli_parm = bcmcli_find_named_parm(session, "sched_parent.priority");
        if (cli_parm != NULL)
        {
            val.priority = (bcmbal_tm_priority) cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_SCHED_PARENT_ID_PRIORITY;
        }

        cli_parm = bcmcli_find_named_parm(session, "sched_parent.weight");
        if (cli_parm != NULL)
        {
            val.weight = (bcmbal_tm_weight) cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_SCHED_PARENT_ID_WEIGHT;
        }

        BCMBAL_CFG_PROP_SET(&cfg, tm_sched, sched_parent, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_tm_sched_parent val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_TM_SCHED_CFG_ID_SCHED_PARENT, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, tm_sched, sched_parent, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "sched_child_type");
    if (cli_parm != NULL)
    {
        bcmbal_tm_sched_child_type val;
        val = (bcmbal_tm_sched_child_type) cli_parm->value.enum_val;
        BCMBAL_CFG_PROP_SET(&cfg, tm_sched, sched_child_type, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, tm_sched, sched_child_type, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_TM_SCHED_CFG_ID_SCHED_CHILD_TYPE, &val);
        bcmcli_log(");\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "rate.");
    if (cli_parm != NULL)
    {
        bcmbal_tm_shaping val = { };
        cli_parm = bcmcli_find_named_parm(session, "rate.sbr");
        if (cli_parm != NULL)
        {
            val.sbr = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_SHAPING_ID_SBR;
        }

        cli_parm = bcmcli_find_named_parm(session, "rate.pbr");
        if (cli_parm != NULL)
        {
            val.pbr = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_SHAPING_ID_PBR;
        }

        cli_parm = bcmcli_find_named_parm(session, "rate.burst");
        if (cli_parm != NULL)
        {
            val.burst = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_SHAPING_ID_BURST;
        }

        BCMBAL_CFG_PROP_SET(&cfg, tm_sched, rate, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_tm_shaping val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_TM_SCHED_CFG_ID_RATE, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, tm_sched, rate, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_parm_by_prefix(session, "tcont_sla.");
    if (cli_parm != NULL)
    {
        bcmbal_tm_tcont_sla val = { };
        cli_parm = bcmcli_find_named_parm(session, "tcont_sla.extra_bw_elig");
        if (cli_parm != NULL)
        {
            val.extra_bw_elig = (bcmbal_extra_bw_eligibility_type) cli_parm->value.enum_val;
            val.presence_mask = val.presence_mask | BCMBAL_TM_TCONT_SLA_ID_EXTRA_BW_ELIG;
        }

        cli_parm = bcmcli_find_named_parm(session, "tcont_sla.nrt_cbr");
        if (cli_parm != NULL)
        {
            val.nrt_cbr = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_TCONT_SLA_ID_NRT_CBR;
        }

        cli_parm = bcmcli_find_named_parm(session, "tcont_sla.rt_cbr");
        if (cli_parm != NULL)
        {
            val.rt_cbr = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_TCONT_SLA_ID_RT_CBR;
        }

        cli_parm = bcmcli_find_named_parm(session, "tcont_sla.rt_profile");
        if (cli_parm != NULL)
        {
            val.rt_profile = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_TCONT_SLA_ID_RT_PROFILE;
        }

        cli_parm = bcmcli_find_named_parm(session, "tcont_sla.nrt_profile");
        if (cli_parm != NULL)
        {
            val.nrt_profile = cli_parm->value.unumber;
            val.presence_mask = val.presence_mask | BCMBAL_TM_TCONT_SLA_ID_NRT_PROFILE;
        }

        BCMBAL_CFG_PROP_SET(&cfg, tm_sched, tcont_sla, val);
        bcmcli_log("{\n");
        bcmcli_log("bcmbal_tm_tcont_sla val = ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_TM_SCHED_CFG_ID_TCONT_SLA, &val);
        bcmcli_log(";\n");
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, tm_sched, tcont_sla, val);\n");
        bcmcli_log("}\n");
    }

    cli_parm = bcmcli_find_named_parm(session, "num_priorities");
    if (cli_parm != NULL)
    {
        uint8_t val;
        val = cli_parm->value.unumber;
        BCMBAL_CFG_PROP_SET(&cfg, tm_sched, num_priorities, val);
        bcmcli_log("BCMBAL_CFG_PROP_SET(&cfg, tm_sched, num_priorities, ");
        bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_CFG, 0, BCMBAL_TM_SCHED_CFG_ID_NUM_PRIORITIES, &val);
        bcmcli_log(");\n");
    }

    /* call API */
    err = bcmbal_cfg_set(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_set(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_cli_tm_sched_cfg_clear(bcmcli_session *session)
{
    bcmcli_cmd_parm *cli_parm;
    bcmos_errno err;
    bcmbal_tm_sched_cfg cfg;        /**< declare main API struct */
    bcmbal_tm_sched_key key = { };  /**< declare key */
    bcmcli_log("bcmbal_tm_sched_cfg cfg;\n");
    bcmcli_log("bcmbal_tm_sched_key key = {  };\n");
    bcmbal_apicli_print_start(session, "bcmbal_cfg_clear");

    /* build key from CLI parameters */
    cli_parm = bcmcli_find_named_parm(session, "dir");
    if (cli_parm != NULL)
    {
        key.dir = (bcmbal_tm_sched_dir) cli_parm->value.enum_val;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "dir is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.dir = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_SCHED_KEY_ID_DIR, &key.dir);
    bcmcli_log(";\n");
    cli_parm = bcmcli_find_named_parm(session, "id");
    if (cli_parm != NULL)
    {
        key.id = (bcmbal_tm_sched_id) cli_parm->value.unumber;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "id is not set\n");
        return BCM_ERR_PARM;
    }

    bcmcli_log("key.id = ");
    bcmbal_apicli_log_prop_val(BCMBAL_OBJ_ID_TM_SCHED, BCMBAL_MGT_GROUP_KEY, 0, BCMBAL_TM_SCHED_KEY_ID_ID, &key.id);
    bcmcli_log(";\n");

    /* init the API struct */
    BCMBAL_CFG_INIT(&cfg, tm_sched, key);
    bcmcli_log("BCMBAL_CFG_INIT(&cfg, tm_sched, key);\n");

    /* call API */
    err = bcmbal_cfg_clear(&cfg.hdr);
    bcmcli_log("bcmbal_cfg_clear(&cfg.hdr);\n");
    bcmbal_apicli_print_complete(session, err, NULL);
    return err;
}

/******************************************************************************/
static bcmos_errno bcmbal_apicli_root(bcmbal_mgt_group group_type, bcmbal_obj_msg_type msg_type, bcmcli_session *session, bcmbal_apicli_byte_pool *byte_pool)
{
    bcmcli_cmd_parm *cli_parm;
    bcmbal_obj_id obj_id;
    cli_parm = bcmcli_find_named_parm(session, "object");
    if (cli_parm != NULL)
    {
        obj_id = cli_parm->value.number;
    }
    else
    {
        bcmbal_apicli_print_complete(session, BCM_ERR_PARM, "object is not set\n");
        return BCM_ERR_PARM;
    }

    switch (obj_id)
    {
        case BCMBAL_OBJ_ID_ACCESS_TERMINAL:
            switch (group_type)
            {
                case BCMBAL_MGT_GROUP_CFG:
                    switch (msg_type)
                    {
                        case BCMBAL_OBJ_MSG_TYPE_GET:
                            return bcmbal_cli_access_terminal_cfg_get(session);
                        case BCMBAL_OBJ_MSG_TYPE_SET:
                            return bcmbal_cli_access_terminal_cfg_set(session);
                        case BCMBAL_OBJ_MSG_TYPE_CLEAR:
                            return bcmbal_cli_access_terminal_cfg_clear(session);
                        default:
                            return BCM_ERR_RANGE;
                    }

                default:
                    return BCM_ERR_RANGE;
            }

        case BCMBAL_OBJ_ID_FLOW:
            switch (group_type)
            {
                case BCMBAL_MGT_GROUP_CFG:
                    switch (msg_type)
                    {
                        case BCMBAL_OBJ_MSG_TYPE_GET:
                            return bcmbal_cli_flow_cfg_get(session);
                        case BCMBAL_OBJ_MSG_TYPE_SET:
                            return bcmbal_cli_flow_cfg_set(session);
                        case BCMBAL_OBJ_MSG_TYPE_CLEAR:
                            return bcmbal_cli_flow_cfg_clear(session);
                        default:
                            return BCM_ERR_RANGE;
                    }

                case BCMBAL_MGT_GROUP_STAT:
                    return bcmbal_cli_flow_stat_get(session);
                default:
                    return BCM_ERR_RANGE;
            }

        case BCMBAL_OBJ_ID_GROUP:
            switch (group_type)
            {
                case BCMBAL_MGT_GROUP_CFG:
                    switch (msg_type)
                    {
                        case BCMBAL_OBJ_MSG_TYPE_GET:
                            return bcmbal_cli_group_cfg_get(session, byte_pool);
                        case BCMBAL_OBJ_MSG_TYPE_SET:
                            return bcmbal_cli_group_cfg_set(session, byte_pool);
                        case BCMBAL_OBJ_MSG_TYPE_CLEAR:
                            return bcmbal_cli_group_cfg_clear(session);
                        default:
                            return BCM_ERR_RANGE;
                    }

                default:
                    return BCM_ERR_RANGE;
            }

        case BCMBAL_OBJ_ID_INTERFACE:
            switch (group_type)
            {
                case BCMBAL_MGT_GROUP_CFG:
                    switch (msg_type)
                    {
                        case BCMBAL_OBJ_MSG_TYPE_GET:
                            return bcmbal_cli_interface_cfg_get(session, byte_pool);
                        case BCMBAL_OBJ_MSG_TYPE_SET:
                            return bcmbal_cli_interface_cfg_set(session, byte_pool);
                        case BCMBAL_OBJ_MSG_TYPE_CLEAR:
                            return bcmbal_cli_interface_cfg_clear(session);
                        default:
                            return BCM_ERR_RANGE;
                    }

                case BCMBAL_MGT_GROUP_STAT:
                    return bcmbal_cli_interface_stat_get(session);
                default:
                    return BCM_ERR_RANGE;
            }

        case BCMBAL_OBJ_ID_PACKET:
            switch (group_type)
            {
                case BCMBAL_MGT_GROUP_CFG:
                    switch (msg_type)
                    {
                        case BCMBAL_OBJ_MSG_TYPE_GET:
                            return bcmbal_cli_packet_cfg_get(session, byte_pool);
                        case BCMBAL_OBJ_MSG_TYPE_SET:
                            return bcmbal_cli_packet_cfg_set(session, byte_pool);
                        case BCMBAL_OBJ_MSG_TYPE_CLEAR:
                            return bcmbal_cli_packet_cfg_clear(session);
                        default:
                            return BCM_ERR_RANGE;
                    }

                default:
                    return BCM_ERR_RANGE;
            }

        case BCMBAL_OBJ_ID_SUBSCRIBER_TERMINAL:
            switch (group_type)
            {
                case BCMBAL_MGT_GROUP_CFG:
                    switch (msg_type)
                    {
                        case BCMBAL_OBJ_MSG_TYPE_GET:
                            return bcmbal_cli_subscriber_terminal_cfg_get(session, byte_pool);
                        case BCMBAL_OBJ_MSG_TYPE_SET:
                            return bcmbal_cli_subscriber_terminal_cfg_set(session, byte_pool);
                        case BCMBAL_OBJ_MSG_TYPE_CLEAR:
                            return bcmbal_cli_subscriber_terminal_cfg_clear(session);
                        default:
                            return BCM_ERR_RANGE;
                    }

                case BCMBAL_MGT_GROUP_STAT:
                    return bcmbal_cli_subscriber_terminal_stat_get(session);
                default:
                    return BCM_ERR_RANGE;
            }

        case BCMBAL_OBJ_ID_TM_QUEUE:
            switch (group_type)
            {
                case BCMBAL_MGT_GROUP_CFG:
                    switch (msg_type)
                    {
                        case BCMBAL_OBJ_MSG_TYPE_GET:
                            return bcmbal_cli_tm_queue_cfg_get(session);
                        case BCMBAL_OBJ_MSG_TYPE_SET:
                            return bcmbal_cli_tm_queue_cfg_set(session);
                        case BCMBAL_OBJ_MSG_TYPE_CLEAR:
                            return bcmbal_cli_tm_queue_cfg_clear(session);
                        default:
                            return BCM_ERR_RANGE;
                    }

                case BCMBAL_MGT_GROUP_STAT:
                    return bcmbal_cli_tm_queue_stat_get(session);
                default:
                    return BCM_ERR_RANGE;
            }

        case BCMBAL_OBJ_ID_TM_SCHED:
            switch (group_type)
            {
                case BCMBAL_MGT_GROUP_CFG:
                    switch (msg_type)
                    {
                        case BCMBAL_OBJ_MSG_TYPE_GET:
                            return bcmbal_cli_tm_sched_cfg_get(session, byte_pool);
                        case BCMBAL_OBJ_MSG_TYPE_SET:
                            return bcmbal_cli_tm_sched_cfg_set(session, byte_pool);
                        case BCMBAL_OBJ_MSG_TYPE_CLEAR:
                            return bcmbal_cli_tm_sched_cfg_clear(session);
                        default:
                            return BCM_ERR_RANGE;
                    }

                default:
                    return BCM_ERR_RANGE;
            }

        default:
            return BCM_ERR_RANGE;
    }
}

/* Perform an API call based on CLI input */
bcmos_errno bcmbal_apicli_call(bcmbal_mgt_group group_type, bcmbal_obj_msg_type msg_type, bcmcli_session *session)
{
    bcmos_errno err;
    bcmbal_apicli_byte_pool byte_pool;

    /* setup memory pool for dynamically-sized list memory allocation */
    err = bcmbal_apicli_byte_pool_create(&byte_pool);
    if (err != BCM_ERR_OK)
    {
        return err;
    }

    /* call the root API handler */
    err = bcmbal_apicli_root(group_type, msg_type, session, &byte_pool);

    /* free all dynamically allocated memory */
    bcmbal_apicli_byte_pool_destroy(&byte_pool);

    return err;
}
