/******************************************************************************
 *
 *  <:copyright-BRCM:2016:DUAL/GPL:standard
 *  
 *     Copyright (c) 2016 Broadcom
 *     All Rights Reserved
 *  
 *  Unless you and Broadcom execute a separate written software license
 *  agreement governing use of this software, this software is licensed
 *  to you under the terms of the GNU General Public License version 2
 *  (the "GPL"), available at http://www.broadcom.com/licenses/GPLv2.php,
 *  with the following added to such license:
 *  
 *     As a special exception, the copyright holders of this software give
 *     you permission to link this software with independent modules, and
 *     to copy and distribute the resulting executable under terms of your
 *     choice, provided that you also meet, for each linked independent
 *     module, the terms and conditions of the license of that module.
 *     An independent module is a module which is not derived from this
 *     software.  The special exception does not apply to any modifications
 *     of the software.
 *  
 *  Not withstanding the above, under no circumstances may you combine
 *  this software in any way with any other Broadcom software provided
 *  under a license other than the GPL, without Broadcom's express prior
 *  written consent.
 *  
 *  :>
 *
 *****************************************************************************/
 
/**
 * @file bal_api.h
 * @brief Function declarations and all inclusions required for the BAL Public API
 *
 * @defgroup api BAL Public API
 */
#ifndef BCMBAL_API_H
#define BCMBAL_API_H

#include <bcmos_system.h>
#include <bcmolt_model_types.h>

#include <bal_objs.h>


/*@{*/

/*
 * This example only controls the default access terminal
 */
#define DEFAULT_ATERM_ID           (0)

/* Max number of access-terminals supported by BAL */
#define MAX_ACC_TERM_ID            (255)

/*------------------------------------------------
 * See the bal_objs.h file for the BAL objects
 *------------------------------------------------
 */

/*
 * An enumeration of the BAL mgmt operations
 */
typedef enum
{
    BCMBAL_MGMT_OPER_ID_SET = 1 << 0,
    BCMBAL_MGMT_OPER_ID_GET = 1 << 1,
    BCMBAL_MGMT_OPER_ID_GET_STATS = 1 << 2,
    BCMBAL_MGMT_OPER_ID_CLEAR = 1 << 3,
    BCMBAL_MGMT_OPER_ID__NUM_OF = 1 << 4,
    BCMBAL_MGMT_OPER_ID_ALL = BCMBAL_MGMT_OPER_ID__NUM_OF - 1,
} bcmbal_mgmt_oper_id;


/* access_terminal:key:acc_term_id max value */
#define BAL_API_MAX_ACC_TERM_ID           (255)

/* flow:key:flow_id max value */
#define BAL_API_MAX_FLOW_ID               (65535)

/* interface:mtu limits */
#define BAL_API_MIN_INTF_MTU_SIZE         (64)
#define BAL_API_MAX_INTF_MTU_SIZE         (9216)

/* Max number of interfaces per interface:type in BAL */
#define BAL_API_MAX_INTF_ID               (15)

/* Max sub_term_id per interface for the various modes */
#define BAL_API_MAX_SUB_TERM_ID_GPON      (127)
#define BAL_API_MAX_SUB_TERM_ID_EPON      (256)
#define BAL_API_MAX_SUB_TERM_ID_XGPON     (511)

/* Max and min values for interface attributes */
#define BAL_API_MIN_LOW_DATA_AGG_PORT_ID      (256)
#define BAL_API_MAX_LOW_DATA_AGG_PORT_ID      (14591)

#define BAL_API_MIN_LOW_DATA_SVC_PORT_ID      (1024)
#define BAL_API_MAX_LOW_DATA_SVC_PORT_ID      (57598)


/* Max and Min values for agg_port_id and svc_port_id */
#define BAL_API_MIN_DATA_SVC_PORT_ID      (256)
#define BAL_API_MAX_DATA_SVC_PORT_ID      (16383)

/* Max and Min values for agg_port_id and svc_port_id */
#define BAL_API_MIN_DATA_AGG_PORT_ID      (256)
#define BAL_API_MAX_DATA_AGG_PORT_ID      (16383)

/* Max value for pbits */
#define MAX_PBITS_VALUE                   (7)

/*
 * BAL CLI max values
 */
/** Max number of supported OLTs */
#define MAX_SUPPORTED_OLTS          (BAL_API_MAX_ACC_TERM_ID+1)

/** Max number of supported subscriber terminals (ONUs) */
#define MAX_SUPPORTED_SUBS          (BAL_API_MAX_SUB_TERM_ID_XGPON+1)

/** Max number of supported flows */
#define MAX_SUPPORTED_FLOWS         (BAL_API_MAX_FLOW_ID+1)

/** Max number of supported PON and NNI interfaces */
#define MAX_SUPPORTED_INTF          (BAL_API_MAX_INTF_ID+1)

/** BAL Indication callback handler function prototype */
typedef void (*f_bcmbal_ind_handler)(bcmbal_obj *obj);

/** BAL Indication callback registration parameters */
typedef struct bcmbal_cb_cfg
{
    bcmbal_obj_id          obj_type; 	       /**< Object type */
    f_bcmbal_ind_handler   ind_cb_hdlr;        /**< Indication callback function. NULL=unregister */
    bcmos_module_id        module;             /**< Target module id.
                                                  If it is BCMOS_MODULE_ID_NONE (0), the callback will be called
                                                  in BAL's context. Otherwise, it will be called in application
                                                  module's context */
} bcmbal_cb_cfg; /* This structure is used for passing the mgmt queue

 * IP:port information between the core main thread and
 * core worker thread and bal API.
 */
typedef struct mgmt_queue_addr_ports
{
    const char *core_mgmt_ip_port;
    const char *balapi_mgmt_ip_port;
} mgmt_queue_addr_ports;

extern dev_log_id log_id_public_api;

const char *bcmbal_get_interface_type_str(bcmbal_intf_type int_type);

/*
 *------------------------------------------------------------------------------------------
 *
 * @brief The BAL Public API
 *
 *------------------------------------------------------------------------------------------
 */

/**
 * @brief Initialize the BAL Public API internal data structures
 *
 * @param balapi_mgmt_ip_port The IP:port of the BAL API management queue
 *
 * @param core_mgmt_ip_port   The IP:port of the core management queue
 *
 * @returns bcmos_errno
 *
 **/
bcmos_errno bcmbal_api_init(const char *balapi_mgmt_ip_port, const char *core_mgmt_ip_port);

/**
 * @brief Un-initialize the BAL Public API internal data structures
 *
 * @returns bcmos_errno == BCM_ERR_OK
 *
 **/
bcmos_errno bcmbal_api_finish(void);

/**
 * @brief BAL Public API Set (or modify) command.
 *
 * Set (or modify) the specified object instance (with implicit creation
 * of dynamic objects) associated with the specified access-terminal device.
 *
 * @param objinfo A pointer to a BAL object
 *
 * @returns bcmos_errno
 *
 **/
bcmos_errno bcmbal_cfg_set(bcmbal_cfg *objinfo);

/**
 * @brief BAL Public API Get command.
 *
 * Get the specified object instance
 *
 * @param objinfo A pointer to a BAL object
 *
 * @returns bcmos_errno
 *
 */
bcmos_errno bcmbal_cfg_get(bcmbal_cfg *objinfo);

/**
 * @brief BAL Public API Packet Send function.
 *
 * Send a packet to the specified destination
 *
 * @param dest    The destination of the user packet
 *
 * @param packet_to_send A pointer to the user packet to send to the specified destination
 *
 * @param packet_len The length of the  user packet (must be <=1600 bytes)
 *
 * @returns bcmos_errno
 *
 */
bcmos_errno bcmbal_pkt_send(bcmbal_dest dest,
                            const char *packet_to_send,
                            uint16_t packet_len);

/**
 * @brief BAL Public API Clear command.
 *
 * Set all attributes to default (or remove the object instance for
 * dynamic objects) for the specified object instance
 *
 * @param objinfo A pointer to a BAL object
 *
 * @returns bcmos_errno
 *
 */
bcmos_errno bcmbal_cfg_clear(bcmbal_cfg * objinfo);

/**
 * @brief BAL Public API Get Stats command.
 *
 * Get (and clear) the stats associated with specified object instance
 *
 * @param objinfo A pointer to a BAL object
 *
 * @returns bcmos_errno
 *
 */
bcmos_errno bcmbal_stat_get(bcmbal_stat *objinfo);

/**
 * @brief BAL Public API indication subscription.
 *
 * Subscription function for the specified indications
 *
 * @param cb_cfg  A pointer to the callback configuration parameters for the
 *                object indications being subscribed to.
 *
 * @returns bcmos_errno
 *
 */
bcmos_errno bcmbal_subscribe_ind(bcmbal_cb_cfg *cb_cfg);

/**
 * @brief BAL Public API indication un-subscription.
 *
 * Un-subscription function for the specified (or all) indications
 *
 * @param cb_cfg  A pointer to the callback configuration parameters for the
 *                object indications being un-subscribed from.
 *
 * @returns bcmos_errno
 *
 */
bcmos_errno bcmbal_unsubscribe_ind(bcmbal_cb_cfg *cb_cfg);

/**
 * @brief Get the number of NNI ports supported by the running system
 *
 * @returns Number of NNI ports
 */
uint16_t bcmbal_num_nni_ports_get(void);

/*@}*/

#endif /* BCMBAL_API_H */
