/******************************************************************************
 *
 *  <:copyright-BRCM:2016:DUAL/GPL:standard
 *  
 *     Copyright (c) 2016 Broadcom
 *     All Rights Reserved
 *  
 *  Unless you and Broadcom execute a separate written software license
 *  agreement governing use of this software, this software is licensed
 *  to you under the terms of the GNU General Public License version 2
 *  (the "GPL"), available at http://www.broadcom.com/licenses/GPLv2.php,
 *  with the following added to such license:
 *  
 *     As a special exception, the copyright holders of this software give
 *     you permission to link this software with independent modules, and
 *     to copy and distribute the resulting executable under terms of your
 *     choice, provided that you also meet, for each linked independent
 *     module, the terms and conditions of the license of that module.
 *     An independent module is a module which is not derived from this
 *     software.  The special exception does not apply to any modifications
 *     of the software.
 *  
 *  Not withstanding the above, under no circumstances may you combine
 *  this software in any way with any other Broadcom software provided
 *  under a license other than the GPL, without Broadcom's express prior
 *  written consent.
 *  
 *  :>
 *
 *****************************************************************************/
 
/**
 * @file bal_dpp_qos.h
 * @brief BAL Switch Util QoS configuration API
 *
 * This include file contains the data structures and API for
 * configuring and managing Quality of Service (QoS) for services on
 * DPP (DUNE Packet Processor).
 *
 *
 */

/*@{*/

#ifndef	_BAL_DPP_QOS_H_
#define	_BAL_DPP_QOS_H_

/* --- project includes --- */

#include "flow_fsm.h"

/* --- local static constants ---*/

/**
 * @brief Value representing 10G in units of kbps
 */
#define BAL_BCM_QOS_10G_kbps 10000000

/**
 * @brief Value representing 2G (Turbo Mode) in units of kbps
 */
#define BAL_BCM_QOS_2G_kbps   2000000

/**
 * @brief Value representing 1G in units of kbps
 */
#define BAL_BCM_QOS_1G_kbps   1000000

/**
 * @brief Scheduler weight for 10G port scheduler
 */
#define BAL_BCM_SCHED_WEIGHT_10G 10

/**
 * @brief Scheduler weight for 1G port scheduler when Turbo Mode is enabled
 */
#define BAL_BCM_SCHED_WEIGHT_2G 2

/**
 * @brief Scheduler weight for 1G port scheduler (when Turbo Mode is disabled)
 */
#define BAL_BCM_SCHED_WEIGHT_1G 1

/**
 * @brief Scheduler weight used for MIN Rate schedulers
 */
#define BAL_BCM_SCHED_WEIGHT_MIN_RATE  1

/**
 * @brief Scheduler weight used for MAX Rate schedulers
 */
#define BAL_BCM_SCHED_WEIGHT_MAX_RATE  1

/**
 * @brief Number of PON port schedulers
 */
#define BAL_BCM_QOS_NUM_PON_PORTS 16

/**
 * @brief PON port Max value as defined in config.bcm
 */
#define BAL_BCM_MAX_PON_NUM 16

/**
 * @brief Number of queues per PON Link (LLID)
 */
#define BAL_BCM_QOS_QUEUES_PER_LLID 4

/**
 * @brief Default CoS level for untagged frames for 10G queues
 */
#define BAL_BCM_QOS_UNTAGGED_DEFAULT_COS_10G 0

/**
 * @brief Default CoS level for untagged frames for 1G queues
 */
#define BAL_BCM_QOS_UNTAGGED_DEFAULT_COS_1G 4

/**
 * @brief Size of ICT and Shared Vlan tag overhead (in bytes) 
 */
#define BAL_BCM_QOS_SINGLE_VLAN_TAG_HDR_SIZE 4

/**
 * @brief Base value for VOQ IDs assigned to the 1G channel
 *
 * All VOQ IDs configured on the 1G channel are configured with an
 * ID value of 0x4000 (16384) and larger. 
 */
#define BAL_BCM_QOS_VOQ_ID_1G_BASE 0x4000

/**
 * @brief WFQ maximum value for a weight
 */
#define BAL_BCM_SCHED_WFQ_MAX_WEIGHT 4096

/**
 * @brief Number of WFQ scheduling priority levels
 */
#define BAL_BCM_SCHED_WFQ_PRI_NUM 8

/**
 * @brief Default rate for PON port shaper in kbps (12.5Gbps)
 */
#define BAL_BCM_QOS_DEFAULT_PORT_RATE 12500000

/**
 * @brief Default rate for 10G PON channel shaper in kbps (10.25Gbps)
 *
 * The extra 0.25 Gbps allows for some overhead (flow control frames, etc.)
 */
#define BAL_BCM_QOS_DEFAULT_10G_CHAN_RATE 10250000

/**
 * @brief Default rate for 1G PON channel shaper in kbps (2.25Gbps)
 *
 * This is set to 2G to allow for Turbo Mode configurations.
 *
 * The extra 0.25 Gbps allows for some overhead (flow control frames, etc.)
 */
#define BAL_BCM_QOS_DEFAULT_1G_CHAN_RATE 2250000

/**
 * @brief Credit adjustment to account for scheduler overhead (percentage)
 *
 * The adjustment is five percent based on the recommendation from the
 * Dune team.
 */
#define BAL_BCM_QOS_CREDIT_RATE_ADJ 0.05

/**
 * @brief Shaper adjustment to account for scheduler overhead (percentage)
 *
 * The adjustment is one percent based on the recommendation from the
 * Dune team.
 */
#define BAL_BCM_QOS_SHAPER_RATE_ADJ 0.01

/**
 * @brief Default maximum bust size for PON port shaper in bytes (0 = disabled)
 */
#define BAL_BCM_QOS_DEFAULT_MAX_BURST 0

/**
 * @brief PFC flow control priority value for the 10G channel
 */
#define BAL_BCM_QOS_10G_CHAN_PFC 0

/**
 * @brief PFC flow control priority value for the 1G channel
 */
#define BAL_BCM_QOS_1G_CHAN_PFC 1

/**
 * @brief First logical PON port number
 */
#define BAL_PON_PORT_START  0

/**
 * @brief Last logical PON port number
 */
#define BAL_PON_PORT_END    7

/**
 * @brief The total number of PON ports
 */
#define NUM_PON_PORTS  (BAL_PON_PORT_END - BAL_PON_PORT_START + 1)

/**
 * @brief Default DPP header size used by scheduler and rate
 * calculations.
 */
#define DEFAULT_QOS_DPP_PKT_HDR_SIZE 19 /* bytes */

/**
 * @brief Size of the Dune PP header used to report the Flow ID (for
 * QoS) in packets trapped to the CPU.
 */
#define BAL_BCM_DPP_FLOWID_HDR_SIZE 5

/**
 * @brief Size of ICT and Shared Vlan tag overhead (in bytes) 
 */
#define BAL_BCM_QOS_SINGLE_VLAN_TAG_HDR_SIZE 4


/**
 * @brief Default VOQ Identifier Base Value for 10G flows
 *
 * 10G VOQ/Flow IDs can range from 3000..16383. This means the blocks
 * will be allocated in groups of four as follows.
 *
 *   blk 1 = 3000, 3001, 3002, 3003
 *   blk 2 = 3004, 3005, 3006, 3007
 *   ...
 *   blk last = 16380, 16381, 16382, 16383
 *
 * This means that the last ID used to allocate a 10G VOQ group will
 * be 16380.
 *
 * VOQ/flow IDs start at 3000 because the SDK assigns ID values
 * between 0..2000+ to port queues.
 */
#define DEFAULT_QOS_VOQ_BASE_10G 3000

/**
 * @brief Default VOQ Identifier Maximum Value for 10G flows
 *
 * The last available value for 10G flows is 16384 - 4.
 */
#define DEFAULT_QOS_VOQ_MAX_10G 16380

/**
 * @brief Default VOQ Identifier Base Value for 1G flows
 *
 * 1G VOQ/Flow IDs can range from 16384..32767. This means the blocks
 * will be allocated in groups of four as follows.
 *
 *   blk 1 = 16384, 16385, 16386, 16387
 *   blk 2 = 16388, 16389, 16390, 16392
 *   ...
 *   blk last = 32764, 32765, 32766, 32767
 *
 * This means that the last ID used to allocate a 1G VOQ group will be
 * 32764.
 */
#define DEFAULT_QOS_VOQ_BASE_1G 16384

/**
 * @brief Default VOQ Identifier Maximum Value for 1G flows
 *
 * The last available value for 1G flows is 32768 - 4.
 */
#define DEFAULT_QOS_VOQ_MAX_1G 32764

/**
 * @brief Get the PP port number for the specified PON port and
 * channel (10G vs. 1G)
 */
#define BAL_BCM_GET_PP_PORT(pon, chan) ((pon) + (NUM_PON_PORTS * (chan)))

/**
 * @brief PON channel (a.k.a speed, 10G vs. 1G)
 *
 * The 1G PON channel value is also used to represent 2G Turbo Mode
 * when enabled.
 */
typedef enum bal_sw_dpp_qos_sched_pon_chan
{
    BAL_BCM_SCHED_PON_CHAN_10G = 0, /**< 10G PON channel */
    BAL_BCM_SCHED_PON_CHAN_1G  = 1, /**< 1G PON channel */
    BAL_BCM_SCHED_PON_CHAN_NUM
} bal_sw_dpp_qos_sched_pon_chan;

/**
 * @brief SLA Type (MIN vs. MAX)
 */
typedef enum bal_sw_dpp_qos_sched_sla_type
{
    BAL_BCM_SCHED_SLA_MIN_RATE = 0, /**< MIN Rate */
    BAL_BCM_SCHED_SLA_MAX_RATE = 1, /**< MAX Rate */
    BAL_BCM_SCHED_SLA_TYPE_NUM
} bal_sw_dpp_qos_sched_sla_type;

/**
 * @brief Scheduling Type
 */
typedef enum bal_sw_dpp_qos_sched_type
{
    BAL_BCM_SCHED_TYPE_SP  = 0, /**< Strict priority scheduling */
    BAL_BCM_SCHED_TYPE_WFQ = 1, /**< Weighted Fair Queuing */
    BAL_BCM_SCHED_TYPE_NUM
} bal_sw_dpp_qos_sched_type;

/**
 * @brief SLA scheduler configuration
 */
typedef struct bal_sw_dpp_qos_sched_sla
{
    bcm_gport_t scheduler_gport;  /**< Scheduler object gport*/
    bcm_gport_t parent_gport;     /**< Parent scheduler gport */
    uint32_t rate;                    /**< Rate (or bandwidth) in Kbps */
    uint32_t burst;                   /**< Max burst in Kbits */
    uint8_t traffic_priority;          /**< Traffic priority */
} bal_sw_dpp_qos_sched_sla;

/**
 * @brief QoS configuration for a PON Link (LLID)
 */
typedef struct bal_sw_dpp_llid_qos
{
    bal_sw_dpp_qos_sched_pon_chan pon_chan; /**< PON Channel (or speed) the link is connected to */
    bal_sw_dpp_qos_sched_sla min;   /**< MIN Rate configuration */
    bal_sw_dpp_qos_sched_sla max;   /**< MAX Rate configuration */
    bcm_gport_t voq_connector_gport; /**< VOQ Connector gport  */
    bcm_gport_t voq_gport;  /**< VOQ gport  */
    int voq_flow_id;         /**< VOQ Flow ID */
    int voq_id;             /**< VOQ ID */
} bal_sw_dpp_llid_qos;

/**
 * @brief QoS configuration for a channelized PON port
 */
typedef struct bal_sw_dpp_port_qos_cfg
{
    bcm_gport_t mod_gport;         /**< MOD port for a channelized PON interface */
    bal_sw_dpp_qos_sched_type sched_type;   /**< Scheduling type used on the PON channel (SP vs. WFQ) */
    bcm_gport_t pon_chan_scheduler; /**< PON channel scheduler */
    bcm_gport_t sp_scheduler;      /**< Strict Priority scheduler for MAX rates (EIRs) */
    bcm_gport_t wfq_scheduler[BAL_BCM_SCHED_WFQ_PRI_NUM]; /**< WFQ+FQ schedulers for MAX rates (EIRs) */
    int32_t pon_chan_weight_cfg[BAL_BCM_SCHED_WFQ_PRI_NUM];     /**< Weight configuration for WFQ scheduler */
} bal_sw_dpp_port_qos_cfg;

/**
 * @brief Flow/VOQ ID Pool Entry
 */
typedef struct bal_sw_dpp_qos_flowid_pool_entry
{
    uint32_t  flow_id;    /**< Flow/VOQ ID value */
    TAILQ_ENTRY(bal_sw_dpp_qos_flowid_pool_entry) entry; /**< Link into the pool table */
} bal_sw_dpp_qos_flowid_pool_entry;

/**
 * @brief Flow/VOQ ID resource pool
 */
typedef struct bal_sw_dpp_qos_flowid_pool
{
    uint32_t next_flow_id_value; /**< Next flow ID to be allocated */
    uint32_t max_value;        /**< Maximum value for the flow IDs in this pool */
    TAILQ_HEAD(bal_sw_dpp_qos_flowid_pool_free_head, bal_sw_dpp_qos_flowid_pool_entry) free_table; /**< Flow/VOQ ID free pool */
} bal_sw_dpp_qos_flowid_pool;

/**
 * @brief DML BCM APP QoS configuration context definition
 */
typedef struct bal_sw_qos_cfg
{
    int32_t mod_id; /**< Module ID for this ARAD device */
    int32_t qos_port_map_id; /**< QoS Port Map Identifier */
    bal_sw_dpp_port_qos_cfg pon[BAL_BCM_QOS_NUM_PON_PORTS][BAL_BCM_SCHED_PON_CHAN_NUM];  /**< DS QoS configuration for channelized PON ports */
    bal_sw_dpp_qos_flowid_pool flow_id_pool_10g; /**< Flow/VOQ ID resource pool for 10G flows */
    bal_sw_dpp_qos_flowid_pool flow_id_pool_1g;  /**< Flow/VOQ ID resource pool for 1G flows */
    int num_channels_per_pon; 
} bal_sw_qos_cfg;

/**
 * @brief WFQ scheduling configuration
 *
 * This structure is used to configure weight values per scheduler
 * level (i.e., DOCSIS Traffic Priority). The range of weight values
 * is 0:4K, where values 1:4K are used for WFQ scheduling and a value
 * of '0' is used for strict priority.
 */
typedef struct bal_sw_dpp_qos_wfq_cfg
{
    uint32_t weights[BAL_BCM_SCHED_WFQ_PRI_NUM];
} bal_sw_dpp_qos_wfq_cfg;

extern bal_sw_qos_cfg *gp_bal_bcm_qos_cfg;

/**
 * @brief Max Number of NNI ports
 */
#define BAL_BCM_QOS_NUM_NNI_PORTS 16

/**
 * @brief Service Configuration Entry Type
 *
 * The type of service configuration that the DmlBcmServiceCfgT entry
 * represents.
 */
typedef enum bal_sw_service_type
{
    BAL_BCM_SVC_TYPE_NONE,  /**< None/uninitialized */
    BAL_BCM_SVC_TYPE_ICT,   /**< Intra-chassis Tagged */
    BAL_BCM_SVC_TYPE_IP,    /**< IP(HSD) */
    BAL_BCM_SVC_TYPE_PB,    /**< MEF Provider Bridging (VLAN tagged) */
    BAL_BCM_SVC_TYPE_SHVLAN,/**< Shared VLAN */
    BAL_BCM_SVC_TYPE_LAST   /**< Number of bal_sw_service_type_t values */
} bal_sw_service_type;

/* --- forward declarations --- */

typedef struct bal_sw_dpp_qos_service_cfg
{
    uint32_t bal_flow_id;
    bcm_port_t pon_port;         /* logical port number from bal flow config */
    uint32_t num_nni_gport;
    bcm_gport_t nni_gport[BAL_BCM_QOS_NUM_NNI_PORTS];
    uint32_t tunnel_id;
    bal_sw_dpp_llid_qos ds_qos; 
    bal_sw_service_type service_type;    
} bal_sw_dpp_qos_service_cfg;


/* --- function prototypes --- */

bcmos_errno bal_sw_dpp_qos_init(int unit, bal_swapp_port_map_indx pon_mode);
void        bal_sw_dpp_qos_cleanup(int unit);
bcmos_errno bal_sw_dpp_port_qos_init(int unit, bcm_port_t pon_port, bal_sw_dpp_qos_sched_pon_chan pon_chan);
void        bal_sw_dpp_port_qos_cleanup(int unit, bcm_port_t pon_port, bal_sw_dpp_qos_sched_pon_chan pon_chan);
bcmos_errno bal_sw_dpp_qos_set_port_bandwidth(int unit, bcm_port_t pon_port, uint32_t bandwidth, uint32_t max_burst);
bcmos_errno bal_sw_dpp_qos_set_portchan_bandwidth(int unit, bcm_port_t pon_port, bal_sw_dpp_qos_sched_pon_chan pon_chan, 
                                    uint32_t bandwidth, uint32_t max_burst);
bcmos_errno bal_sw_dpp_qos_set_ponchan_wfq_cfg(int unit, bcm_port_t pon_port,
                                 bal_sw_dpp_qos_sched_pon_chan pon_chan, bal_sw_dpp_qos_wfq_cfg *p_wfq_cfg);
bcmos_errno bal_sw_dpp_qos_set_llid_bandwidth(int unit, bal_sw_dpp_qos_service_cfg *p_service_cfg,
                                 bal_sw_dpp_qos_sched_sla_type sla_type, uint32_t bandwidth, uint32_t max_burst);
bcmos_errno bal_sw_dpp_llid_qos_config(int unit, bal_sw_dpp_qos_service_cfg *p_service_cfg_entry);
bcmos_errno bal_sw_dpp_llid_set_qos_port_map(int unit, bal_sw_dpp_qos_service_cfg *p_service_cfg_entry);
void        bal_sw_dpp_llid_qos_cleanup(int unit, bal_sw_dpp_qos_service_cfg *p_service_cfg_entry);
void        bal_sw_dpp_qos_clear_voq_stats(int unit, bcm_gport_t gport);
void        bal_sw_dpp_print_all_port_qos(int unit);
void        bal_sw_dpp_qos_print_flowid_pool(int unit);

#endif /* #ifndef _BAL_DPP_QOS_H_ */

/*@}*/
