/******************************************************************************
 *
 *  <:copyright-BRCM:2016:DUAL/GPL:standard
 *
 *     Copyright (c) 2016 Broadcom
 *     All Rights Reserved
 *
 *  Unless you and Broadcom execute a separate written software license
 *  agreement governing use of this software, this software is licensed
 *  to you under the terms of the GNU General Public License version 2
 *  (the "GPL"), available at http://www.broadcom.com/licenses/GPLv2.php,
 *  with the following added to such license:
 *
 *     As a special exception, the copyright holders of this software give
 *     you permission to link this software with independent modules, and
 *     to copy and distribute the resulting executable under terms of your
 *     choice, provided that you also meet, for each linked independent
 *     module, the terms and conditions of the license of that module.
 *     An independent module is a module which is not derived from this
 *     software.  The special exception does not apply to any modifications
 *     of the software.
 *
 *  Not withstanding the above, under no circumstances may you combine
 *  this software in any way with any other Broadcom software provided
 *  under a license other than the GPL, without Broadcom's express prior
 *  written consent.
 *
 *  :>
 *
 *****************************************************************************/

/**
 * @file bal_mac_util_common_itu_pon.h
 *
 * @brief mac util interfaces definition used by Bal Core, for ITU PON: GPON, XG-PON1, XGS-PON
 *
 * This file expose the APIs to the core to configure the mac
 * with regarding to the operation of access terminal, interface, subscriber terminal and flow.
 *
 * @addtogroup mac_util
 */

/*@{*/

#include <bal_mac_util.h>

/**
 * @brief Maple SLA bandwidth config request is aligned to 8000 Bytes/sec
 *
 * @note similar macros are defined in bcm_common_gpon.h embedded code.
 */
#define MAPLE_BW_GRANULARITY_BYTES 8000
#define MAPLE_BW_GRANULARITY_BYTES_10G 16000
#define MAPLE_MIN_BYTES_PER_SEC_FOR_DEFAULT_ALLOC_ID 32000
#define SLA_BW_NKBYTES_ALIGNED(_rate, _us_rate_factor)    \
    (_us_rate_factor) == 8 ?                              \
        ((((_rate) + MAPLE_BW_GRANULARITY_BYTES_10G-1) / MAPLE_BW_GRANULARITY_BYTES_10G) * MAPLE_BW_GRANULARITY_BYTES_10G) :                                                      \
        ((((_rate) + MAPLE_BW_GRANULARITY_BYTES-1) / MAPLE_BW_GRANULARITY_BYTES) * MAPLE_BW_GRANULARITY_BYTES)

/* The thumb-rule is GPON PON capacity (155520000 Bps) divided by the maximum number of ONUs (128), rounded by granularity constraint. */
#define SLA_GUARANTEED_BW_DEFAULT_BYTES_PER_SEC 1208000
/* Like the default guaranteed BW, but since we are going to configure best-effort class, additional BW must be bigger than guaranteed BW in at least 32000. */
#define SLA_MAX_BW_DEFAULT_BYTES_PER_SEC (SLA_GUARANTEED_BW_DEFAULT_BYTES_PER_SEC + MAPLE_MIN_BYTES_PER_SEC_FOR_DEFAULT_ALLOC_ID)
#define KILOBITS_PER_SEC2BYTES_PER_SEC(x) ((uint64_t)(x) * 1000 / 8)


char *mac_util_indication_get_obj_type_str (bcmolt_obj_id obj_type, mac_util_ind_obj_and_handlers handlers[], size_t handlers_size);
bcmos_errno mac_util_handle_indication (bcmolt_devid device_id, bcmolt_msg *p_msg, mac_util_ind_obj_and_handlers handlers[], size_t handlers_size);
bcmos_bool  mac_util_check_flow_is_double_tag_remove_o_tag (const bcmbal_flow_cfg *p_flow_req);
void mac_util_assign_agg_port_sla (bcmbal_tm_sched_cfg *p_tm_sched_req, bcmolt_pon_alloc_sla *p_agg_sla);

bcmos_errno maple_mac_unicast_flow_add (bcmbal_flow_cfg *p_flow, uint16_t pbit, uint16_t per_flow_mode_vlan_id, bal_util_oper_flow op_type, flow_list_entry **pp_mac_util_flow_entry);
bcmos_errno maple_mac_unicast_flow_remove (bcmbal_flow_cfg *p_flow, bal_util_oper_flow op_type, flow_list_entry *p_mac_util_flow_entry);
bcmos_errno maple_mac_broadcast_flow_add (bcmbal_flow_cfg *p_flow, uint16_t per_flow_mode_vlan_id, bal_util_oper_flow op_type,
                                            flow_list_entry **pp_mac_util_flow_entry);
bcmos_errno maple_mac_broadcast_flow_remove (bcmbal_flow_cfg *p_flow, uint16_t per_flow_mode_vlan_id, bal_util_oper_flow op_type, flow_list_entry *p_mac_util_flow_entry);
void mac_util_mark_flow_config_complete(flow_list_entry *p_flow_entry);
bcmos_errno maple_gpon_mac_check_gem_port_id(uint32_t if_id, uint32_t onu_id, uint16_t svc_port_id,
                                                       bcmos_bool *is_configured, bcmos_bool *is_activated);
bcmos_errno maple_xgpon_mac_check_gem_port_id(uint32_t if_id, uint32_t onu_id, uint16_t svc_port_id,
                                                         bcmos_bool *is_configured, bcmos_bool *is_activated);
bcmos_errno maple_gpon_gem_port_id_add(uint32_t if_id, uint16_t svc_port_id, uint32_t onu_id, bcmolt_gem_port_configuration *configuration);
bcmos_errno maple_xgpon_gem_port_id_add(uint32_t if_id, uint16_t svc_port_id, uint32_t onu_id, bcmolt_gem_port_configuration *configuration);
bcmos_errno maple_gpon_gem_port_id_remove(uint32_t if_id, uint16_t svc_port_id);
bcmos_errno maple_xgpon_gem_port_id_remove(uint32_t if_id, uint16_t svc_port_id);
bcmos_errno maple_gpon_mac_get_alloc_id_config (uint32_t if_id, uint32_t onu_id, uint16_t agg_id, bcmolt_alloc_state *alloc_id_state);

bcmos_errno maple_xgpon_mac_get_alloc_id_config (uint32_t if_id, uint32_t onu_id, uint16_t agg_id, bcmolt_alloc_state *alloc_id_state);


bcmos_errno maple_gpon_us_alloc_id_add(uint32_t if_id, uint32_t onu_id, uint16_t agg_id, bcmolt_pon_alloc_sla agg_sla);
bcmos_errno maple_xgpon_us_alloc_id_add(uint32_t if_id, uint32_t onu_id, uint16_t agg_id, bcmolt_pon_alloc_sla agg_sla);
bcmos_errno maple_gpon_us_alloc_id_remove(uint32_t if_id, uint16_t agg_id);
bcmos_errno maple_xgpon_us_alloc_id_remove(uint32_t if_id, uint16_t agg_id);

bcmos_errno  maple_mac_agg_port_add(uint32_t if_id, uint32_t onu_id, uint16_t agg_id, bcmolt_pon_alloc_sla sla);
bcmos_errno  maple_mac_agg_port_remove(uint32_t if_id, uint16_t agg_id);



/* functions to register for specific auto indications from maple */
bcmos_errno _mac_util_register_for_auto_indications (struct bcmolt_rx_cfg *p_rx_cfg,
                                                 mac_util_ind_obj_and_handlers obj_types_and_handlers[], uint16_t num_obj_types, bcmolt_devid device_id);
bcmos_errno mac_util_register_for_device_auto_indications (struct bcmolt_rx_cfg *p_rx_cfg, bcmolt_devid device_id);
bcmos_errno mac_util_register_for_gpon_auto_indications (struct bcmolt_rx_cfg *p_rx_cfg, bcmolt_devid device_id);
bcmos_errno mac_util_register_for_xgpon_auto_indications (struct bcmolt_rx_cfg *p_rx_cfg, bcmolt_devid device_id);


/** @brief generic struct for object set/validate handlers based on mac protocols */
typedef bcmos_errno (*mac_util_bal_group_validate_handler) (const bcmbal_group_cfg *p_group_req);
typedef bcmos_errno (*mac_util_bal_group_set_handler) (bcmbal_group_cfg *p_group_req, bal_util_oper_group op_type, group_inst *p_group_context);
typedef bcmos_errno (*mac_util_bal_flow_validate_handler) (const bcmbal_flow_cfg *p_flow_req);
typedef bcmos_errno (*mac_util_bal_flow_set_handler) (bcmbal_flow_cfg *p_flow_req, bal_util_oper_flow op_type, flow_inst *p_flow_core);
typedef bcmos_errno (*mac_util_bal_sub_term_validate_handler) (const bcmbal_subscriber_terminal_cfg *p_sub_term_req);
typedef bcmos_errno (*mac_util_bal_sub_term_set_handler) (sub_term_inst *p_sub_term_inst, bal_util_oper_sub_term op_type, bcmos_bool is_post_discovery);
typedef bcmos_errno (*mac_util_bal_if_validate_handler) (acc_term_inst *p_acc_term, bal_util_oper_if op_type);
typedef bcmos_errno (*mac_util_bal_if_set_handler) (acc_term_interface *p_interface_inst, bal_util_oper_if op_type);
typedef bcmos_errno (*mac_util_bal_acc_term_set_handler) (acc_term_inst *p_acc_term, bal_util_oper_acc_term op_type, bcmolt_devid device_id);
typedef bcmos_errno (*mac_util_bal_acc_term_post_indication_set_handler) (bcmolt_devid device_id);


/**
 * Structure for handler routines for different pon modes and loopback mode
 */
typedef struct
{
    mac_util_bal_if_validate_handler        if_validate;
    mac_util_bal_if_set_handler             if_set;

    mac_util_bal_sub_term_validate_handler  sub_term_validate;
    mac_util_bal_sub_term_set_handler       sub_term_set;

    mac_util_bal_flow_validate_handler      flow_validate;
    mac_util_bal_flow_set_handler           flow_set;

    mac_util_bal_group_validate_handler     group_validate;
    mac_util_bal_group_set_handler          group_set;

    uint32_t                                sla_us_rate_factor;
} mac_util_handlers_per_pon_mode_t;


typedef bcmos_errno (* maple_mac_check_gem_port_id)(uint32_t if_id, uint32_t onu_id, uint16_t svc_port_id,
                                                                 bcmos_bool *is_configured, bcmos_bool *is_activated);

typedef bcmos_errno (* maple_mac_check_agg_port_id)(uint32_t if_id, uint32_t onu_id, uint16_t agg_id, bcmolt_alloc_state *alloc_id_state);



/** access terminal set */
bcmos_errno mac_util_access_terminal_set_for_gpon_16 (acc_term_inst *p_acc_term, bal_util_oper_acc_term op_type, bcmolt_devid device_id);
bcmos_errno mac_util_access_terminal_set_for_gpon_8 (acc_term_inst *p_acc_term, bal_util_oper_acc_term op_type, bcmolt_devid device_id);
bcmos_errno mac_util_access_terminal_set_for_xgpon_8 (acc_term_inst *p_acc_term, bal_util_oper_acc_term op_type, bcmolt_devid device_id);
bcmos_errno mac_util_access_terminal_set_for_xgs (acc_term_inst *p_acc_term, bal_util_oper_acc_term op_type, bcmolt_devid device_id);
bcmos_errno mac_util_access_terminal_set_for_10G_itu_pon (bcmolt_device_cfg dev_cfg, bcmolt_devid device_id);

bcmos_errno mac_util_access_terminal_post_indication_set_for_gpon(bcmolt_devid device_id);
bcmos_errno mac_util_access_terminal_post_indication_set_for_xgpon_xgs(bcmolt_devid device_id);

bcmos_errno mac_util_access_terminal_set_for_loopback (acc_term_inst *p_acc_term, bal_util_oper_acc_term op_type, bcmolt_devid device_id);

bcmos_errno mac_util_update_flows_w_sub_term_update(uint32_t pon_if, uint32_t sub_term_id, maple_mac_check_gem_port_id check_gem_port_id_func, maple_mac_check_agg_port_id check_agg_port_id_func);
bcmos_errno  maple_mac_util_agg_port_set(bcmbal_tm_sched_cfg *p_tm_sched_req, bal_util_oper_agg_port op_type, tm_sched_inst *p_tm_sched_inst);



/**
 * Structure for handler routines for diffrent system modes and loopback mode
 */
typedef struct
{
    mac_util_bal_acc_term_set_handler       acc_term_set;
    mac_util_bal_acc_term_post_indication_set_handler acc_term_post_indication_set;
} mac_util_bal_req_handlers_for_system_mode_t;
#define MAC_UTIL_PMDB(pon) mac_util_bal_req_handlers_for_pon_mode[bcm_topo_pon_get_pon_mode((pon))]

typedef struct
{
    bcmolt_iwf_mode iwf_mode;
} mac_util_topo_pon_context;



typedef struct
{
    bcmolt_system_mode system_mode;
    bcmos_bool is_connected;
} mac_util_device_context;

typedef struct
{
    mac_util_device_context devices[BCM_TOPO_MAX_NUM_OF_DEVS];
    bcmos_bool fail_reported;   /*more that one device may fail to connect, we would like to send the fail event to core only once*/
} mac_util_acc_term_connectivity;

mac_util_acc_term_connectivity acc_term_connectivity;

extern mac_util_handlers_per_pon_mode_t mac_util_bal_req_handlers_for_pon_mode[];

/*@}*/
