/******************************************************************************
 *
 *  <:copyright-BRCM:2016:DUAL/GPL:standard
 *  
 *     Copyright (c) 2016 Broadcom
 *     All Rights Reserved
 *  
 *  Unless you and Broadcom execute a separate written software license
 *  agreement governing use of this software, this software is licensed
 *  to you under the terms of the GNU General Public License version 2
 *  (the "GPL"), available at http://www.broadcom.com/licenses/GPLv2.php,
 *  with the following added to such license:
 *  
 *     As a special exception, the copyright holders of this software give
 *     you permission to link this software with independent modules, and
 *     to copy and distribute the resulting executable under terms of your
 *     choice, provided that you also meet, for each linked independent
 *     module, the terms and conditions of the license of that module.
 *     An independent module is a module which is not derived from this
 *     software.  The special exception does not apply to any modifications
 *     of the software.
 *  
 *  Not withstanding the above, under no circumstances may you combine
 *  this software in any way with any other Broadcom software provided
 *  under a license other than the GPL, without Broadcom's express prior
 *  written consent.
 *  
 *  :>
 *
 *****************************************************************************/
 
/**
 * @file tm_queue_fsm.c
 * @brief Code to support the BAL tm_queue FSM
 *
 *
 */

/*@{*/

#include <bcmos_system.h>
#include <tm_queue_fsm.h>
#include <bal_msg.h>
#include <bal_osmsg.h>
#include "bal_worker.h"
#include "bal_mac_util.h"
#include "bal_switch_util.h"
#include "rsc_mgr.h"

#include <bal_objs.h>
#include <fsm_common.h>

#ifdef ENABLE_LOG
#include <bcm_dev_log.h>

 /*
 * @brief The logging device id for tm queue
 */
static dev_log_id log_id_tm_queue;
#endif

/* local function declarations */

static bcmos_errno tm_queue_free_by_entry(tm_queue_inst *p_entry);
static bcmos_errno tm_queue_fsm_state_err(tm_queue_inst *p_tm_queue_inst, void *msg, tm_queue_fsm_event *p_event);
static void tm_queue_inst_entry_obj_init(tm_queue_inst *p_entry);

/*
 * @brief The Global tm_queue fsm context data structure
 */
static tm_queue_fsm_ctx g_tm_queue_fsm_tm_queue_list_ctx;

/*
 * Macros for tm_queue ctx access
 */
#define TM_QUEUE_FSM_TM_QUEUE_LIST_CTX      (g_tm_queue_fsm_tm_queue_list_ctx)
#define TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR  (&g_tm_queue_fsm_tm_queue_list_ctx)


/*
 *  @brief The definition of a tm_queue FSM state processing function
 */
typedef bcmos_errno (* tm_queue_fsm_state_processor)(tm_queue_inst *, void *, tm_queue_fsm_event *);
static bcmos_errno bcmbal_tm_queue_create(tm_queue_inst *p_tm_queue_inst, 
										   void *msg, 
										   tm_queue_fsm_event *p_event);

static bcmos_errno bcmbal_tm_queue_active_destroy (tm_queue_inst *p_tm_queue_inst, 
										   void *msg, 
										   tm_queue_fsm_event *p_event);
static bcmos_errno bcmbal_tm_queue_inactive_destroy(tm_queue_inst *p_tm_queue_inst, 
										   void *msg, 
										   tm_queue_fsm_event *p_event);


static tm_queue_fsm_state_processor tm_queue_states[TM_QUEUE_FSM_STATE__NUM_OF][TM_QUEUE_FSM_EVENT_TYPE__NUM_OF] = 
{
    [TM_QUEUE_FSM_STATE_NULL] = 
    {
        [TM_QUEUE_FSM_EVENT_TYPE_CREATE] = bcmbal_tm_queue_create,
    },
    [TM_QUEUE_FSM_STATE_INACTIVE] = 
    {
        [TM_QUEUE_FSM_EVENT_TYPE_DESTROY] = bcmbal_tm_queue_inactive_destroy,
    },
    [TM_QUEUE_FSM_STATE_ACTIVE] = 
    {
        [TM_QUEUE_FSM_EVENT_TYPE_DESTROY] = bcmbal_tm_queue_active_destroy,
    },

};

static char *state_name_str[] =
{
    "TM_QUEUE_FSM_STATE_NULL",
    "TM_QUEUE_FSM_STATE_INACTIVE",	
    "TM_QUEUE_FSM_STATE_ACTIVE",
    "TM_QUEUE_FSM_STATE_IN_USE",
    "TM_QUEUE_FSM_STATE_DELETING",
};

/* Ensure that the name array size matches the associated enum */
BAL_STATIC_ASSERT(TM_QUEUE_FSM_STATE__LAST == (sizeof (state_name_str) / sizeof (char *)), tm_queue_fsm_state);

static char *tm_queue_state_name_get(tm_queue_fsm_state state)
{
    if(state < TM_QUEUE_FSM_STATE__LAST)
    {
        return state_name_str[state];
    }
    else
    {
        return "TM_QUEUE_UNKNOWN";
    }
}

static char *event_name_str[] =
{
    "TM_QUEUE_FSM_CREATE_EVENT",
    "TM_QUEUE_FSM_DESTROY_EVENT",
    "FLOW_FSM_UTIL_MSG_EVENT",
};

/* Ensure that the name array size matches the associated enum */
BAL_STATIC_ASSERT(TM_QUEUE_FSM_EVENT_TYPE__LAST == (sizeof (event_name_str) / sizeof (char *)), tm_queue_fsm_event_type);

static char *tm_queue_event_name_get(tm_queue_fsm_event_type event)
{
    if(event < TM_QUEUE_FSM_EVENT_TYPE__LAST)
    {
        return event_name_str[event];
    }
    else
    {
        return "TM_QUEUE_EVT_UNKNOWN";
    }
}

/*****************************************************************************/
/**
 * @brief The Tm queue FSM function which is executed when an error
 *        is encountered during FSM processing.
 *
 * @param p_tm_queue_inst     Pointer to a tm_queue instance
 * @param msg              Pointer to a BAL message received from one of
 *                         the BAL apps.
 * @param p_event          Pointer to a tm_queue event structure
 *
 * @returns bcmos_errno
 *****************************************************************************/
static bcmos_errno tm_queue_fsm_state_err(tm_queue_inst *p_tm_queue_inst,
                                      void *msg,
                                      tm_queue_fsm_event *p_event)
{
    bcmos_errno ret = BCM_ERR_INVALID_OP;

    BCM_LOG(DEBUG, log_id_tm_queue,
            "Error encountered processing TM_queue FSM"
            " - BAD EVENT ()\n");

    return ret;
}


/*****************************************************************************/
/**
 * @brief A function to initialize the Tm Queue FSM infrastructure.
 *
 *        NOTE: This is called once on startup and NOT for each FSM instance.
 *
 * @returns bcmos_errno
 *****************************************************************************/
bcmos_errno tm_queue_fsm_init(void)
{
    int ii;
    tm_queue_inst *new_entry;
    bcmos_errno ret = BCM_ERR_OK;

#ifdef ENABLE_LOG
    log_id_tm_queue = bcm_dev_log_id_register("TM_QUEUE", DEV_LOG_LEVEL_INFO, DEV_LOG_ID_TYPE_BOTH);
    BUG_ON(log_id_tm_queue == DEV_LOG_INVALID_ID);
#endif

    /* Initialize all of the tm_queue queues */
    TAILQ_INIT(&TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->free_tm_queue_list);
    TAILQ_INIT(&TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->active_tm_queue_list);

    /* Populate the free list with it's initial set of tm_queues
     */
    for(ii=0; ii<TM_QUEUE_ALLOCATION_BLOCK_SIZE; ii++)
    {
        new_entry = bcmos_calloc(sizeof(tm_queue_inst));
        if(NULL == new_entry)
        {
            BCM_LOG(FATAL, log_id_tm_queue,  "Failed to initialize the tm_queue free list - FATAL\n");
            ret = BCM_ERR_NOMEM;
            break;
        }
       tm_queue_free_by_entry(new_entry);
    }
    return ret;
}

/*****************************************************************************/
/**
 * @brief A function to un-initialize the Tm Queue FSM infrastructure.
 *
 *        NOTE: This is called once on shutdown and NOT for each FSM instance.
 *
 * @returns bcmos_errno
 *****************************************************************************/
bcmos_errno tm_queue_fsm_finish(void)
{

    tm_queue_inst *current_entry, *p_temp_entry;

    /* Free all the entries on the active list */
    TAILQ_FOREACH_SAFE(current_entry,
                       &TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->active_tm_queue_list,
                       tm_queue_inst_next,
                       p_temp_entry)
    {
        /* Remove it from the active list */
        TAILQ_REMOVE(&TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->active_tm_queue_list, current_entry, tm_queue_inst_next);

        bcmos_free(current_entry);

    }

    /* Free all the entries on the free list */
    TAILQ_FOREACH_SAFE(current_entry,
                       &TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->free_tm_queue_list,
                       tm_queue_inst_next,
                       p_temp_entry)
    {
        /* Remove it from the active list */
        TAILQ_REMOVE(&TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->free_tm_queue_list, current_entry, tm_queue_inst_next);

        bcmos_free(current_entry);
    }

    return BCM_ERR_OK;
}

static void tm_queue_inst_entry_obj_init(tm_queue_inst *p_entry)
{
    /* The actual key content is irrelevant for free tm_queues */
    bcmbal_tm_queue_key key = { .id = 0, .sched_id = 0, .sched_dir = BCMBAL_TM_SCHED_DIR_US};

    /* And add it to the free list */
    p_entry->fsm_state = TM_QUEUE_FSM_STATE_NULL;

    BCMBAL_CFG_INIT(&p_entry->current_tm_queue_info,
                    tm_queue,
                    key);

    BCMBAL_OBJ_IN_PROGRESS_SET(&(p_entry->current_tm_queue_info), BCMOS_FALSE);
}
/*****************************************************************************/
/**
 * @brief A function to free a tm_queue instance specified by a the supplied
 *        entry pointer.
 *
 * @param p_entry A pointer to the entry to be freed
 *
 *
 * @returns bcmos_errno
 *****************************************************************************/
static bcmos_errno tm_queue_free_by_entry(tm_queue_inst *p_entry)
{
    bcmos_errno ret = BCM_ERR_OK;
    tm_queue_inst *current_entry;
    tm_queue_inst *p_temp_entry;

    /*
     * First, check the active list (an active tm_queue can be in the adding or removing state)
     */
    TAILQ_FOREACH_SAFE(current_entry,
                       &TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->active_tm_queue_list,
                       tm_queue_inst_next,
                       p_temp_entry)
    {
        if(current_entry == p_entry)
        {
           /* Remove it from the active list */
            TAILQ_REMOVE(&TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->active_tm_queue_list, current_entry, tm_queue_inst_next);
            break;
        }
    }
	tm_queue_inst_entry_obj_init(p_entry);
    TAILQ_INSERT_TAIL(&TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->free_tm_queue_list, p_entry, tm_queue_inst_next);

    return ret;
}

/*****************************************************************************/
/**
 * @brief The Tm queue FSM state processing executive function
 * @param p_tm_queue_inst  Pointer to a tm_queu instance
 * @param p_event      Pointer to a tm_Queue event structure
 *
 * @returns bcmos_errno
 *****************************************************************************/
static bcmos_errno tm_queue_fsm_exec(tm_queue_inst *p_tm_queue_inst, tm_queue_fsm_event *p_event)
{
    bcmos_errno ret = BCM_ERR_OK;
    tm_queue_fsm_state pre_state;
    tm_queue_fsm_state_processor tm_queue_state_processor;

    /* Parameter checks */
    BUG_ON(NULL == p_tm_queue_inst);
    BUG_ON(NULL == p_event);

    /* Record the present state for debug printing
     */
    pre_state = p_tm_queue_inst->fsm_state;

    /*
     * Get the state processing function
     */
    tm_queue_state_processor = tm_queue_states[p_tm_queue_inst->fsm_state][p_event->event_type];

    /*
     * If there's a state processing function for this event and state, execute it.
     * Otherwise, process a generic error.
     */
    if(tm_queue_state_processor)
    {
        ret = tm_queue_state_processor(p_tm_queue_inst, p_event->msg, p_event);
    } 
	else
    {
        tm_queue_fsm_state_err(p_tm_queue_inst, p_event->msg, p_event);
    }

    if(BCM_ERR_OK != ret)
    {
        BCM_LOG(ERROR, log_id_tm_queue, "*** Error detected during state processing\n");
        p_tm_queue_inst->fsm_state = pre_state;
    }

    BCM_LOG(DEBUG, log_id_tm_queue,  "*** Event %s, State: %s --> %s\n\n",
            tm_queue_event_name_get(p_event->event_type),
            tm_queue_state_name_get(pre_state),
            tm_queue_state_name_get(p_tm_queue_inst->fsm_state));

    return ret;
}

/*****************************************************************************/
/**
 * @brief A function to retrieve a tm_queue instance of the specified class.
 *
 * @param key            A pointer to the key of the tm_queue being referenced
 * @param search_flag    A flag specifying the type of tm_queue
 *                       instance to be retrieved
 *
 * @returns tm_queue_inst_t* A pointer to the found tm_queue instance,
 *                       or NULL if one is not found
 *****************************************************************************/
tm_queue_inst *tm_queue_inst_get(bcmbal_tm_queue_key key, tm_queue_flag search_flag)
{
    tm_queue_inst *current_entry = NULL;

    /*
     * First, check the active list if the caller has chosen to do so
     */
    if(TM_QUEUE_FLAG_ACTIVE & search_flag)
    {
        TAILQ_FOREACH(current_entry,
                      &TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->active_tm_queue_list,
                      tm_queue_inst_next)
        {

            if(current_entry->api_req_tm_queue_info.key.id == key.id
				&& current_entry->api_req_tm_queue_info.key.sched_id == key.sched_id
                && current_entry->api_req_tm_queue_info.key.sched_dir == key.sched_dir)
            {
                BCM_LOG(DEBUG, log_id_tm_queue,  "Found active tm_queue\n");
                /* The tm_queue instance pointer is in current_entry */
                break;
            }
        }
    }
    /*
     * Next, check the free list if the caller has chosen to do so
     */
    if((TM_QUEUE_FLAG_FREE & search_flag) && (NULL == current_entry))
    {
        /* Now check the free list */
        if(!TAILQ_EMPTY(&TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->free_tm_queue_list))
        {
            /* Just grab the first entry */
            current_entry = TAILQ_FIRST(&TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->free_tm_queue_list);

            /* Remove it from the free list */
            TAILQ_REMOVE(&TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->free_tm_queue_list, current_entry, tm_queue_inst_next);

            /* And add it to the active list */
            TAILQ_INSERT_TAIL(&TM_QUEUE_FSM_TM_QUEUE_LIST_CTX_PTR->active_tm_queue_list, current_entry, tm_queue_inst_next);

            /*
             * Initialize the fsm state
             */
            current_entry->fsm_state = TM_QUEUE_FSM_STATE_NULL;

            BCM_LOG(DEBUG, log_id_tm_queue,  "Using new tm_queue\n");
        }
    }

    if((TM_QUEUE_FLAG_ANY & search_flag) && (NULL == current_entry))
    {
        /*A tm_queue was not found on either list*/

        BCM_LOG(DEBUG, log_id_tm_queue,  "************** ERROR: no tm_queue found\n");
    }

    return current_entry;
}

static bcmos_errno bcmbal_tm_queue_validate(bcmbal_tm_queue_cfg *p_tm_queue_cfg)
{
    bcmos_errno  ret = BCM_ERR_OK;
    bcmbal_tm_sched_key tm_sched_key = {.id = p_tm_queue_cfg->key.sched_id, .dir = p_tm_queue_cfg->key.sched_dir};
    tm_sched_inst *p_tm_sched_inst = NULL;
    do
    {
        /*get the tm sched instance*/
        p_tm_sched_inst = tm_sched_inst_get(tm_sched_key, TM_SCHED_FLAG_ACTIVE);
        if(NULL == p_tm_sched_inst)
        {
            BCM_LOG(ERROR, log_id_tm_queue,  "ERROR: no related tm_sched found (id = %d dir = %s)\n",
                                               tm_sched_key.id, TM_SCHED_DIR_TO_STR(tm_sched_key.dir));
            ret = BCM_ERR_NOENT;
            break;
        }
        /*check it is not an agg port tm sched - as there are no queues to such node*/      
        if(BCMBAL_CFG_PROP_IS_SET(&p_tm_sched_inst->req_tm_sched_info, tm_sched, owner))
		{
            if(BCMBAL_TM_SCHED_OWNER_TYPE_AGG_PORT == p_tm_sched_inst->req_tm_sched_info.data.owner.type)
            {
                BCM_LOG(ERROR, log_id_tm_queue,  "ERROR: should not attach queue to node which is owned by agg port (id = %d dir = %s)\n",
                                                   tm_sched_key.id, TM_SCHED_DIR_TO_STR(tm_sched_key.dir));
                ret = BCM_ERR_PARM;
                break;
            }				
		}

        /*sched level should be 'queue'*/
        if(BCMBAL_TM_SCHED_CHILD_TYPE_QUEUE != p_tm_sched_inst->req_tm_sched_info.data.sched_child_type)		
        {
            BCM_LOG(ERROR, log_id_tm_queue,  "ERROR: tm sched sched level is invalid (%d), should be BCMBAL_TM_SCHED_CHILD_TYPE_QUEUE\n",
                    p_tm_sched_inst->req_tm_sched_info.data.sched_child_type);
            ret = BCM_ERR_PARM;
            break;
        }

        /*based on sched type, check priority / weight is set and is valid */
        if(BCMBAL_TM_SCHED_TYPE_SP == p_tm_sched_inst->req_tm_sched_info.data.sched_type)
        {
            if(!BCMBAL_CFG_PROP_IS_SET(p_tm_queue_cfg, tm_queue, priority))
            {
                BCM_LOG(ERROR, log_id_tm_queue, "Queue must be set with a priority, as its scheduler' sched_type is sp\n");
                ret = BCM_ERR_PARM;
                break;
            }
            if(BCMBAL_CFG_PROP_IS_SET(p_tm_queue_cfg, tm_queue, weight))
            {
                BCM_LOG(ERROR, log_id_tm_queue, "Queue can not be set with a weight, as its scheduler' sched_type is sp\n");
                ret = BCM_ERR_PARM;
                break;
            }
            if(p_tm_queue_cfg->data.priority >= p_tm_sched_inst->req_tm_sched_info.data.num_priorities)
            {
                BCM_LOG(ERROR, log_id_tm_queue, "Queue priority (%d) is higher than the allowed at parent scheduler (id = %d dir = %s num of priorities = %d )\n",
                        p_tm_queue_cfg->data.priority , tm_sched_key.id, TM_SCHED_DIR_TO_STR(tm_sched_key.dir),
                        p_tm_sched_inst->req_tm_sched_info.data.num_priorities);
                
                ret = BCM_ERR_PARM;
                break;
            }
        }     
        else
            if(BCMBAL_TM_SCHED_TYPE_WFQ == p_tm_sched_inst->req_tm_sched_info.data.sched_type)
            {
                if(!BCMBAL_CFG_PROP_IS_SET(p_tm_queue_cfg, tm_queue, weight))
                {
                    BCM_LOG(ERROR, log_id_tm_queue, "Queue must be set with a weight, as its scheduler' sched_type sched_type is wfq\n");
                    ret = BCM_ERR_PARM;
                    break;
                }
                if(BCMBAL_CFG_PROP_IS_SET(p_tm_queue_cfg, tm_queue, priority))
                {
                    BCM_LOG(ERROR, log_id_tm_queue, "Queue can not be set with a priority, as its scheduler' sched_type sched_type is wfq \n");
                    ret = BCM_ERR_PARM;
                    break;
                }
            }
        else
            if(BCMBAL_TM_SCHED_TYPE_SP_WFQ == p_tm_sched_inst->req_tm_sched_info.data.sched_type)
            {
                if(BCMBAL_CFG_PROP_IS_SET(p_tm_queue_cfg, tm_queue, priority)
                    && BCMBAL_CFG_PROP_IS_SET(p_tm_queue_cfg, tm_queue, weight))
                {
                    BCM_LOG(ERROR, log_id_tm_queue, "Queue must be set with either weight or priority, not both. as its scheduler' sched_type is sp_wfq\n");
                    ret = BCM_ERR_PARM;
                    break;
                }        
                if(!BCMBAL_CFG_PROP_IS_SET(p_tm_queue_cfg, tm_queue, priority)
                    && !BCMBAL_CFG_PROP_IS_SET(p_tm_queue_cfg, tm_queue, weight))
                {
                    BCM_LOG(ERROR, log_id_tm_queue, "Queue must be set with either weight or priority, as its scheduler' sched_type is sp_wfq\n");
                    ret = BCM_ERR_PARM;
                    break;
                }
            }
        }while(0);
    return ret;
}

/*****************************************************************************/
/**
 * @brief A function called by the core worker thread to process an
 *        tm_queue object message (SET, GET, CLEAR, STATS) received
 *        from the BAL Public API.
 *
 * @param msg_payload      Pointer to a BAL message received from the
 *                         BAL Public API.
 *
 * @returns bcmos_errno
 *****************************************************************************/
bcmos_errno process_tm_queue_object(void *msg_payload)
{
    bcmos_errno ret = BCM_ERR_OK;	
    bcmbal_tm_queue_cfg *p_tm_queue_cfg = (bcmbal_tm_queue_cfg *)msg_payload;
    tm_queue_inst *p_tm_queue_inst = NULL;
    tm_queue_fsm_event fsm_event;
    bcmbal_tm_queue_key tm_queue_key;
    bcmbal_obj_msg_type oper_type;
    bcmbal_tm_sched_key tm_sched_key;    
    tm_sched_inst *p_tm_sched_inst = NULL;
    bcmbal_interface_key intf_key;
    bcmbal_tm_sched_owner owner;
	
    BUG_ON(NULL == msg_payload);
    
    BCM_LOG(DEBUG, log_id_tm_queue,  "Processing a tm queue object\n");
    
    tm_queue_key = p_tm_queue_cfg->key;   
    oper_type = p_tm_queue_cfg->hdr.hdr.type;
    
    /*
    * A message pointer may be passed inside the event structure.
    */
    fsm_event.msg = msg_payload;
    
    /* SET or GET or CLEAR...? */
    switch(oper_type)
    {
        case(BCMBAL_OBJ_MSG_TYPE_SET):
        {
            do
            {
                BCM_LOG(DEBUG, log_id_tm_queue,  "Processing a tm queue SET REQ mgmt message\n");
                if(BCMBAL_STATUS_UP != acc_term_status_get())
                {
                    BCM_LOG(ERROR, log_id_tm_queue, "ERROR - Access-terminal is not UP. No further processing\n");
                    ret = BCM_ERR_STATE;
                    break;
                }

                /*Find the specified tm_queue instance is already created */
                p_tm_queue_inst = tm_queue_inst_get(tm_queue_key, TM_QUEUE_FLAG_ACTIVE);
				
                if(NULL != p_tm_queue_inst)
                {
                    /* This is a fatal error condition */
                    BCM_LOG(ERROR, log_id_tm_queue,
                        "ERROR - tm queue found.  Set after create is currently not supported. No further processing\n");
                    ret = BCM_ERR_NOT_SUPPORTED;
                    break;
                }
				
                p_tm_queue_inst = tm_queue_inst_get(tm_queue_key, TM_QUEUE_FLAG_FREE);
                if(NULL == p_tm_queue_inst )
                {
                    /* This is a fatal error condition */
                    BCM_LOG(ERROR, log_id_tm_queue,
                        "ERROR - tm queue not found.  No further processing\n");
                    ret = BCM_ERR_NOMEM;
                    break;
                }
				if(BCM_ERR_OK != (ret = bcmbal_tm_queue_validate(p_tm_queue_cfg)))
								  {
									  BCM_LOG(ERROR, log_id_tm_queue, "tm_queue fsm validation Failed (%d)\n", ret);
									  tm_queue_free_by_entry(p_tm_queue_inst);
									  break;
								  }
								  
								  
								  /* Perform the validation check(s) that the utils require */
								  if(BCM_ERR_OK != (ret = sw_util_tm_queue_validate(p_tm_queue_cfg)))
								  {
									  BCM_LOG(ERROR, log_id_tm_queue, "tm_queue switch validation Failed (%d)\n", ret);
									  tm_queue_free_by_entry(p_tm_queue_inst);
									  break;
								  }

				

				BCM_LOG(DEBUG, log_id_tm_queue,
					"Creating a new tm queue\n");
			
            }while(0);
			
            ret = mgmt_msg_send_balapi_rsp(ret, msg_payload, oper_type, log_id_tm_queue);
            
            if(BCM_ERR_OK != ret )
            {
                break;
            }
			
            BCMBAL_CFG_PROP_SET(p_tm_queue_cfg, tm_queue, creation_mode, BCMBAL_TM_CREATION_MODE_MANUAL);
            
            bcmbal_tm_queue_object_overlay(&p_tm_queue_inst->api_req_tm_queue_info, p_tm_queue_cfg);
            
            fsm_event.event_type = TM_QUEUE_FSM_EVENT_TYPE_CREATE;
            fsm_event.msg = msg_payload;
            ret = tm_queue_fsm_exec(p_tm_queue_inst,&fsm_event);
        }
        break;
		
        case(BCMBAL_OBJ_MSG_TYPE_GET):
        {        
            BCM_LOG(DEBUG, log_id_tm_queue,  "Processing a tm queue GET REQ mgmt message\n");
            if(BCMBAL_STATUS_UP != acc_term_status_get())
            {
                BCM_LOG(ERROR, log_id_tm_queue, "ERROR - Access-terminal is not UP.  No further processing\n");
                ret = BCM_ERR_STATE;
            }
            else
            {
            /* Find the specified tm_queue instance */
                p_tm_queue_inst = tm_queue_inst_get(tm_queue_key, TM_QUEUE_FLAG_ACTIVE);
            }
            
            if(NULL == p_tm_queue_inst)
            {
                BCM_LOG(ERROR, log_id_tm_queue,  "ERROR - Specified tm_queue (%d) not found\n", tm_queue_key.id);
                ret =  BCM_ERR_NOENT;
                
           	}
            else
            {
                /* We respond to the BAL public API backend with a result. We always
                            * send a complete msg_payload back to the API, but the data portion
                            * of the object is only relevant when a GET or GET-STATS has been requested */
                p_tm_queue_inst->api_req_tm_queue_info.hdr.hdr.comm_hdr = ((bcmbal_obj *)msg_payload)->comm_hdr;
                *((bcmbal_tm_queue_cfg *)msg_payload) = p_tm_queue_inst->api_req_tm_queue_info;
            }            
            mgmt_msg_send_balapi_rsp(ret, msg_payload, oper_type, log_id_tm_queue);
        }
        break;
        
        case(BCMBAL_OBJ_MSG_TYPE_CLEAR):
        {
            do
            {
                BCM_LOG(DEBUG, log_id_tm_queue,  "Processing a tm queue CLEAR REQ mgmt message\n");
                if(BCMBAL_STATUS_UP != acc_term_status_get())
                {
                    BCM_LOG(ERROR, log_id_tm_queue, "ERROR - Access-terminal is not UP.  No further processing\n");
                    ret = BCM_ERR_STATE;
                    break;
                }
                /*
                * Find the specified tm_queue instance
                */
                p_tm_queue_inst = tm_queue_inst_get(tm_queue_key, TM_QUEUE_FLAG_ACTIVE);
                
                if(NULL == p_tm_queue_inst)
                {
                    BCM_LOG(ERROR, log_id_tm_queue, "Specified tm_queue not found on CLEAR\n");
                    ret = BCM_ERR_NOENT;
                    break;                
                }
				

                if(TM_QUEUE_FSM_STATE_IN_USE == p_tm_queue_inst->fsm_state)
                {
                    BCM_LOG(ERROR, log_id_tm_queue, "Specified tm_queue is in use and can not be removed\n");
                    ret = BCM_ERR_PARM;
                    break;
                }

				/*TM_QUEUEs (either auto or manually created) that are associated with a TM_NODE with owner interface 
				may only be deleted when the associated interface is in the ADMIN-DOWN state*/
			    tm_sched_key.id = tm_queue_key.sched_id;
				tm_sched_key.dir = tm_queue_key.sched_dir;
				
				p_tm_sched_inst = tm_sched_inst_get(tm_sched_key, TM_SCHED_FLAG_ACTIVE);
				if(NULL == p_tm_sched_inst)
                {
					BCM_LOG(ERROR, log_id_tm_queue,  "ERROR: no related tm_sched found (id = %d dir = %s)\n",
													   tm_sched_key.id, TM_SCHED_DIR_TO_STR(tm_sched_key.dir));
					ret = BCM_ERR_NOENT;
                    break;
                }
				if(BCMBAL_CFG_PROP_IS_SET(&(p_tm_sched_inst->req_tm_sched_info), tm_sched, owner))
				{
					owner = p_tm_sched_inst->req_tm_sched_info.data.owner;
					switch(owner.type)
					{
                        case BCMBAL_TM_SCHED_OWNER_TYPE_INTERFACE:
                        {
							intf_key.intf_type = owner.u.interface.intf_type;
							intf_key.intf_id = owner.u.interface.intf_id;
							if(BCMBAL_STATUS_DOWN != bcmbal_interface_status_get(intf_key))
							{
								BCM_LOG(ERROR, log_id_tm_queue, "Specified tm_queue is attached to sched that is owned by interface that is not admin down, and can be removed only when the associated interface is in the ADMIN-DOWN state\n");
								ret = BCM_ERR_PARM;
								break;	   
							}		
                       	}
						break;

						default:
							break;
					}
				}

                /*generate destroy event*/
                
                fsm_event.event_type = TM_QUEUE_FSM_EVENT_TYPE_DESTROY;
                fsm_event.msg = msg_payload;
                ret = tm_queue_fsm_exec(p_tm_queue_inst,&fsm_event);
                
            }while(0);
            mgmt_msg_send_balapi_rsp(ret, msg_payload, oper_type, log_id_tm_queue);			
        }
        break;

        default:
        {
            BCM_LOG(ERROR, log_id_tm_queue,  "Unsupported operation on tm queue object (%d)\n", oper_type );
            ret = BCM_ERR_NOT_SUPPORTED;
            
            /* We respond to the BAL public API backend with a result. We always
            * send a complete msg_payload back to the API, but the data portion
            * of the object is only relevant when a GET or GET-STATS has been requested.
            */
            mgmt_msg_send_balapi_rsp(ret, msg_payload, oper_type, log_id_tm_queue);
        }
    }
    
    BCM_LOG(DEBUG, log_id_tm_queue, "%s returns\n", __FUNCTION__);
    return ret;

}

bcmos_errno bcmbal_tm_queue_auto_create(bcmbal_tm_queue_cfg cfg)
{
    bcmos_errno ret = BCM_ERR_OK;
    tm_queue_inst *p_tm_queue_inst = NULL;
    do
    {
        /*Find a free tm_queue instance */
        p_tm_queue_inst = tm_queue_inst_get(cfg.key, TM_QUEUE_FLAG_FREE);
        if(NULL == p_tm_queue_inst)
        {
            /* This is a fatal error condition*/
            BCM_LOG(ERROR, log_id_tm_queue,"ERROR - tm queue not found.  No further processing\n");
            ret = BCM_ERR_NOMEM;
            break;
        }
        p_tm_queue_inst->api_req_tm_queue_info = cfg;
        ret = bcmbal_tm_queue_create(p_tm_queue_inst,NULL, NULL);
    }while(0);
    return ret;
}

bcmos_errno bcmbal_tm_queue_activate(tm_queue_inst *p_tm_queue_inst)
{
    bcmos_errno ret = BCM_ERR_OK;
    do
    {

        ret = sw_util_tm_queue_set(p_tm_queue_inst);
        if(BCM_ERR_OK != ret)
        {
            BCM_LOG(ERROR, log_id_tm_queue, "queue	could not be set by switch\n");
            break;
        }

        p_tm_queue_inst->fsm_state = TM_QUEUE_FSM_STATE_ACTIVE; 
        /* The hardware has properly accepted the object info, so the request object becomes
        * the current state.
        */
        bcmbal_tm_queue_object_overlay(&p_tm_queue_inst->current_tm_queue_info,
            &p_tm_queue_inst->api_req_tm_queue_info);
    
    }while(0);
    return ret;

}

bcmos_errno bcmbal_tm_queue_deactivate(tm_queue_inst *p_tm_queue_inst)
{
    bcmos_errno ret = BCM_ERR_OK;
    
    BUG_ON(NULL == p_tm_queue_inst);
    
    do
    {
        ret = sw_util_tm_queue_clear(p_tm_queue_inst);
        if(BCM_ERR_OK != ret)
        {
            BCM_LOG(ERROR, log_id_tm_queue, "queue	could not be set by switch\n");
            break;
        }
        p_tm_queue_inst->fsm_state = TM_QUEUE_FSM_STATE_INACTIVE;     
    }while(0);
    return ret;

}
static bcmos_errno bcmbal_tm_queue_create(tm_queue_inst *p_tm_queue_inst, 
										   void *msg, 
										   tm_queue_fsm_event *p_event)
{
    bcmos_errno ret = BCM_ERR_OK;
    do
    {
        p_tm_queue_inst->fsm_state = TM_QUEUE_FSM_STATE_INACTIVE;

        /*referenced tm sched exist for sure as it was already validated*/
        ret = bcmbal_tm_sched_set_queue(p_tm_queue_inst);        
        if(BCM_ERR_OK != ret)
        {
            BCM_LOG(ERROR, log_id_tm_queue, "queue  could not be attched to node \n");
            break;
        }
    }while(0);
    return ret;

}

bcmos_errno bcmbal_tm_queue_set_owner(bcmbal_tm_queue_key key)
{
    bcmos_errno ret = BCM_ERR_OK;
    tm_queue_inst *p_tm_queue_inst;
    do
    {
        p_tm_queue_inst = tm_queue_inst_get(key,TM_QUEUE_FLAG_ACTIVE);
        if(NULL == p_tm_queue_inst)
        {
            BCM_LOG(ERROR, log_id_tm_queue,"ERROR - tm queue not found.  No further processing\n");
            ret = BCM_ERR_NOENT;
            break;
        }
        ret = bcmbal_tm_queue_activate(p_tm_queue_inst);
    }while(0);
    return ret;
}

bcmos_errno bcmbal_tm_queue_unset_owner(bcmbal_tm_queue_key key)
{
    bcmos_errno ret = BCM_ERR_OK;    
    tm_queue_inst *p_tm_queue_inst;
    do
    {
        p_tm_queue_inst = tm_queue_inst_get(key,TM_QUEUE_FLAG_ACTIVE);
        if(NULL == p_tm_queue_inst)
        {
            BCM_LOG(ERROR, log_id_tm_queue,"ERROR - tm queue not found.  No further processing\n");
            ret = BCM_ERR_NOENT;
            break;
        }
        ret = bcmbal_tm_queue_deactivate(p_tm_queue_inst);
    
    }while(0);
    
    return ret;
}

bcmos_errno bcmbal_tm_queue_use_set(tm_queue_inst *p_tm_queue_inst, bcmos_bool is_in_use)
{
    bcmos_errno  ret = BCM_ERR_OK;
    
    BUG_ON(NULL == p_tm_queue_inst);
    if(is_in_use)
    {
        /*queue might already in requested state*/
        p_tm_queue_inst->fsm_state = TM_QUEUE_FSM_STATE_IN_USE;
        p_tm_queue_inst->api_req_tm_queue_info.data.ref_count++;
    }    
    else
    {
        if(p_tm_queue_inst->api_req_tm_queue_info.data.ref_count)
        {
            p_tm_queue_inst->api_req_tm_queue_info.data.ref_count--;
            
            if(0 == p_tm_queue_inst->api_req_tm_queue_info.data.ref_count)
            {
                p_tm_queue_inst->fsm_state = TM_QUEUE_FSM_STATE_ACTIVE;		
            }
        }
        else
        {
            BCM_LOG(ERROR, log_id_tm_queue, "queue (sched.id = %d sched.dir = %s queue.id = %d) could not be unset as no flows assign to it \n", p_tm_queue_inst->api_req_tm_queue_info.key.sched_id, TM_SCHED_DIR_TO_STR(p_tm_queue_inst->api_req_tm_queue_info.key.sched_dir), p_tm_queue_inst->api_req_tm_queue_info.key.id);
            ret = BCM_ERR_INTERNAL;
        }
    }
    return ret;
}

bcmos_errno bcmbal_tm_queue_destroy(tm_queue_inst *p_tm_queue_inst, bcmos_bool remove_from_node)
{
    bcmos_errno ret = BCM_ERR_OK;
    do
    {
        p_tm_queue_inst->fsm_state = TM_QUEUE_FSM_STATE_DELETING;
        if(remove_from_node)
        {
            ret = bcmbal_tm_sched_remove_queue(p_tm_queue_inst);
            if(BCM_ERR_OK != ret)
            {
                BCM_LOG(ERROR, log_id_tm_queue, "queue could not be cleaared from its tm sched\n");
            break;
            }
        }	
        ret = tm_queue_free_by_entry(p_tm_queue_inst);
    }while(0);
    return ret;
}

static bcmos_errno bcmbal_tm_queue_inactive_destroy(tm_queue_inst *p_tm_queue_inst, 
										   void *msg, 
										   tm_queue_fsm_event *p_event)
{
    return bcmbal_tm_queue_destroy(p_tm_queue_inst, BCMOS_TRUE);
}

static bcmos_errno bcmbal_tm_queue_active_destroy(tm_queue_inst *p_tm_queue_inst, 
										   void *msg, 
										   tm_queue_fsm_event *p_event)
{
    bcmos_errno ret = BCM_ERR_OK;
    do
    {
        p_tm_queue_inst->fsm_state = TM_QUEUE_FSM_STATE_DELETING;
        ret = sw_util_tm_queue_clear(p_tm_queue_inst);
        if(BCM_ERR_OK != ret)
        {
            BCM_LOG(ERROR, log_id_tm_queue, "queue could not be cleaared from the switch\n");
            break;
        }
        ret = bcmbal_tm_queue_inactive_destroy(p_tm_queue_inst, msg, p_event);
    }while(0);
    return ret;
}
/*@}*/
