/******************************************************************************
 *
 *  <:copyright-BRCM:2016:DUAL/GPL:standard
 *  
 *     Copyright (c) 2016 Broadcom
 *     All Rights Reserved
 *  
 *  Unless you and Broadcom execute a separate written software license
 *  agreement governing use of this software, this software is licensed
 *  to you under the terms of the GNU General Public License version 2
 *  (the "GPL"), available at http://www.broadcom.com/licenses/GPLv2.php,
 *  with the following added to such license:
 *  
 *     As a special exception, the copyright holders of this software give
 *     you permission to link this software with independent modules, and
 *     to copy and distribute the resulting executable under terms of your
 *     choice, provided that you also meet, for each linked independent
 *     module, the terms and conditions of the license of that module.
 *     An independent module is a module which is not derived from this
 *     software.  The special exception does not apply to any modifications
 *     of the software.
 *  
 *  Not withstanding the above, under no circumstances may you combine
 *  this software in any way with any other Broadcom software provided
 *  under a license other than the GPL, without Broadcom's express prior
 *  written consent.
 *  
 *  :>
 *
 *****************************************************************************/
 
/**
 * @file acc_term_fsm.h
 * @brief Code to support the BAL Access Terminal FSM
 *
 * @defgroup access_terminal Access Terminal
 * @ingroup core
 */


#ifndef ACC_TERM_FSM_H
#define ACC_TERM_FSM_H

/*@{*/

#include <bcm_topo.h>
#include <bal_api.h>
#include <sub_term_fsm.h>
#include <tm_sched_fsm.h>


/** An enumeration of the access-terminal FSM events.
 */
typedef enum
{
  ACC_TERM_FSM_EVENT_TYPE_NONE                 = -1,
  ACC_TERM_FSM_EVENT_TYPE_ADMIN_UP        ,
  ACC_TERM_FSM_EVENT_TYPE_ADMIN_DN        ,
  ACC_TERM_FSM_EVENT_TYPE_INT_ADMIN_UP    ,
  ACC_TERM_FSM_EVENT_TYPE_INT_ADMIN_DN    ,
  ACC_TERM_FSM_EVENT_TYPE_UTIL_MSG        ,
  ACC_TERM_FSM_EVENT_TYPE_UTIL_AUTO_MSG   ,
  ACC_TERM_FSM_EVENT_TYPE_TIMEOUT         ,


  ACC_TERM_FSM_EVENT_TYPE__LAST,
  ACC_TERM_FSM_EVENT_TYPE__NUM_OF
} acc_term_fsm_event_type;


/** An enumeration of the access-terminal FSM states.
 */
typedef enum
{
  ACC_TERM_FSM_STATE_NONE     = -1,
  ACC_TERM_FSM_STATE_NULL         ,
  ACC_TERM_FSM_STATE_ADDING       ,
  ACC_TERM_FSM_STATE_ADDED        ,
  ACC_TERM_FSM_STATE_REMOVING     ,

  ACC_TERM_FSM_STATE__LAST,
  ACC_TERM_FSM_STATE__NUM_OF
} acc_term_fsm_state;

/**
 * A structure that defines the information associated with an
 * access-terminal FSM event
 */
typedef struct acc_term_fsm_event
{
  acc_term_fsm_event_type  event_type; /**< An FSM event */
  void                  *msg;       /**< A pointer to the message being processed by the FSM during an event */
} acc_term_fsm_event;

/**
 * @brief A temporary set of definitions to support access-terminal interfaces.
 * This should be replaced by a more generic mapping method.
 * @note The max PON interface of 16 is a practical limit that access term will support for now. 
 */
#define NUM_SUPPORTED_SUBSCRIBER_INTERFACES     (16)
#define NUM_SUPPORTED_NNI_INTERFACES            (BCM_TOPO_MAX_NNI_PORTS)
#define NUM_SUPPORTED_INTERFACES                (NUM_SUPPORTED_SUBSCRIBER_INTERFACES + NUM_SUPPORTED_NNI_INTERFACES)
/** invalid interface index - used for any error in mapping port type/id to interface index */
#define INVALID_INTERFACE_INDEX                 (NUM_SUPPORTED_INTERFACES)

#define BAL_ACC_TERM_MAX_FLOWS_PER_PON          (1024)
#define BAL_ACC_TERM_MAX_FLOWS                  (NUM_SUPPORTED_SUBSCRIBER_INTERFACES * BAL_ACC_TERM_MAX_FLOWS_PER_PON)

typedef struct sub_term_id_entry
{
    bcmbal_sub_id sub_term_id;
    TAILQ_ENTRY(sub_term_id_entry) next; /**< TAILQ link */    
}sub_term_id_entry;

/**
 * Two interface objects associated with the access-terminal. 
 * One stores the API requests, and one records the current
 * state of the object.
 */
typedef struct acc_term_interface
{
    bcmbal_interface_cfg current_int_obj_info;   /**< The current information for this interface (used for GET) */
    bcmbal_interface_cfg api_req_int_obj_info;   /**< The last interface object info received from the Public API */
    uint16_t             num_sub_terms_on_int;   /**< The number of subscriber terminals on this interface */
    TAILQ_HEAD(sub_term_id_list_head, sub_term_id_entry) sub_term_id_list;
} acc_term_interface;

/**
 * The interface key of the instance being manipulated, and an array of interface objects associated
 * with an access terminal instance.
 */
typedef struct acc_term_interface_info
{ 
    acc_term_interface   interface[NUM_SUPPORTED_INTERFACES]; /**< All interface instances */

}acc_term_interface_info;


/**
 * An structure defining an access terminal instance and its associated interfaces
 */
typedef struct acc_term_inst
{
  /**< The current information for this access-terminal (used for GET)  */
  bcmbal_access_terminal_cfg   current_acc_term_obj_info;
  
  /**< The last access-terminal object info received from the Public API */ 
  bcmbal_access_terminal_cfg   api_req_acc_term_obj_info; 

  acc_term_fsm_state            fsm_state;      /**< The access-terminal FSM state */
  acc_term_interface_info       intf_info;      /**< The access-terminal interfaces */
  bcmos_timer                   timer_info;     /**< A structure used for the state machine timeout timer */
  
} acc_term_inst;


/*--- external function declarations ---*/
extern bcmos_errno process_access_terminal_object(void *msg_payload);

extern bcmos_errno process_access_terminal_util_msg(void *msg_payload);

extern bcmos_errno process_interface_object(void *msg_payload);

extern bcmos_errno process_interface_util_msg(void *msg_payload);

extern bcmos_errno bcmbal_interface_sub_term_list_add(bcmbal_subscriber_terminal_key sub_term_key);

extern bcmos_errno bcmbal_interface_sub_term_list_remove(bcmbal_subscriber_terminal_key sub_term_key);

extern void access_terminal_fsm_init(void);

extern bcmbal_status bcmbal_interface_status_get(bcmbal_interface_key key);

extern bcmbal_status bcmbal_get_intf_oper_status_from_admin_state (bcmbal_state intf_admin_state);

extern uint32_t bcmbal_port_type_and_id_to_interface_index(bcmbal_intf_type intf_type, 
                                                           bcmbal_intf_id intf_id);
extern bcmbal_status acc_term_status_get(void);

extern bcmos_errno bcmbal_interface_sub_term_list_entry_add(bcmbal_subscriber_terminal_key sub_term_key);

extern bcmos_errno bcmbal_interface_sub_term_list_entry_remove(bcmbal_subscriber_terminal_key sub_term_key);

extern bcmos_errno bcmbal_interface_tm_get(bcmbal_interface_key key, bcmbal_tm_sched_id *id);

extern bcmos_errno interface_tm_sched_unset(bcmbal_interface_key intf_key, bcmbal_tm_sched_key sched_key);

extern bcmos_errno bcmbal_tm_sched_interface_tm_auto_create(bcmbal_interface_cfg *p_interface_info);

extern bcmos_errno bcmbal_tm_sched_set_interface_owner(bcmbal_interface_key interface_key, tm_sched_inst *p_tm_sched_inst);

extern bcmos_errno bcmbal_tm_sched_set_sub_term_owner(	bcmbal_tm_sched_key tm_sched_key, const bcmbal_subscriber_terminal_cfg *p_sub_term_cfg);


/*@}*/

#endif /*ACC_TERM_FSM_H */


