/*-
 * ============LICENSE_START=======================================================
 * OSAM Core
 * ================================================================================
 * Copyright (C) 2018 Netsia
 * ================================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ============LICENSE_END=========================================================
 */



package org.onap.osam.external.aai;

import lombok.extern.slf4j.Slf4j;
import org.onap.osam.common.exception.BadFormatException;
import org.onap.osam.common.exception.InvalidOperationException;
import org.onap.osam.common.exception.NotFoundException;
import org.onap.osam.external.aai.config.AaiConfig;
import org.onap.osam.external.aai.exception.ExternalSystemException;
import org.onap.osam.external.aai.model.PNF;
import org.onap.osam.external.aai.util.AaiHeaderUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.client.ClientResponse;

import javax.net.ssl.SSLException;


/**
 * Created by cemturker on 12.04.2018.
 */

@Service
@Slf4j
public class AaiClientImpl implements AaiClient {

    private AaiConfig aaiConfig;

    @Autowired
    public AaiClientImpl(AaiConfig aaiConfig) {
        this.aaiConfig = aaiConfig;
    }

    @Override
    public PNF queryPnf(String name) {
        try {
            ClientResponse response = AaiWebClient.webClient(aaiConfig).build().queryPNF(name);
            commonErrorCheck(name,response);
            log.info("Pnf query response code {} for {} id",response.statusCode(),name);
            if (response.statusCode().is2xxSuccessful()) {
                return AaiHeaderUtil.convertToPnf(response.bodyToMono(String.class).block());
            }
        } catch (SSLException e) {
            log.error("",e);
        }
        throw new InvalidOperationException("");
    }

    @Override
    public void putPnf(PNF pnf) {
        try {
            ClientResponse response = AaiWebClient.webClient(aaiConfig).build().putPNF(pnf);
            commonErrorCheck(pnf,response);
            log.info("Pnf query response code {} for {} ",response.statusCode(),pnf);
            if (response.statusCode().is2xxSuccessful()) {
                return;
            }
        } catch (SSLException e) {
            log.error("",e);
        }
        throw new InvalidOperationException("");
    }

    private <T> void commonErrorCheck(T t, ClientResponse response){
        if (response.statusCode().is5xxServerError()) {
            throw new ExternalSystemException("Aai error code:"+response.statusCode().value());
        }
        if (HttpStatus.BAD_REQUEST.equals(response.statusCode())) {
            throw new BadFormatException("Bad format exception is received from AAI");
        }
        if (HttpStatus.NOT_FOUND.equals(response.statusCode())) {
            throw new NotFoundException(t+" is not found in AAI");
        }
    }
}
