/*-
 * ============LICENSE_START=======================================================
 * OSAM Core
 * ================================================================================
 * Copyright (C) 2018 Netsia
 * ================================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ============LICENSE_END=========================================================
 */

package org.onap.osam.core;

import junitparams.JUnitParamsRunner;
import junitparams.Parameters;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.onap.osam.api.service.AccessPodService;
import org.onap.osam.api.service.DeviceService;
import org.onap.osam.common.exception.AbstractOLTException;
import org.onap.osam.common.exception.InvalidOperationException;
import org.onap.osam.common.exception.NotFoundException;
import org.onap.osam.external.grpc.AbstractOLTClient;
import org.onap.osam.model.dao.*;
import org.onap.osam.model.repository.ChassisRepository;
import org.onap.osam.model.repository.OLTPortRepository;
import org.onap.osam.model.repository.OLTSlotRepository;
import org.onap.osam.model.repository.ONTDeviceRepository;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatThrownBy;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

/**
 * Created by Zafer Kaban on 26.11.2018.
 */

@RunWith(JUnitParamsRunner.class)
public class DeviceServiceImplTest {

    private static String TEST_PNF_ID = "TEST_PNF_ID";
    private static String TEST_CLLI   = "TEST_CLLI";
    private static String TEST_SERIAL = "SERIAL_NUMBER";

    @Mock
    private ChassisRepository chassisRepository;
    @Mock
    private OLTPortRepository oltPortRepository;
    @Mock
    private OLTSlotRepository oltSlotRepository;
    @Mock
    private ONTDeviceRepository ontDeviceRepository;

    @Mock
    private AbstractOLTClient abstractOLTClient;

    @Mock
    private AccessPodService accessPodService;

    @Captor
    private ArgumentCaptor<ONTDevice> ontDeviceCaptor;

    @Captor
    private ArgumentCaptor<OLTPort> portCaptor;


    private Chassis chassis;

    private AccessPod accessPod;

    private OLTSlot oltSlot;

    private OLTPort oltPort;

    private ONTDevice ontDevice;

    @InjectMocks
    private DeviceServiceImpl deviceService;

    @Before
    public void initMocks() {
        MockitoAnnotations.initMocks(this);
        accessPod = new AccessPod();
        accessPod.setPnfId(TEST_PNF_ID);
        chassis = new Chassis();
        chassis.setClli(TEST_CLLI);
        chassis.setAccessPod(accessPod);
        chassis.setId(1L);
        oltSlot = new OLTSlot();
        oltSlot.setId(1L);
        oltSlot.setNumber(1);
        oltPort = new OLTPort();
        oltPort.setId(1L);
        oltPort.setPortNumber(1);
        ontDevice = new ONTDevice();
        ontDevice.setId(1L);
        ontDevice.setOLTPort(oltPort);
        oltPort.setOltSlot(oltSlot);
        oltSlot.setChassis(chassis);

        when(chassisRepository.findByClli(TEST_CLLI)).thenReturn(Optional.ofNullable(chassis));
        when(chassisRepository.findById(1L)).thenReturn(Optional.ofNullable(chassis));
    }

    @Test
    public void whenAddChassis_sunnyFlow(){

        // TEST Sunshine scenario
        when(accessPodService.findByPnfId(TEST_PNF_ID)).thenReturn(accessPod);
        when(chassisRepository.save(chassis)).thenReturn(chassis);

        Chassis chassisResult = deviceService.addChassis(chassis);
        assertThat(chassisResult).isSameAs(chassis);
    }


    @Test
    public void whenAddChassisPnfNotFound_shouldThrowException() {
        // TEST when PNF registration does not exist so that Access POD does not exist in OSAM DB

        when(accessPodService.findByPnfId(TEST_PNF_ID)).thenThrow(NotFoundException.class);
        assertThatThrownBy(() -> deviceService.addChassis(chassis)).isInstanceOf(NotFoundException.class);
        //verify we save nothing to DB
        verifyZeroInteractions(chassisRepository);
    }

    @Test
    public void whenAddChassisAbstractOltReturnsNull_shouldThrowException() {
        // TEST grpc failure case

        when(accessPodService.findByPnfId(TEST_PNF_ID)).thenReturn(accessPod);
        doThrow(AbstractOLTException.class).when(abstractOLTClient).createChassis(chassis);
        assertThatThrownBy(() -> deviceService.addChassis(chassis)).isInstanceOf(AbstractOLTException.class);
        //verify we save nothing to DB
        verifyZeroInteractions(chassisRepository);
    }


    @Test
    public void whenDeleteChassisById_sunnyFlow() {
        deviceService.deleteChassis(1L);
        verify(chassisRepository, times(1)).deleteById(1L);
    }

    @Test
    public void whenDeleteChassisByClli_sunnyFlow () {
        deviceService.deleteChassisByClli(TEST_CLLI);
        //Test chassis has clli TEST_CLLI and id 1L, thus the verify
        verify(chassisRepository, times(1)).deleteById(1L);
    }



    @Test
    public void whenGetChassisById_sunnyFlow(){
        Chassis testChassis = deviceService.getChassisById(1L);
        assertThat(testChassis).isSameAs(chassis);
    }

    @Test
    public void whenGetChassisByNotExistingId_throwNotFoundException(){
        assertThatThrownBy(() -> deviceService.getChassisById(100L)).isInstanceOf(NotFoundException.class);
    }

    @Test
    public void whenGetChassisByTestClli_sunnyFlow(){
        Chassis testChassis = deviceService.getChassisByClli(TEST_CLLI);
        assertThat(testChassis).isSameAs(chassis);
    }

    @Test
    public void whenGetChassisByNotExistingClli_throwNotFoundException(){
        assertThatThrownBy(() -> deviceService.getChassisByClli("SOME_FAKE_CLLI")).isInstanceOf(NotFoundException.class);
    }

    @Test
    public void whenCountChassis_sunnyFlow(){
        when(chassisRepository.count()).thenReturn(1L);
        long count = deviceService.getChassisCount();
        assertThat(count).isEqualTo(1L);
    }

    @Test
    public void whenGetByPnfId_sunnyFlow(){
        when(chassisRepository.findByAccessPodPnfId(TEST_PNF_ID)).thenReturn(Optional.of(new ArrayList<Chassis>()));
        ArrayList<Chassis> chassisResult = (ArrayList<Chassis>) deviceService.getByPnfId(TEST_PNF_ID);
        assertThat(chassisResult).isNotNull();
    }


    @Test
    public void whenGetByPnfIdNotExisting_shouldThrowException(){
        assertThatThrownBy(() -> deviceService.getByPnfId("SOME_FAKE_PNFID")).isInstanceOf(NotFoundException.class);
    }


    @Test
    public void whenGetAllChassis_sunnyFlow() {
        when(chassisRepository.findAll()).thenReturn(new ArrayList<Chassis>());
        ArrayList<Chassis> chassisResult = (ArrayList<Chassis>) deviceService.getAllChassis();
        assertThat(chassisResult).isNotNull();
    }


    @Test
    public void whenAddOLTSlot_sunnyFlow() {

        Set<OLTSlot> oltSlots = new HashSet<OLTSlot>();
        chassis.setOltSlots(oltSlots);
        when(oltSlotRepository.save(oltSlot)).thenReturn(oltSlot);
        when(abstractOLTClient.createOLTChassis(oltSlot)).thenReturn(TEST_CLLI);

        OLTSlot oltResult = deviceService.addOLTSlot(oltSlot, chassis);

        //verify creation of 16 ports
        verify(oltPortRepository, times(16)).save(portCaptor.capture());
        final List<OLTPort> allOltPortsValues = portCaptor.getAllValues();
        allOltPortsValues.forEach(x -> {
            assertThat(x.getOltSlot()).isSameAs(oltSlot);
            assertThat(x.getAdminState()).isEqualTo(AdminState.ENABLED);
            assertThat(x.getPortAuthState()).isEqualTo(ActivityState.ACTIVE);
        });

        //verify added to chassis
        assertThat(chassis.getOltSlots()).hasSize(1);

        //verify oltSlot logic
        assertThat(oltResult).isSameAs(oltSlot);
        assertThat(oltResult.getAdminState()).isEqualTo(AdminState.ENABLED);
        assertThat(oltResult.getOperationalState()).isEqualTo(ActivityState.ACTIVE);
        assertThat(oltResult.getPortAuthState()).isEqualTo(ActivityState.ACTIVE);

    }

    public void whenAddOLTSlotTooManySlotsOnChassis_shouldThrowException() {
        //already add 16 slots, cannot add another one
        Set<OLTSlot> oltSlots = new HashSet<OLTSlot>();
        for (int i = 0; i < 16; i++) {
            oltSlots.add(new OLTSlot());
        }
        chassis.setOltSlots(oltSlots);
        assertThatThrownBy(()-> deviceService.addOLTSlot(oltSlot, chassis)).isInstanceOf(InvalidOperationException.class);
        //verify no DB interactions
        verifyZeroInteractions(oltSlotRepository, oltPortRepository, chassisRepository);
    }

    public void whenAddOLTSlotAbstractOLTReturnsNull_shouldThrowException() {
        when(abstractOLTClient.createOLTChassis(oltSlot)).thenReturn(null);
        assertThatThrownBy(()-> deviceService.addOLTSlot(oltSlot, chassis)).isInstanceOf(AbstractOLTException.class);
        //verify no DB interactions
        verifyZeroInteractions(oltSlotRepository, oltPortRepository, chassisRepository);
    }

    @Test
    public void whenDeleteOLTSlot_repositoryMethodCalledOnce() {
        deviceService.deleteOLTSlot(1L);
        verify(oltSlotRepository, times(1)).deleteById(1L);
    }


    @Test
    public void whenGetOLTSlotById_sunnyFlow(){
        when(oltSlotRepository.findById(1L)).thenReturn(Optional.of(oltSlot));
        OLTSlot oltActualSlot = deviceService.getOLTSlotById(1L);
        assertThat(oltActualSlot).isSameAs(oltSlot);
    }

    @Test
    public void whenGetOLTSlotByIdNotExisting_shouldThrowException(){
        assertThatThrownBy(()-> deviceService.getOLTSlotById(100L)).isInstanceOf(NotFoundException.class);
    }

    @Test
    public void whenGetOLTSlotBySerialId_sunnyFlow() {
        when(oltSlotRepository.findBySerialNumber(TEST_SERIAL)).thenReturn(Optional.of(oltSlot));
        OLTSlot oltActualSlot = deviceService.getOLTSlotBySerialNumber(TEST_SERIAL);
        assertThat(oltActualSlot).isSameAs(oltSlot);
    }

    @Test
    public void whenGetOLTSlotBySerialIdNotExisting_shouldThrowException(){
        assertThatThrownBy(()-> deviceService.getOLTSlotBySerialNumber("SOME_FAKE_SERIAL")).isInstanceOf(NotFoundException.class);
    }

    @Test
    public void whenGetAllOLTSlot_sunnyFlow() {
        final ArrayList<OLTSlot> slotArrayList = new ArrayList<>();
        when(oltSlotRepository.findAll()).thenReturn(slotArrayList);
        ArrayList<OLTSlot> oltSlots = (ArrayList<OLTSlot>) deviceService.getAllOLTSlots();
        assertThat(oltSlots).isEqualTo(slotArrayList);
    }

    @Test
    public void whenDeleteOLTPort_sunnyFlow() {
        deviceService.deleteOLTPort(1L);
        verify(oltPortRepository, times(1)).deleteById(1L);
    }

    @Test
    public void whenGetOLTPort_sunnyFlow() {
        when(oltPortRepository.findById(1L)).thenReturn(Optional.of(oltPort));
        final OLTPort actualPort = deviceService.getOLTPortById(1L);
        assertThat(actualPort).isSameAs(oltPort);
    }

    @Test
    public void whenGetOLTPortByIdNotExisting_shouldThrowException(){
        assertThatThrownBy(()-> deviceService.getOLTPortById(100L)).isInstanceOf(NotFoundException.class);
    }

    @Test
    @Parameters(method = "provTypeToTestForOnt")
    public void whenProvisionONTDevice_sunnyFlow(DeviceService.OntProvisioningType provType) {


        //setting up ontDevice
        ontDevice.setSerialNumber("SOME_SERIAL_NUMBER");
        ontDevice.setNumber(23);
        ontDevice.setCTag(111);
        ontDevice.setSTag(222);
        ontDevice.setCircuitId("CIRCUIT_ID");
        ontDevice.setNasPortId("NAS_PORT_ID");
        ontDevice.setAdminState(AdminState.ENABLED);
        ontDevice.setOperationalState(ActivityState.ACTIVE);
        ontDevice.setPortAuthState(ActivityState.ACTIVE);

        when(oltPortRepository.findByPortNumberAndOltSlot_NumberAndOltSlot_ChassisClli(1,1,TEST_CLLI)).thenReturn(Optional.ofNullable(oltPort));
        when(abstractOLTClient.provisionONT(ontDevice)).thenReturn(true);
        when(abstractOLTClient.preProvisionOnt(ontDevice)).thenReturn(true);
        when(abstractOLTClient.provisionOntFull(ontDevice)).thenReturn(true);

        //This is because in order to be inserted to hashset of oltSlots at the next command of tested function,
        //the ONTDevice has to have ID
        when(ontDeviceRepository.save(any())).thenReturn(ontDevice);

       deviceService.provisionONTDevice(ontDevice, provType);

       //checking that the copy to persisted object was as expected
        verify(ontDeviceRepository, times(1)).save(ontDeviceCaptor.capture());
        final ONTDevice capturedONTDevice = ontDeviceCaptor.getValue();

        //TODO Pavel fix after discussion with Netsia regarding the returned object.
        //Currently the assert will fail because internal object has no ID.
        //I didn't want to use isEqualUsingSpecificFields, to catch set operations we might miss in the future
        //assertThat(capturedONTDevice).isEqualToComparingFieldByField(ontDevice);

        verify(oltPortRepository, times(1)).save(oltPort);
    }

    private Object[] provTypeToTestForOnt() {
        return new Object[] {
                new Object[] { DeviceService.OntProvisioningType.PROVISION },
                new Object[] { DeviceService.OntProvisioningType.PREPROVISION },
                new Object[] { DeviceService.OntProvisioningType.FULL }
        };
    }

    @Test
    @Parameters(method = "provTypeToTestForOnt")
    public void whenAddONTDeviceNoPortFound_shouldThrowException(DeviceService.OntProvisioningType provType) {
        when(oltPortRepository.findByPortNumberAndOltSlot_NumberAndOltSlot_ChassisClli(1,1,TEST_CLLI)).thenReturn(Optional.ofNullable(null));
        assertThatThrownBy(()-> deviceService.provisionONTDevice(ontDevice, provType)).isInstanceOf(NotFoundException.class);

    }

    @Test
    @Parameters(method = "provTypeToTestForOnt")
    public void whenAddONTDeviceAbstractOLTError_shouldThrowException(DeviceService.OntProvisioningType provType) {
        when(oltPortRepository.findByPortNumberAndOltSlot_NumberAndOltSlot_ChassisClli(1,1,TEST_CLLI)).thenReturn(Optional.ofNullable(oltPort));
        when(abstractOLTClient.provisionONT(ontDevice)).thenReturn(false);
        when(abstractOLTClient.preProvisionOnt(ontDevice)).thenReturn(false);
        when(abstractOLTClient.provisionOntFull(ontDevice)).thenReturn(false);
        assertThatThrownBy(()-> deviceService.provisionONTDevice(ontDevice, provType)).isInstanceOf(AbstractOLTException.class);

    }

    @Test
    public void whenDeleteONTDevice_sunnyFlow() {
        deviceService.deleteONTDevice(1L);
        verify(ontDeviceRepository, times(1)).deleteById(1L);
    }

    @Test
    public void whenGetONTDeviceById_sunnyFlow() {
        when(ontDeviceRepository.findById(1L)).thenReturn(Optional.of(ontDevice));
        ONTDevice actualOntDevice = deviceService.getONTDeviceById(1L);
        assertThat(actualOntDevice).isSameAs(ontDevice);
    }

    @Test
    public void whenGetONTDeviceNoId_shouldThrowException() {
        assertThatThrownBy(()-> deviceService.getONTDeviceById(100L)).isInstanceOf(NotFoundException.class);

    }

    @Test
    public void whenGetONTDeviceBySerialNumber_sunnyFlow() {
        when(ontDeviceRepository.findBySerialNumber(TEST_SERIAL)).thenReturn(Optional.of(ontDevice));
        ONTDevice actualOntDevice = deviceService.getONTDeviceBySerialNumber(TEST_SERIAL);
        assertThat(actualOntDevice).isSameAs(ontDevice);
    }

    @Test
    public void whenGetONTDeviceNoSerialNumber_shouldThrowException() {
        assertThatThrownBy(()-> deviceService.getONTDeviceBySerialNumber(TEST_SERIAL)).isInstanceOf(NotFoundException.class);

    }

    @Test
    public void whenGetAllONTDevices_sunnyFlow() {
        final ArrayList<ONTDevice> devices = new ArrayList<>();
        devices.add(ontDevice);
        when(ontDeviceRepository.findAll()).thenReturn(devices);
        ArrayList<ONTDevice> ontDevices = (ArrayList<ONTDevice>) deviceService.getAllONTDevices();
        assertThat(ontDevices).isEqualTo(devices);
    }
}
