/*-
 * ============LICENSE_START=======================================================
 * OSAM
 * ================================================================================
 * Copyright (C) 2018 AT&T
 * ================================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ============LICENSE_END=========================================================
 */
package org.onap.osam.controllers;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.onap.osam.common.dto.ChassisDTO;
import org.onap.osam.common.dto.OLTChassisDTO;
import org.onap.osam.common.dto.ONTDTO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.client.RestTemplate;

import java.util.List;
import java.util.Map;

// TODO log
@RestController
@RequestMapping("device/chassis")
public class OnapEnablerDeviceController extends OnapEnablerController {

	private static final String DEVICE_CHASSIS = "device/chassis";

	@Autowired
	public OnapEnablerDeviceController(RestTemplate restTemplate){
		this.restTemplate = restTemplate;
	}

    @ApiOperation(value = "Register a chassis on top of a POD",
            response = ChassisDTO.class)
	@RequestMapping(method = RequestMethod.POST)
	public ResponseEntity<ChassisDTO> postChassis(@RequestBody ChassisDTO chassisDTO) {
		return restTemplate.postForEntity(buildRequestPath(DEVICE_CHASSIS), new HttpEntity<>(chassisDTO), ChassisDTO.class);
	}

    @ApiOperation(value = "Unregister a chassis from a POD")
	@RequestMapping(method = RequestMethod.DELETE, value = "{clli}")
	public void deleteChassisByClli(@PathVariable String clli) {
		restTemplate.delete(buildRequestPath(DEVICE_CHASSIS, clli));
	}

    @ApiOperation(value = "Register OLT-specific chassis on top of a chassis",
            response = OLTChassisDTO.class)
	@RequestMapping(method = RequestMethod.POST, value = "olt")
	public ResponseEntity<OLTChassisDTO> postOLTChassis(@RequestBody OLTChassisDTO oltChassisDTO) {
		return restTemplate.postForEntity(buildRequestPath(DEVICE_CHASSIS, "olt"), new HttpEntity<>(oltChassisDTO), OLTChassisDTO.class);
	}

    @ApiOperation(value = "Register an ONT device on top of a OLT-specific chassis",
            response = ONTDTO.class)
	@RequestMapping(method = RequestMethod.POST, value = "olt/ont")
	public ResponseEntity<ONTDTO> postONTDevice(@RequestBody ONTDTO ontDTO) {
		return restTemplate.postForEntity(buildRequestPath(DEVICE_CHASSIS, "olt/ont"), new HttpEntity<>(ontDTO), ONTDTO.class);
	}

    @ApiOperation(value = "Get all chassis entities from all PODs",
            response = ChassisDTO.class,
    responseContainer = "List")
	@RequestMapping(method = RequestMethod.GET)
	public ResponseEntity<List<ChassisDTO>> getAllChassis() {
		return restTemplate.exchange(buildRequestPath(DEVICE_CHASSIS), HttpMethod.GET, HttpEntity.EMPTY, new ParameterizedTypeReference<List<ChassisDTO>>(){});
	}


    @ApiOperation(value = "Get all OLT-specific chassis entities from all PODs, grouped by POD pnfId",
            response = OLTChassisDTO.class,
            responseContainer = "Map")
	@RequestMapping(method = RequestMethod.GET, value = "olt")
	public ResponseEntity<Map<String, List<OLTChassisDTO>>> getAllOLTDevices(){
		return restTemplate.exchange(buildRequestPath(DEVICE_CHASSIS, "olt"), HttpMethod.GET, HttpEntity.EMPTY, new ParameterizedTypeReference<Map<String, List<OLTChassisDTO>>>(){});
	}

    @ApiOperation(value = "Get all ONT devices from all PODs, grouped by POD pnfId",
            response = ONTDTO.class,
            responseContainer = "Map")
	@RequestMapping(method = RequestMethod.GET, value = "olt/ont")
	public ResponseEntity<Map<String, List<ONTDTO>>> getAllONTDevices(){
		return restTemplate.exchange(buildRequestPath(DEVICE_CHASSIS, "olt/ont"), HttpMethod.GET, HttpEntity.EMPTY, new ParameterizedTypeReference<Map<String, List<ONTDTO>>>(){});
	}
}
