/*-
 * ============LICENSE_START=======================================================
 * OSAM
 * ================================================================================
 * Copyright (C) 2018 AT&T
 * ================================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ============LICENSE_END=========================================================
 */
package org.onap.osam.job.impl;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.hamcrest.Matcher;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.onap.osam.job.dao.job.JobStatus;
import org.onap.osam.job.dao.job.OsamJob;
import org.onap.osam.job.IJobFactory;
import org.onap.osam.job.IJobCommand;
import org.onap.osam.job.JobType;
import org.onap.osam.job.NextCommand;
import org.onap.osam.job.command.HttpCallCommand;
import org.onap.osam.job.command.JobCommandFactory;
import org.onap.osam.job.impl.JobFactory;
import org.onap.osam.job.impl.JobWorker;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import java.util.Map;
import java.util.UUID;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.*;
import static org.hamcrest.core.Is.is;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class JobWorkerTest {


    @Mock
    private JobCommandFactory jobCommandFactory;

    @InjectMocks
    private JobWorker jobWorker = new JobWorker();

    private final IJobCommand jobCommand = mock(IJobCommand.class);
    private OsamJob jobUnderTest;
    private IJobFactory.AsyncJobRequest originalData;
    private JobType originalType;

    @BeforeMethod
    public void initMocks() {
        MockitoAnnotations.initMocks(this);

        when(jobCommandFactory.toCommand(any())).thenReturn(jobCommand);

        originalData = new IJobFactory.AsyncJobRequest() {
            public final Map datum = ImmutableMap.of("some", "data");
            public final String foobar = "aux";
        };

        originalType = JobType.HttpCall;
        jobUnderTest = new JobFactory().createRootJob(
                originalType,
                originalData,
                "my user id",
                RandomUtils.nextInt(),
                ImmutableMap.of()
        );
    }

    @Test
    public void executeJobAndStepToNext_givenNull_onlyStatusModified() {

        assertNextJobAfterExecuteJob(null, new String[]{"status"}, allOf(
                hasProperty("status", is(JobStatus.STOPPED)),
                hasProperty("sharedData", hasProperty("request", is(originalData))),
                hasProperty("type", is(originalType)))
        );
    }

    @Test
    public void executeJobAndStepToNext_givenNextJob_jobDataIsModified() {

        final JobStatus nextStatus = JobStatus.IN_PROGRESS;

        final UUID jobUuid = UUID.randomUUID();
        final NextCommand nextCommand = new NextCommand(nextStatus, new HttpCallCommand("my strange url", jobUuid));

        String[] excludedFields = {"status", "data", "type"};

        assertNextJobAfterExecuteJob(nextCommand, excludedFields, allOf(
                hasProperty("status", is(nextStatus)),
                hasProperty("dataMap", is(nextCommand.getCommand().getData())),
                hasProperty("type", is(nextCommand.getCommand().getType())))
        );
    }

    private void assertNextJobAfterExecuteJob(NextCommand nextCommand, String[] excludedFields, Matcher<OsamJob> jobMatcher) {
        when(jobCommand.call()).thenReturn(nextCommand);

        String jobBefore = new ReflectionToStringBuilder(jobUnderTest, ToStringStyle.SHORT_PREFIX_STYLE).setExcludeFieldNames(excludedFields).toString();

        ////// FUNCTION UNDER TEST /////
        OsamJob nextJob = jobWorker.executeJobAndGetNext(jobUnderTest);
        ////////////////////////////////

        String jobAfter = new ReflectionToStringBuilder(nextJob, ToStringStyle.SHORT_PREFIX_STYLE).setExcludeFieldNames(excludedFields).toString();

        assertThat(nextJob, jobMatcher);
        assertThat(jobAfter, equalTo(jobBefore));
    }
}
