/*-
 * ============LICENSE_START=======================================================
 * OSAM
 * ================================================================================
 * Copyright (C) 2018 AT&T
 * ================================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ============LICENSE_END=========================================================
 */
package org.onap.osam.job.impl;

import com.google.common.collect.ImmutableMap;
import lombok.extern.slf4j.Slf4j;
import org.onap.osam.common.exception.GenericUncheckedException;
import org.onap.osam.job.dao.job.JobStatus;
import org.onap.osam.job.IJobsDataAccessService;
import org.onap.osam.job.command.JobCommandFactory;
import org.quartz.JobBuilder;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SimpleTrigger;
import org.quartz.TriggerBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.quartz.SchedulerFactoryBean;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;

import static org.quartz.SimpleScheduleBuilder.simpleSchedule;

@Slf4j
@Component
public class JobSchedulerInitializer {

    private IJobsDataAccessService jobsDataAccessService;
    private SchedulerFactoryBean schedulerFactoryBean;
    private JobCommandFactory jobCommandFactory;

    @Autowired
    public JobSchedulerInitializer(
            IJobsDataAccessService jobsDataAccessService,
            SchedulerFactoryBean schedulerFactoryBean,
            JobCommandFactory JobCommandFactory
    ) {
        this.jobsDataAccessService = jobsDataAccessService;
        this.schedulerFactoryBean = schedulerFactoryBean;
        this.jobCommandFactory = JobCommandFactory;

    }

    @PostConstruct
    public void init() {
        scheduleJobWorker(JobStatus.PENDING, 1);
        scheduleJobWorker(JobStatus.CREATING, 1);
        scheduleJobWorker(JobStatus.IN_PROGRESS, 1);
        scheduleJobWorker(JobStatus.RESOURCE_IN_PROGRESS, 1);
    }

    private void scheduleJobWorker(JobStatus topic, int intervalInSeconds) {
        final Scheduler scheduler = schedulerFactoryBean.getScheduler();
        JobDetail jobDetail = JobBuilder.newJob().ofType(JobWorker.class)
                .withIdentity("AsyncWorkersJob" + topic)
                .withDescription("Job that run async worker for " + topic)
                .setJobData(new JobDataMap(ImmutableMap.of(
                        "jobsDataAccessService", jobsDataAccessService,
                        "jobCommandFactory", jobCommandFactory,
                        "topic", topic
                )))
                .build();
        SimpleTrigger asyncWorkerTrigger = TriggerBuilder.newTrigger().forJob(jobDetail)
                .withIdentity("AsyncWorkersTrigger" + topic)
                .withDescription("Trigger to run async worker for " + topic)
                .withSchedule(simpleSchedule().repeatForever().withIntervalInSeconds(intervalInSeconds))
                .build();
        try {
            scheduler.scheduleJob(jobDetail, asyncWorkerTrigger);
        } catch (SchedulerException e) {
            log.error("Failed to schedule trigger for async worker jobs: {}", e.getMessage());
            throw new GenericUncheckedException(e);
        }
    }
}
