/*-
 * ============LICENSE_START=======================================================
 * OSAM
 * ================================================================================
 * Copyright (C) 2018 AT&T
 * ================================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ============LICENSE_END=========================================================
 */
package org.onap.osam.job.dao.job;


import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.MoreObjects;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.annotations.DynamicUpdate;
import org.hibernate.annotations.SelectBeforeUpdate;
import org.hibernate.annotations.Type;
import org.onap.osam.job.JobType;
import org.onap.osam.job.impl.JobData;
import org.onap.osam.job.impl.JobSharedData;
import org.onap.osam.model.dao.BaseEntity;
import org.springframework.data.annotation.CreatedDate;
import org.springframework.data.annotation.LastModifiedDate;

import javax.persistence.AttributeConverter;
import javax.persistence.Column;
import javax.persistence.Converter;
import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.Lob;
import java.io.IOException;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

/*
 The following 2 annotations let hibernate to update only fields that actually have been changed.
 DynamicUpdate tell hibernate to update only dirty fields.
 SelectBeforeUpdate is needed since during update the entity is detached (get and update are in different sessions)
*/
@DynamicUpdate()
@SelectBeforeUpdate()
@Entity
@Getter
@Setter
@Slf4j
public class OsamJob extends BaseEntity {

    @Column(unique = true, nullable = false, columnDefinition = "CHAR(36)")
    @Type(type="org.hibernate.type.UUIDCharType")
    private UUID uuid;

    @CreatedDate
    private Date createdDate;

    @LastModifiedDate
    private Date modifiedDate;

    @Enumerated(value = EnumType.STRING)
    private JobStatus status;

    @Enumerated(value = EnumType.STRING)
    private JobType type;

    @Lob
    @Column
    private JobData data = new JobData();

    @Column
    private String takenBy;

    @Column
    private String userId;

    @Column(nullable = false)
    private Integer age = 0;

    @Column(nullable = false)
    private Integer indexInBulk = 0;

    @Column
    private Date deletedAt;

    public Map<String, Object> getDataMap() {
        return data.getCommandData().get(getType());
    }

    public void setTypeAndData(JobType jobType, Map<String, Object> data) {
        // *add* the data to map,
        // then change state to given type
        this.type = jobType;
        this.data.getCommandData().put(jobType, data);
    }

    public void setSharedData(JobSharedData sharedData) {
        this.data.setSharedData(sharedData);
    }

    public JobSharedData getSharedData(){
        return this.data.getSharedData();
    };

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof OsamJob)) return false;
        OsamJob daoJob = (OsamJob) o;
        return Objects.equals(getUuid(), daoJob.getUuid());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getUuid());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("status", status)
                .add("type", type)
                .add("uuid", uuid)
                .add("takenBy", takenBy)
                .add("userId", userId)
                .add("age", age)
                .add("createdDate", createdDate)
                .add("modifiedDate", modifiedDate)
                .add("deletedAt", deletedAt)
                .add("data", data)
                .toString();
    }

    @Converter(autoApply = true)
    public static class JobDataConverter implements AttributeConverter<JobData, String> {

        @Override
        public String convertToDatabaseColumn(JobData jobData) {
            if( jobData == null )
                return null;

            ObjectMapper mapper = new ObjectMapper();

            try {
                return mapper.writeValueAsString(jobData);
            } catch (JsonProcessingException e) {
                log.error("Couldn't persist JobData object {}, error: {}. Persisting null", jobData, e);
                return null;
            }
        }

        @Override
        public JobData convertToEntityAttribute(String s) {
            if( s == null )
                return null;

            ObjectMapper mapper = new ObjectMapper();

            try {
                return mapper.readValue(s, JobData.class);
            } catch (IOException e) {
                log.error("Couldn't deserialize {} to JobData object, error: {}. Returning null", s, e);
                return null;
            }
        }
    }
}
