/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.fpcagent.workers;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicResponseHandler;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.HttpVersion;
import org.json.JSONObject;
import org.onosproject.fpcagent.protocols.DpnNgicCommunicator;
import org.onosproject.restconf.utils.RestconfUtils;
import org.onosproject.yang.gen.v1.ietfdmmfpcagent.rev20160803.ietfdmmfpcagent.DefaultYangAutoPrefixNotify;
import org.onosproject.yang.gen.v1.ietfdmmfpcagent.rev20160803.ietfdmmfpcagent.yangautoprefixnotify.value.DownlinkDataNotification;
import org.onosproject.yang.model.DefaultModelObjectData;
import org.onosproject.yang.model.ModelConverter;
import org.onosproject.yang.model.ResourceData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;

import static org.onosproject.fpcagent.util.FpcUtil.module;

public class HTTPNotifier implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(HTTPNotifier.class);
    private static HTTPNotifier _instance = null;
    private final BlockingQueue<Map.Entry<String, DefaultYangAutoPrefixNotify>> blockingQueue;
    public ModelConverter modelConverter = null;
    private boolean run;
    private ResponseHandler<String> handler = new BasicResponseHandler();

    protected HTTPNotifier(ModelConverter modelConverter) {
        this.run = true;
        this.blockingQueue = new LinkedBlockingQueue<>();
        this.modelConverter = modelConverter;
    }

    public static HTTPNotifier createInstance(ModelConverter modelConverter) {
        if (_instance == null) {
            _instance = new HTTPNotifier(modelConverter);
        }
        return _instance;
    }

    public static HTTPNotifier getInstance() {
        return _instance;
    }

    public void send(Map.Entry<String, DefaultYangAutoPrefixNotify> buf) {
        try {
            blockingQueue.put(buf);
        } catch (InterruptedException e) {
            log.error(ExceptionUtils.getFullStackTrace(e));
        }
    }


    public void open() {
        ExecutorService executorService = Executors.newSingleThreadExecutor();
        executorService.submit(() -> {
            while ((!Thread.currentThread().isInterrupted()) && run) {
                try {
                    Map.Entry<String, DefaultYangAutoPrefixNotify> entry = blockingQueue.take();
                    DefaultYangAutoPrefixNotify notify = entry.getValue();

                    CloseableHttpClient client = HttpClients.createDefault();
                    HttpPost httpPost = new HttpPost(entry.getKey());
                    httpPost.addHeader("User-Agent", "ONOS Notification Agent");
                    httpPost.addHeader("Charset", "utf-8");
                    httpPost.addHeader("Content-type", "application/json");
                    httpPost.setProtocolVersion(HttpVersion.HTTP_1_1);

                    ResourceData dataNode = modelConverter.createDataNode(
                            DefaultModelObjectData.builder()
                                    .addModelObject(notify)
                                    .build()
                    );
                    ObjectNode jsonNodes = RestconfUtils.convertDataNodeToJson(module, dataNode.dataNodes().get(0));
                    ObjectMapper mapper = new ObjectMapper();

                    StringEntity params = new StringEntity(mapper.writeValueAsString(jsonNodes));
                    httpPost.setEntity(params);

                    log.info("Sending HTTP POST {}", httpPost);
                    HttpResponse response = client.execute(httpPost);

                    if (notify.value() instanceof DownlinkDataNotification) {
                        String msg = handler.handleResponse(response);
                        JSONObject json_body = new JSONObject(msg);
                        DpnNgicCommunicator.send_ddn_ack(json_body);
                        log.info("Response {}", response);
                    }
                } catch (Exception e) {
                    log.error(ExceptionUtils.getFullStackTrace(e));
                }
            }
        });
    }

    @Override
    public void close() {
        run = false;
    }
}
