/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opencord.ce.api.models.codecs;

import com.fasterxml.jackson.databind.node.ObjectNode;
import org.onlab.util.Bandwidth;
import org.onosproject.codec.CodecContext;
import org.onosproject.codec.JsonCodec;
import org.opencord.ce.api.models.CarrierEthernetBandwidthProfile;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Codec for {@link CarrierEthernetBandwidthProfile}.
 */
public class CarrierEthernetBwProfileCodec extends JsonCodec<CarrierEthernetBandwidthProfile> {
    private static final String ID = "bwpId";
    private static final String CFG_ID = "cfgId";
    private static final String TYPE = "bwpType";
    private static final String CIR = "cir";
    private static final String CBS = "cbs";
    private static final String EIR = "eir";
    private static final String EBS = "ebs";

    @Override
    public ObjectNode encode(CarrierEthernetBandwidthProfile bwProfile, CodecContext context) {
        checkNotNull(bwProfile, "BW profile cannot be null");
        ObjectNode result = context.mapper().createObjectNode();
        result.put(ID, bwProfile.id())
                .put(CFG_ID, bwProfile.cfgId())
                .put(TYPE, bwProfile.type().name())
                .put(CIR, bwProfile.cir().bps())
                .put(EIR, bwProfile.eir().bps())
                .put(CBS, bwProfile.cbs())
                .put(EBS, bwProfile.ebs());
        return result;
    }

    @Override
    public CarrierEthernetBandwidthProfile decode(ObjectNode json, CodecContext context) {
        return CarrierEthernetBandwidthProfile.builder()
                .id(json.path(ID).asText())
                .cfgId(json.path(CFG_ID).asText())
                .type(CarrierEthernetBandwidthProfile.Type.valueOf(json.path(TYPE).asText()))
                .cir(Bandwidth.bps(json.path(CIR).asDouble()))
                .eir(Bandwidth.bps(json.path(EIR).asDouble()))
                .cbs(json.path(CBS).asLong())
                .ebs(json.path(EBS).asLong())
                .build();
    }
}
