/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opencord.ce.api.models.codecs;

import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.onlab.packet.VlanId;
import org.onosproject.codec.CodecContext;
import org.onosproject.codec.JsonCodec;
import org.onosproject.net.ConnectPoint;
import org.opencord.ce.api.models.CarrierEthernetBandwidthProfile;
import org.opencord.ce.api.models.CarrierEthernetEnni;
import org.opencord.ce.api.models.CarrierEthernetGenericNi;
import org.opencord.ce.api.models.CarrierEthernetInni;
import org.opencord.ce.api.models.CarrierEthernetNetworkInterface;
import org.opencord.ce.api.models.CarrierEthernetUni;

import java.util.HashSet;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.opencord.ce.api.models.CarrierEthernetNetworkInterface.Type;

/**
 * Codec for {@link org.opencord.ce.api.models.CarrierEthernetNetworkInterface}.
 */
public class CarrierEthernetNetworkInterfaceCodec extends JsonCodec<CarrierEthernetNetworkInterface> {

    private static final String TYPE = "type";
    private static final String CONNECT_POINT = "cp";
    private static final String NI_ID = "niId";
    private static final String CFG_ID = "cfgId";
    private static final String BWP = "bwp";
    private static final String C_TAGS = "cTags";
    private static final String ROLE = "role";
    private static final String CAPACITY = "capacity";
    private static final String USED_CAPACITY = "usedCapacity";
    private static final String S_TAG = "sTag";
    private static final String C_TAG = "cTag";

    @Override
    public ObjectNode encode(CarrierEthernetNetworkInterface networkInterface, CodecContext context) {
        checkNotNull(networkInterface, "Java networkInterface class cannot be null");
        ObjectNode result = context.mapper().createObjectNode()
                .put(CFG_ID, networkInterface.id())
                .put(TYPE, networkInterface.type().toString())
                .put(ROLE, networkInterface.role().toString());
        if (networkInterface.capacity() != null) {
            result.put(CAPACITY, networkInterface.capacity().bps());
        }
        if (networkInterface.usedCapacity() != null) {
            result.put(USED_CAPACITY, networkInterface.usedCapacity().bps());
        }
        JsonCodec<ConnectPoint> cpCodec = context.codec(ConnectPoint.class);
        result.set(CONNECT_POINT, cpCodec.encode(networkInterface.cp(), context));
        switch (networkInterface.type()) {
            case UNI:
                CarrierEthernetUni uni = (CarrierEthernetUni) networkInterface;
                JsonCodec<CarrierEthernetBandwidthProfile> bwpCodec =
                        context.codec(CarrierEthernetBandwidthProfile.class);
                result.set(BWP, bwpCodec.encode(uni.bwp(), context));
                ArrayNode ctags = context.mapper().createArrayNode();
                uni.ceVlanIdSet().forEach(ceVlan -> ctags.add(ceVlan.toShort()));
                result.set(C_TAGS, ctags);
                break;
            case INNI:
                CarrierEthernetInni inni = (CarrierEthernetInni) networkInterface;
                result.put(S_TAG, inni.sVlanId().toShort());
                break;
            case ENNI:
                CarrierEthernetEnni enni = (CarrierEthernetEnni) networkInterface;
                result.put(S_TAG, enni.sVlanId().toShort());
                break;
            case GENERIC:
            default:
        }
        return result;
    }

    @Override
    public CarrierEthernetNetworkInterface decode(ObjectNode json, CodecContext context) {
        String id = json.path(NI_ID).asText();
        String cfgId = json.path(CFG_ID).asText();
        Type type = Type.valueOf(json.path(TYPE).asText());

        JsonCodec<ConnectPoint> cpCodec = context.codec(ConnectPoint.class);
        ConnectPoint cp = cpCodec.decode((ObjectNode) json.path(CONNECT_POINT), context);
        CarrierEthernetNetworkInterface ni;
        switch (type) {
            case UNI:
                JsonCodec<CarrierEthernetBandwidthProfile> bwpCodec =
                        context.codec(CarrierEthernetBandwidthProfile.class);
                Set<VlanId> ceVlanIds = new HashSet<>();
                json.path(C_TAGS).forEach(jsonCeVlan ->
                ceVlanIds.add(VlanId.vlanId(jsonCeVlan.shortValue())));
                CarrierEthernetUni.Role role = null;
                for (CarrierEthernetUni.Role r : CarrierEthernetUni.Role.values()) {
                    if (r.toString().equals(json.path(ROLE).asText())) {
                        role = r;
                        break;
                    }
                }
                ni = CarrierEthernetUni.builder()
                        .cp(cp)
                        .cfgId(cfgId)
                        .bwp(bwpCodec.decode((ObjectNode) json.path(BWP), context))
                        .ceVlanId(ceVlanIds.iterator().next())
                        .role(role)
                        .build();
                break;
            case ENNI:
                VlanId sTag = VlanId.vlanId(json.path(S_TAG).shortValue());
                CarrierEthernetEnni.Role roleEnni = null;
                for (CarrierEthernetEnni.Role r : CarrierEthernetEnni.Role.values()) {
                    if (r.toString().equals(json.path(ROLE).asText())) {
                        roleEnni = r;
                        break;
                    }
                }
                ni = CarrierEthernetEnni.builder()
                        .cp(cp)
                        .cfgId(cfgId)
                        .sVlanId(sTag)
                        .role(roleEnni)
                        .build();
                break;
            case INNI:
                VlanId sTagInni = VlanId.vlanId(json.path(S_TAG).shortValue());
                CarrierEthernetInni.Role roleInni = null;
                for (CarrierEthernetInni.Role r : CarrierEthernetInni.Role.values()) {
                    if (r.toString().equals(json.path(ROLE).asText())) {
                        roleInni = r;
                        break;
                    }
                }
                ni = CarrierEthernetInni.builder()
                        .cp(cp)
                        .cfgId(cfgId)
                        .sVlanId(sTagInni)
                        .role(roleInni)
                        .build();
                break;
            case GENERIC:
            default:
                ni = new CarrierEthernetGenericNi(cp, cfgId);
        }
        return ni;
    }

}
