/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opencord.ce.api.models.codecs;

import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.onlab.packet.VlanId;
import org.onlab.util.Bandwidth;
import org.onosproject.codec.CodecContext;
import org.onosproject.codec.JsonCodec;
import org.onosproject.net.ConnectPoint;
import org.opencord.ce.api.models.CarrierEthernetBandwidthProfile;
import org.opencord.ce.api.models.CarrierEthernetConnection;
import org.opencord.ce.api.models.CarrierEthernetNetworkInterface;
import org.opencord.ce.api.models.CarrierEthernetUni;
import org.opencord.ce.api.models.CarrierEthernetVirtualConnection;
import org.opencord.ce.api.models.EvcConnId;
import org.slf4j.Logger;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static org.onlab.util.Tools.nullIsIllegal;
import static org.opencord.ce.api.models.CarrierEthernetVirtualConnection.MAX_NUM_UNI;
import static org.slf4j.LoggerFactory.getLogger;
import static org.opencord.ce.api.models.CarrierEthernetVirtualConnection.builder;

/**
 * Codec for EVC.
 */
public class CarrierEthernetEvcCodec extends JsonCodec<CarrierEthernetVirtualConnection> {

    private static final String EVC_ID_REQUIRED = "EVC Id Must be specified";
    private static final String EVC_TYPE_REQUIRED = "EVC Type Must be specified";
    private static final String ARRAY_REQUIRED = "UNI array was not specified";

    private final Logger log = getLogger(getClass());

    @Override
    public ObjectNode encode(CarrierEthernetVirtualConnection evc, CodecContext context) {
        ObjectNode evcRoot = context.mapper().createObjectNode();
        evcRoot.put("evcCfgId", evc.cfgId());
        evcRoot.put("evcId", evc.id().id());
        ArrayNode uniList = context.mapper()
                .valueToTree(evc.uniSet().stream()
                        .map(CarrierEthernetNetworkInterface::id)
                        .collect(Collectors.toList()));
        evcRoot.putArray("uniList").addAll(uniList);
        evcRoot.put("maxNumUni", evc.maxNumUni());
        CarrierEthernetUni uni = evc.uniSet().iterator().next();
        evcRoot.put("vlanId", uni.ceVlanId().toString());
        CarrierEthernetBandwidthProfile bwp = uni.bwp();
        if (bwp != null) {
            evcRoot.put("cir", bwp.cir().bps());
            evcRoot.put("eir", bwp.eir().bps());
            evcRoot.put("cbs", bwp.cbs());
            evcRoot.put("ebs", bwp.ebs());
        }
        return evcRoot;
    }

    @Override
    public CarrierEthernetVirtualConnection decode(ObjectNode json, CodecContext context) {
        String argEvcCfgId = json.get("evcCfgId").asText(null);
        String argEvcId = null;
        if (json.has("evcId")) {
            argEvcId = json.get("evcId").asText();
        }
        ArrayNode uniArray = nullIsIllegal((ArrayNode) json.get("uniList"),
                ARRAY_REQUIRED);
        List<String> uniList = new ArrayList<>();
        uniArray.forEach(jsonNode -> uniList.add(jsonNode.asText()));
        String evcTypeString = nullIsIllegal(json.get("evcType").asText(),
                EVC_TYPE_REQUIRED);
        int maxNumUni = -1;
        if (json.has("maxNumUni")) {
            maxNumUni = json.get("maxNumUni").asInt(-1);
        }

        short vlanId = -1;
        if (json.has("vlanId")) {
            vlanId = json.get("vlanId").shortValue();
        }
        double cir = json.get("cir").asDouble(0.0);
        double eir = json.get("eir").asDouble(0.0);
        long cbs = json.get("cbs").asLong(0L);
        long ebs = json.get("ebs").asLong(0L);
        log.info("Received REST call with parameters: " + "evcCfgId={}, evcId={}," +
                        " uniList={}, evcType={}, maxNumUni={}, vlanId={}, cir={}, " +
                        "eir={}, cbs={}, ebs={}", argEvcCfgId, argEvcId, uniList,
                evcTypeString, maxNumUni, vlanId, cir, eir, cbs, ebs);
        return builder().id(EvcConnId.of(argEvcId)).cfgId(argEvcCfgId)
                .type(generateEvcType(evcTypeString, uniList))
                .maxNumUni(generateMaxNumUni(maxNumUni, evcTypeString, uniList))
                .uniSet(generateUniSet(evcTypeString, uniList.subList(1, uniList.size()),
                        vlanId, uniList.get(0), argEvcCfgId, cir, eir,
                        cbs, ebs))
                .build();
    }

    /**
     * Return the CE-VLAN ID for the CE evc based on the CLI-supplied argument.
     *
     * @param argCeVlanId vlanID
     * @return CE-VLAN ID for the CE evc
     */
    public static VlanId generateCeVlanId(short argCeVlanId) {
        return ((argCeVlanId == -1) ? null : VlanId.vlanId(argCeVlanId));
    }

    /**
     * Return the CE evc type based on the CLI-supplied arguments.
     *
     * @param evcTypeString EVC type
     * @param uniList       UNIs list
     * @return the CE evc type
     */
    public static CarrierEthernetConnection.Type generateEvcType(String evcTypeString, List<String> uniList) {
        if (evcTypeString == null) {
            return ((uniList.size() > 2) ?
                    CarrierEthernetConnection.Type.MULTIPOINT_TO_MULTIPOINT :
                    CarrierEthernetConnection.Type.POINT_TO_POINT);
        } else {
            // TODO: Catch exception
            return CarrierEthernetConnection.Type.valueOf(evcTypeString);
        }
    }

    /**
     * Return the EVC maxNumUni parameter based on the CLI-supplied arguments.
     *
     * @param maxNumUni     maximum numbers of UNIs
     * @param evcTypeString EVC Type
     * @param uniList       UNIs list
     * @return the maxNumUni parameter
     */
    public static Integer generateMaxNumUni(int maxNumUni, String evcTypeString, List<String> uniList) {
        if (maxNumUni == -1) {
            if (evcTypeString == null) {
                return ((uniList.size() > 2) ?
                        MAX_NUM_UNI : 2);
            } else {
                // TODO: Catch exception
                CarrierEthernetConnection.Type evcType =
                        CarrierEthernetConnection.Type.valueOf(evcTypeString);
                return (evcType.equals(CarrierEthernetConnection.Type.POINT_TO_POINT) ? 2 :
                        MAX_NUM_UNI);
            }
        } else {
            return maxNumUni;
        }
    }

    /**
     * Return the BW profile type based on the CLI-supplied arguments.
     *
     * @param ceVlanId vlanID
     * @return the BWP profile type
     */
    public static CarrierEthernetBandwidthProfile.Type generateBandwidthProfileType(short ceVlanId) {
        // TODO: Add the CoS BW profile case
        return ((ceVlanId == -1) ?
                CarrierEthernetBandwidthProfile.Type.INTERFACE : CarrierEthernetBandwidthProfile.Type.EVC);
    }

    /**
     * Return the BW profile id based on the CLI-supplied arguments.
     *
     * @param uniId    UNI id
     * @param evcCfgId EVC configuration Id
     * @param ceVlanId vlanID
     * @return the BW profile id
     */
    public static String generateBandwidthProfileId(String uniId, String evcCfgId, int ceVlanId) {
        // TODO: Add the CoS BW profile case
        return ((ceVlanId == -1) ? uniId : evcCfgId);
    }

    /**
     * Return the set of UNIs for the CE EVC based on the CLI-supplied arguments.
     *
     * @param evcTypeString Type of the UNI
     * @param uniList       list of remaning UNIs
     * @param ceVlanId      vlanID
     * @param firstUni      first UNI id
     * @param evcCfgId      EVC configuration Id
     * @param cir           The CIR in Mbps
     * @param eir           The EIR in Mbps
     * @param cbs           The CBS in Byte
     * @param ebs           The EBS in Byte
     * @return the set of UNIs for the CE EVC
     */
    public static Set<CarrierEthernetUni> generateUniSet(String evcTypeString, List<String> uniList,
                                                         short ceVlanId, String firstUni, String evcCfgId,
                                                         Double cir, Double eir, Long cbs, Long ebs) {

        Set<CarrierEthernetUni> uniSet = new HashSet<>();

        CarrierEthernetConnection.Type evcType = generateEvcType(evcTypeString, uniList);

        // We assume that first UNI supplied is always root
        uniSet.add(CarrierEthernetUni.builder()
                .cp(ConnectPoint.deviceConnectPoint(firstUni))
                .role(CarrierEthernetUni.Role.ROOT)
                .ceVlanId(generateCeVlanId(ceVlanId))
                .bwp(CarrierEthernetBandwidthProfile.builder()
                        .id(generateBandwidthProfileId(firstUni, evcCfgId, ceVlanId))
                        .type(generateBandwidthProfileType(ceVlanId))
                        .cir(Bandwidth.mbps(cir))
                        .eir(Bandwidth.mbps(eir))
                        .cbs(cbs)
                        .ebs(ebs)
                        .build())
                .build());

        final CarrierEthernetUni.Role role;
        // For E-Line and E-LAN all UNIs are roots. For E-Tree all UNIs are leafs except from one
        role = ((evcType == CarrierEthernetConnection.Type.ROOT_MULTIPOINT) ?
                CarrierEthernetUni.Role.LEAF : CarrierEthernetUni.Role.ROOT);

        uniList.forEach(argUni -> uniSet.add(
                CarrierEthernetUni.builder()
                        .cp(ConnectPoint.deviceConnectPoint(argUni))
                        .role(role)
                        .ceVlanId(generateCeVlanId(ceVlanId))
                        .bwp(CarrierEthernetBandwidthProfile.builder()
                                .id(generateBandwidthProfileId(argUni, evcCfgId, ceVlanId))
                                .type(generateBandwidthProfileType(ceVlanId))
                                .cir(Bandwidth.mbps(cir))
                                .eir(Bandwidth.mbps(eir))
                                .cbs(cbs)
                                .ebs(ebs)
                                .build())
                        .build()));

        return uniSet;
    }

}