/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opencord.ce.api.models.codecs;

import com.fasterxml.jackson.databind.node.ObjectNode;
import org.onosproject.codec.CodecContext;
import org.onosproject.codec.JsonCodec;
import org.opencord.ce.api.models.CarrierEthernetLogicalTerminationPoint;
import org.opencord.ce.api.models.CarrierEthernetNetworkInterface;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.opencord.ce.api.models.CarrierEthernetLogicalTerminationPoint.Role;

/**
 * Codec for {@link CarrierEthernetLogicalTerminationPoint}.
 */
public class LogicalTerminationEndPointCodec extends JsonCodec<CarrierEthernetLogicalTerminationPoint> {

    private static final String ROLE = "role";
    private static final String LTP_ID = "ltpId";
    private static final String CFG_ID = "cfgId";
    private static final String NETWORK_INTERFACE = "ni";

    @Override
    public ObjectNode encode(CarrierEthernetLogicalTerminationPoint ltp, CodecContext context) {
        checkNotNull(ltp, "Java ltp object cannot be null");

        JsonCodec<CarrierEthernetNetworkInterface> niCodec =
                context.codec(CarrierEthernetNetworkInterface.class);

        ObjectNode result = context.mapper().createObjectNode()
                .put(LTP_ID, ltp.id())
                .put(CFG_ID, ltp.cfgId())
                .put(ROLE, ltp.role().toString());
        result.set(NETWORK_INTERFACE, niCodec.encode(ltp.ni(), context));
        return result;
    }

    @Override
    public CarrierEthernetLogicalTerminationPoint decode(ObjectNode json, CodecContext context) {

        String ltpId = json.path(LTP_ID).asText();
        String cfgId = json.path(CFG_ID).asText();
        JsonCodec<CarrierEthernetNetworkInterface> niCodec =
                context.codec(CarrierEthernetNetworkInterface.class);
        ObjectNode niJson = get(json, NETWORK_INTERFACE);
        CarrierEthernetNetworkInterface networkInterface =
                niCodec.decode(niJson, context);
        CarrierEthernetLogicalTerminationPoint ltp =
                new CarrierEthernetLogicalTerminationPoint(cfgId, networkInterface);
        for (Role r : Role.values()) {
            if (r.toString().equals(json.path(ROLE).asText())) {
                ltp.setRole(r);
            }
        }
        return ltp;
    }
}
