/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opencord.ce.api.models.codecs;

import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.onlab.packet.VlanId;
import org.onosproject.codec.CodecContext;
import org.onosproject.codec.JsonCodec;
import org.opencord.ce.api.models.CarrierEthernetConnection;
import org.opencord.ce.api.models.CarrierEthernetForwardingConstruct;
import org.opencord.ce.api.models.CarrierEthernetLogicalTerminationPoint;
import org.opencord.ce.api.models.EvcConnId;

import java.time.Duration;
import java.util.HashSet;
import java.util.Set;

/**
 * Codec {@link CarrierEthernetForwardingConstruct}.
 */
public class ForwardingConstructCodec extends JsonCodec<CarrierEthernetForwardingConstruct> {
    private static final String FC_ID = "fcId";
    private static final String CFG_ID = "cfgId";
    private static final String TYPE = "type";
    private static final String STATE = "state";
    private static final String MAX_LATENCY = "maxLatency";
    private static final String VLAN_ID = "vlanId";
    private static final String LTP_LIST = "ltpList";

    @Override
    public ObjectNode encode(CarrierEthernetForwardingConstruct fc, CodecContext context) {
        JsonCodec<CarrierEthernetLogicalTerminationPoint> ltpCodec =
                context.codec(CarrierEthernetLogicalTerminationPoint.class);
        ObjectNode result = context.mapper().createObjectNode()
                .put(FC_ID, fc.id().id())
                .put(CFG_ID, fc.cfgId())
                .put(TYPE, fc.type().toString())
                .put(STATE, fc.type().toString())
                .put(MAX_LATENCY, fc.maxLatency().toString());
        result.put(VLAN_ID, fc.vlanId().toShort());
        ArrayNode ltpsNode = context.mapper().createArrayNode();
        fc.ltpSet().forEach(ltp ->
            ltpsNode.add(ltpCodec.encode(ltp, context)));
        result.set(LTP_LIST, ltpsNode);
        return result;
    }

    @Override
    public CarrierEthernetForwardingConstruct decode(ObjectNode json, CodecContext context) {
        JsonCodec<CarrierEthernetLogicalTerminationPoint> ltpCodec =
                context.codec(CarrierEthernetLogicalTerminationPoint.class);
        Set<CarrierEthernetLogicalTerminationPoint> ltps =
                new HashSet<>();
        json.path(LTP_LIST).forEach(ltpJson ->
            ltps.add(ltpCodec.decode((ObjectNode) ltpJson, context)));

        CarrierEthernetForwardingConstruct fc = CarrierEthernetForwardingConstruct.builder()
                .id(EvcConnId.of(json.path(FC_ID).asText()))
                .cfgId(json.path(CFG_ID).asText())
                .type(CarrierEthernetConnection.Type.valueOf(json.path(TYPE).asText()))
                .maxLatency(Duration.parse(json.path(MAX_LATENCY).asText()))
                .ltpSet(ltps)
                .build();
        fc.setVlanId(VlanId.vlanId(json.path(VLAN_ID).shortValue()));
        return fc;
    }
}
