/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opencord.ce.api.services.virtualprovider;


import org.onosproject.net.DeviceId;
import org.onosproject.net.Port;
import org.onosproject.net.PortNumber;
import org.onosproject.net.device.PortDescription;
import org.onosproject.net.domain.DomainId;

import java.util.List;

/**
 * Service used by the southbound communication channel to notify virtual device presence.
 */
public interface EcordDeviceProviderService {

    /**
     * Signals the domain provider to connect a device of another domain.
     *
     * @param domainVirtualDevice device object
     */
    void connectRemoteDevice(DomainVirtualDevice domainVirtualDevice);

    /**
     * Adds or updates ports on a remote domain.
     *
     * @param domainId domain ID
     * @param deviceId device ID
     * @param ports list of port descriptions
     */
    void addOrUpdateRemotePorts(DomainId domainId, DeviceId deviceId, List<PortDescription> ports);

    /**
     * Signals the domain provider that a specific domain has added a new port to a device.
     *
     * @param domainId domain identifier
     * @param deviceId device identifier
     * @param port port
     */
    void addRemotePort(DomainId domainId, DeviceId deviceId, Port port);

    /**
     * Signals a remote port update.
     *
     * @param domainId domain identifier
     * @param deviceId device identifier
     * @param port port description
     */
    void updateRemotePortState(DomainId domainId, DeviceId deviceId, Port port);

    /**
     * Signals the disconnection of a remote domain device.
     *
     * @param domainId domain identifier
     * @param deviceId device identifier
     */
    void disconnectRemoteDevice(DomainId domainId, DeviceId deviceId);

    /**
     * Signals the deletion of a remote domain device port.
     * @param domainId domain identifier
     * @param deviceId device identifier
     * @param portNumber port identifier
     */
    void removeRemotePort(DomainId domainId, DeviceId deviceId, PortNumber portNumber);
}
