/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opencord.ce.api.models;


import org.onlab.util.Bandwidth;

import static com.google.common.base.MoreObjects.toStringHelper;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Bandwidth profile for a CE UNI.
 */
// TODO: Methods to associate a CE BWP to a generic BWP and a Meter
public final class CarrierEthernetBandwidthProfile {

    public enum Type {
        INTERFACE, EVC, COS
    }

    protected String id;
    protected String cfgId;
    protected Type type;
    protected Bandwidth cir;
    protected Bandwidth eir;
    protected long cbs;
    protected long ebs;

    // TODO: Remove id from constructor - should be generated by some manager
    private CarrierEthernetBandwidthProfile(String id, String cfgId, Type type,
                                            Bandwidth cir, Bandwidth eir,
                                            long cbs, long ebs) {
        checkNotNull(cir, "BW Profile must have an associated CIR");
        checkNotNull(type, "BW profile must have a type");
        this.id = id;
        this.cfgId = cfgId;
        this.type = type;
        this.cir = cir;
        this.eir = eir;
        this.cbs = cbs;
        this.ebs = ebs;
    }

    /**
     * Returns BWP string identifier.
     *
     * @return BWP string identifier
     */
    public String id() {
        return id;
    }

    /**
     * Returns BWP string config identifier.
     *
     * @return BWP string config identifier
     */
    public String cfgId() {
        return cfgId;
    }

    /**
     * Returns BWP type (INTERFACE, EVC, COS).
     *
     * @return BWP type
     */
    public Type type() {
        return type;
    }

    /**
     * Returns BWP CIR rate.
     *
     * @return BWP CIR rate
     */
    public Bandwidth cir() {
        return cir;
    }

    /**
     * Returns BWP EIR rate.
     *
     * @return BWP EIR rate
     */
    public Bandwidth eir() {
        return eir;
    }

    /**
     * Returns BWP CBS in Bytes.
     *
     * @return BWP CBS in Bytes
     */
    public long cbs() {
        return cbs;
    }

    /**
     * Returns BWP EBS in Bytes.
     *
     * @return BWP EBS in Bytes
     */
    public long ebs() {
        return ebs;
    }

    /**
     * Sets BWP id.
     *
     * @param id the id to set
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * Sets BWP CIR rate.
     *
     * @param cir the CIR to set
     */
    public void setCir(Bandwidth cir) {
        this.cir = cir;
    }

    /**
     * Sets BWP EIR rate.
     *
     * @param eir the EIR to set
     */
    public void setEir(Bandwidth eir) {
        this.eir = eir;
    }

    public String toString() {

        return toStringHelper(this)
                .add("id", id)
                .add("type", type)
                .add("cir", cir)
                .add("cbs", cbs)
                .add("eir", eir)
                .add("ebs", ebs).toString();
    }

    /**
     * Returns a new builder.
     *
     * @return new builder
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * Builder of CarrierEthernetBandwidthProfile entities.
     */
    public static final class Builder {

        String id;
        String cfgId;
        Type type;
        Bandwidth cir;
        Bandwidth eir;
        long cbs;
        long ebs;

        /**
         * Sets the id of this builder.
         *
         * @param id the builder id to set
         * @return this builder instance
         */
        public Builder id(String id) {
            this.id = id;
            return this;
        }

        /**
         * Sets the cfgId of this builder.
         *
         * @param cfgId the builder cfgId to set
         * @return this builder instance
         */
        public Builder cfgId(String cfgId) {
            this.cfgId = cfgId;
            return this;
        }

        /**
         * Sets the type of this builder.
         *
         * @param type the builder type to set
         * @return this builder instance
         */
        public Builder type(Type type) {
            this.type = type;
            return this;
        }

        /**
         * Sets the cir of this builder.
         *
         * @param cir the builder cir to set
         * @return this builder instance
         */
        public Builder cir(Bandwidth cir) {
            this.cir = cir;
            return this;
        }

        /**
         * Sets the eir of this builder.
         *
         * @param eir the builder eir to set
         * @return this builder instance
         */
        public Builder eir(Bandwidth eir) {
            this.eir = eir;
            return this;
        }

        /**
         * Sets the cbs of this builder.
         *
         * @param cbs the builder cbs to set
         * @return this builder instance
         */
        public Builder cbs(long cbs) {
            this.cbs = cbs;
            return this;
        }

        /**
         * Sets the ebs of this builder.
         *
         * @param ebs the builder ebs to set
         * @return this builder instance
         */
        public Builder ebs(long ebs) {
            this.ebs = ebs;
            return this;
        }

        /**
         * Builds a new CarrierEthernetBandwidthProfile instance based on this builder's parameters.
         *
         * @return a new CarrierEthernetBandwidthProfile instance
         */
        public CarrierEthernetBandwidthProfile build() {
            return new CarrierEthernetBandwidthProfile(id, cfgId, type, cir, eir, cbs, ebs);
        }
    }
}

