/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opencord.ce.api.models.codecs;

import com.fasterxml.jackson.databind.node.ObjectNode;
import org.onosproject.codec.CodecContext;
import org.onosproject.codec.JsonCodec;
import org.opencord.ce.api.models.CarrierEthernetConnection;
import org.opencord.ce.api.models.EvcConnId;

import java.time.Duration;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.opencord.ce.api.models.CarrierEthernetConnection.Type;
import static org.opencord.ce.api.models.CarrierEthernetConnection.State;
/**
 * Codec for {@link org.opencord.ce.api.models.CarrierEthernetConnection}.
 */
public class CarrierEthernetConnectionCodec extends JsonCodec<CarrierEthernetConnection> {

    private static final String ID = "connectionId";
    private static final String CFG_ID = "cfgId";
    private static final String TYPE = "type";
    private static final String STATE = "state";
    private static final String MAX_LATENCY = "maxLatency";

    @Override
    public ObjectNode encode(CarrierEthernetConnection connection, CodecContext context) {
        checkNotNull(connection, "Java connection object cannot be null");
        ObjectNode result = context.mapper().createObjectNode()
                .put(ID, connection.id().id())
                .put(CFG_ID, connection.cfgId())
                .put(TYPE, connection.type().toString())
                .put(STATE, connection.type().toString())
                .put(MAX_LATENCY, connection.maxLatency().toString());
        return result;
    }

    @Override
    public CarrierEthernetConnection decode(ObjectNode json, CodecContext context) {
        if (json == null || !json.isObject()) {
            return null;
        }

        EvcConnId id = EvcConnId.of(json.path(ID).asText());
        String cfgId = json.path(CFG_ID).asText();
        Type type = Type.valueOf(json.path(TYPE).asText());
        State state = State.valueOf(json.path(STATE).asText());
        Duration maxLatency = Duration.parse(json.path(MAX_LATENCY).asText());
        return null;
    }

}
