/*
 * Copyright 2017-present Open Networking Laboratory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opencord.ce.global.channel;


import com.google.common.collect.Sets;

import org.apache.commons.lang3.tuple.Pair;
import org.apache.felix.scr.annotations.Activate;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Deactivate;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferenceCardinality;
import org.apache.felix.scr.annotations.Service;

import org.onlab.packet.IpAddress;
import org.onlab.util.KryoNamespace;
import org.onosproject.cluster.ClusterService;
import org.onosproject.cluster.LeadershipService;
import org.onosproject.cluster.NodeId;
import org.onosproject.core.ApplicationId;
import org.onosproject.core.CoreService;
import org.onosproject.net.config.ConfigFactory;
import org.onosproject.net.config.NetworkConfigEvent;
import org.onosproject.net.config.NetworkConfigListener;
import org.onosproject.net.config.NetworkConfigRegistry;
import org.onosproject.net.config.NetworkConfigService;
import org.onosproject.net.config.basics.SubjectFactories;
import org.onosproject.net.domain.DomainId;
import org.onosproject.net.domain.DomainService;
import org.onosproject.store.serializers.KryoNamespaces;
import org.onosproject.store.service.ConsistentMap;
import org.onosproject.store.service.Serializer;
import org.onosproject.store.service.StorageService;
import org.opencord.ce.api.services.channel.ConnectionService;
import org.opencord.ce.api.services.channel.ControlChannelListenerService;
import org.opencord.ce.api.services.channel.EndPoint;
import org.opencord.ce.global.channel.client.ConnectionConfig;
import org.opencord.ce.global.channel.client.HttpClientInstance;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.client.Client;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;


/**
 * Southbound component that adds a listener to the
 * {@link ControlChannelListenerService}.
 *
 * Accomplishes network tasks through HTTP requests,
 * acting as a client that send {@link org.opencord.ce.api.models.CarrierEthernetForwardingConstruct}
 * requests to the underlying domains
 */
@Component(immediate = true)
@Service(ConnectionService.class)
public class HttpClientComponent implements ConnectionService {
    private static final String APP_NAME = "org.opencord.ce.global.channel.http";

    // temporary
    private static final String TOPIC_ONE = "ecord-domains-topic-one";
    private static final String TOPIC_TWO = "ecord-domains-topic-two";
    private static final String TOPIC_THREE = "ecord-domains-topic-three";

    private final Logger log = LoggerFactory.getLogger(getClass());
    private ApplicationId appId;

    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY)
    protected ClusterService clusterService;

    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY)
    protected LeadershipService leadershipService;

    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY)
    protected StorageService storageService;

    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY)
    protected NetworkConfigRegistry configRegistry;

    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY)
    protected NetworkConfigService configService;

    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY)
    protected CoreService coreService;

    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY)
    protected ControlChannelListenerService channelListenerService;

    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY)
    protected DomainService domainService;

    private final Set<EndPoint> endPoints = Sets.newHashSet();

    private ConsistentMap<EndPoint, IpAddress> endPointMasterIpMap;

    private final ExecutorService eventExecutor =
            Executors.newSingleThreadExecutor();

    private final ConfigFactory<ApplicationId, ConnectionConfig> configFactory =
            new ConfigFactory<ApplicationId, ConnectionConfig>(SubjectFactories.APP_SUBJECT_FACTORY,
                    ConnectionConfig.class, "endPoints") {
                @Override
                public ConnectionConfig createConfig() {
                    return new ConnectionConfig();
                }
            };

    private final NetworkConfigListener configListener = new InternalConfigListener();

    @Activate
    protected void activate() {
        log.info("Started");

        appId = coreService.registerApplication(APP_NAME);
        endPointMasterIpMap = storageService.<EndPoint, IpAddress>consistentMapBuilder()
                .withName("ecord-domain-endpoints")
                .withSerializer(Serializer.using(
                        new KryoNamespace.Builder()
                                .register(KryoNamespaces.API)
                                .register(DomainId.class)
                                .register(EndPoint.class)
                                .build()
                )).build();
        configRegistry.registerConfigFactory(configFactory);
        configService.addListener(configListener);
        channelListenerService.addListener(HttpClientInstance.INSTANCE);
        leadershipService.runForLeadership(TOPIC_ONE);
        leadershipService.runForLeadership(TOPIC_TWO);
        leadershipService.runForLeadership(TOPIC_THREE);
    }

    @Deactivate
    protected void deactivate() {
        log.info("Stopped");

        configService.removeListener(configListener);
        configRegistry.unregisterConfigFactory(configFactory);
        channelListenerService.removeListener(HttpClientInstance.INSTANCE);
        leadershipService.withdraw(TOPIC_ONE);
        leadershipService.withdraw(TOPIC_TWO);
        leadershipService.withdraw(TOPIC_THREE);
        HttpClientInstance.INSTANCE.stopNetworkTasks();
    }

    @Override
    public IpAddress getLocalMasterIp(String domainId) {
        synchronized (this) {
            for (EndPoint ep : endPoints) {
                if (ep.domainId().id().equals(domainId)) {
                    String topic = ep.topic();
                    String masterIp;
                    String leaderId = leadershipService.getLeader(topic).id();
                    log.info("local leaderId: " + leaderId);
                    masterIp = clusterService.getNode(NodeId.nodeId(leaderId)).ip().toString();

                    return IpAddress.valueOf(masterIp);
                }
            }
        }
        log.info("Found no leader for domain " + domainId +
                "-- endPoints size: " + endPoints.size());
        return null;
    }

    @Override
    public IpAddress getRemoteMasterIp(String domainId) {
        synchronized (this) {
            for (EndPoint ep : endPoints) {
                if (ep.domainId().id().equals(domainId)) {
                    return HttpClientInstance.INSTANCE.getRemoteMasterIp(ep);
                }
            }
        }
        log.info("Found no master ip for domain {}", domainId);
        return null;
    }

    @Override
    public Pair<Client, IpAddress> getConnectionInfo(DomainId domainId) {
        Client client = HttpClientInstance.INSTANCE
                .getConnectionInfo(domainId);
        return Pair.of(client, getRemoteMasterIp(domainId.id()));
    }

    private void readConfig() {
        ConnectionConfig config = configRegistry.getConfig(appId, ConnectionConfig.class);
        log.debug("Domains connections config received");

        synchronized (this) {
            endPoints.addAll(config.endPoints());
        }

        HttpClientInstance.INSTANCE.configure(clusterService, leadershipService, domainService,
                endPointMasterIpMap, config);
    }

    private class InternalConfigListener implements NetworkConfigListener {

        @Override
        public void event(NetworkConfigEvent event) {
            if (!event.configClass().equals(ConnectionConfig.class)) {
                return;
            }
            switch (event.type()) {
                case CONFIG_ADDED:
                    log.info("Network configuration added");
                    eventExecutor.execute(HttpClientComponent.this::readConfig);
                    break;
                case CONFIG_UPDATED:
                    log.info("Network configuration updated");
                    eventExecutor.execute(HttpClientComponent.this::readConfig);
                    break;
                default:
                    break;
            }
        }
    }
}
