/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opencord.ce.global.channel.client;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.Sets;
import org.onlab.packet.IpAddress;
import org.onosproject.core.ApplicationId;
import org.onosproject.net.config.Config;
import org.onosproject.net.domain.DomainId;
import org.opencord.ce.api.models.DomainEndPoint;

import java.util.ArrayList;
import java.util.Set;

/**
 * Configuration class for this bundle.
 *
 * Look at /config-samples/ecord-global-config.json for a sample configuration
 */
public class ConnectionConfig extends Config<ApplicationId> {

    private static final String PORT = "port";

    private static final String TOPICS = "topics";

    private static final String DOMAINS = "domains";
    private static final String DOMAIN_ID = "domainId";
    private static final String PUBLIC_IP = "publicIp";
    private static final String USERNAME = "username";
    private static final String PASSWD = "password";
    private static final String TOPIC = "topic";

    /**
     * List of topics to distribute network operations among ONOS instances.
     * @return list of topics
     */
    public ArrayList<String> topics() {
        ArrayList<String> topics = new ArrayList<>();
        object.path(TOPICS).forEach(
                topic -> topics.add(topic.asText())
        );
        return topics;
    }

    /**
     * Returns set of domain end points.
     * @return set of domain end points
     */
    public Set<DomainEndPoint> endPoints() {
        JsonNode peersNode = object.get(DOMAINS);
        Set<DomainEndPoint> endPoints = Sets.newHashSet();

        peersNode.forEach(jsonNode -> {
            DomainId domainId = DomainId.domainId(
                    jsonNode.path(DOMAIN_ID).asText());
            IpAddress publicIp = IpAddress.valueOf(jsonNode.path(PUBLIC_IP).asText());
            int port = jsonNode.path(PORT).asInt();

            String username = jsonNode.path(USERNAME).asText();
            String password = jsonNode.path(PASSWD).asText();

            String topic = jsonNode.path(TOPIC).asText();

            endPoints.add(new DomainEndPoint(domainId, publicIp, port,
                    username, password, topic));
        });

        return endPoints;
    }
}
