/*
 * Copyright 2017-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.opencord.ce.local.fabric;

import com.google.common.collect.Sets;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.onlab.junit.TestUtils;
import org.onlab.osgi.ServiceDirectory;
import org.onlab.osgi.TestServiceDirectory;
import org.onlab.packet.VlanId;
import org.onosproject.core.CoreServiceAdapter;
import org.onosproject.event.AbstractListenerManager;
import org.onosproject.net.ConnectPoint;
import org.onosproject.net.DeviceId;
import org.onosproject.net.PortNumber;
import org.onosproject.net.config.NetworkConfigRegistryAdapter;
import org.onosproject.net.device.DeviceService;
import org.onosproject.net.device.DeviceServiceAdapter;
import org.onosproject.net.device.PortDescription;
import org.onosproject.net.domain.DomainId;
import org.opencord.ce.api.models.CarrierEthernetConnection;
import org.opencord.ce.api.models.CarrierEthernetEnni;
import org.opencord.ce.api.models.CarrierEthernetForwardingConstruct;
import org.opencord.ce.api.models.CarrierEthernetInni;
import org.opencord.ce.api.models.CarrierEthernetLogicalTerminationPoint;
import org.opencord.ce.api.models.CarrierEthernetNetworkInterface;
import org.opencord.ce.api.models.CarrierEthernetUni;
import org.opencord.ce.api.models.EvcConnId;
import org.opencord.ce.local.bigswitch.BigSwitchEvent;
import org.opencord.ce.local.bigswitch.BigSwitchListener;
import org.opencord.ce.local.bigswitch.BigSwitchService;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

import static org.junit.Assert.assertEquals;

/**
 * Unit tests for {@link CarrierEthernetFabricManager}.
 */
public class CarrierEthernetFabricManagerTest {

    private CarrierEthernetFabricManager target;
    private static ServiceDirectory original;
    private static TestServiceDirectory directory;

    private CarrierEthernetLogicalTerminationPoint ltpSrc;
    private CarrierEthernetLogicalTerminationPoint ltpDst;
    // Physical connect points
    private static ConnectPoint srcUniCp = ConnectPoint.deviceConnectPoint("netconf:10.0.0.10/1");
    private ConnectPoint dstUniCp = ConnectPoint.deviceConnectPoint("netconf:10.0.0.20/1");
    private ConnectPoint inniPhyCp = ConnectPoint.deviceConnectPoint("of:0000000000000001/1");
    private ConnectPoint enni1PhyCp = ConnectPoint.deviceConnectPoint("of:0000000000000001/2");
    private ConnectPoint enni2PhyCp = ConnectPoint.deviceConnectPoint("of:0000000000000001/3");
    // Bigswitch (logical) connect points
    private ConnectPoint inniBsCp = ConnectPoint.deviceConnectPoint("domain:0000000000000001/1");
    private ConnectPoint enni1BsCp = ConnectPoint.deviceConnectPoint("domain:0000000000000001/2");
    private ConnectPoint enni2BsCp = ConnectPoint.deviceConnectPoint("domain:0000000000000001/3");
    // Carrier Ethernet models
    private CarrierEthernetForwardingConstruct fc1;
    private CarrierEthernetForwardingConstruct fc2;
    private CarrierEthernetNetworkInterface srcUni;
    private CarrierEthernetNetworkInterface dstUni;
    private CarrierEthernetNetworkInterface inni;
    private CarrierEthernetNetworkInterface enni1;
    private CarrierEthernetNetworkInterface enni2;

    private DeviceId deviceId = DeviceId.deviceId("of:0000000000000001");
    private static VlanId vlan1 = VlanId.vlanId((short) 100);
    private static VlanId vlan2 = VlanId.vlanId((short) 200);

    @BeforeClass
    public static void setUpBaseConfigClass() throws TestUtils.TestUtilsException {
        directory = new TestServiceDirectory();
        directory.add(DeviceService.class, new DeviceServiceAdapter(Collections.emptyList()));
        original = TestUtils.getField(CarrierEthernetNetworkInterface.class, "serviceDirectory");
        TestUtils.setField(CarrierEthernetNetworkInterface.class, "serviceDirectory", directory);
    }

    @AfterClass
    public static void tearDownBaseConfigClass() throws TestUtils.TestUtilsException {
        TestUtils.setField(CarrierEthernetNetworkInterface.class, "serviceDirectory", original);
    }

    @Before
    public void setUp() throws Exception {
        // Initialize the fabric manager
        target = new CarrierEthernetFabricManager();
        target.bigSwitchService = new TestBigSwitchService();
        target.cfgService = new NetworkConfigRegistryAdapter();
        target.coreService = new CoreServiceAdapter();

        // Build UNIs
        srcUni = CarrierEthernetUni.builder()
                .cp(srcUniCp)
                .build();
        dstUni = CarrierEthernetUni.builder()
                .cp(dstUniCp)
                .build();
        ltpSrc = new CarrierEthernetLogicalTerminationPoint("srcUni", srcUni);
        ltpDst = new CarrierEthernetLogicalTerminationPoint("dstUni", dstUni);

        // Build FC connecting UNIs
        fc1 = CarrierEthernetForwardingConstruct.builder()
                .cfgId("test1")
                .id(EvcConnId.of("test1"))
                .type(CarrierEthernetConnection.Type.POINT_TO_POINT)
                .ltpSet(Sets.newHashSet(ltpSrc, ltpDst))
                .build();
        fc1.setVlanId(vlan1);
        fc2 = CarrierEthernetForwardingConstruct.builder()
                .cfgId("test2")
                .id(EvcConnId.of("test2"))
                .type(CarrierEthernetConnection.Type.POINT_TO_POINT)
                .ltpSet(Sets.newHashSet(ltpSrc, ltpDst))
                .build();
        fc2.setVlanId(vlan2);
        inni = CarrierEthernetInni.builder()
                .cp(inniBsCp)
                .build();
        enni1 = CarrierEthernetEnni.builder()
                .cp(enni1BsCp)
                .build();
        enni2 = CarrierEthernetEnni.builder()
                .cp(enni2BsCp)
                .build();

        target.activate();
    }

    @After
    public void tearDown() {
        target.deactivate();
    }

    @Test
    public void testBuildConfig() {
        TestUtils.setField(target, "deviceId", deviceId);
        JsonParser parser = new JsonParser();
        JsonObject json;
        String expected;

        // Add first fc
        target.addCrossconnect(fc1, inni, Sets.newHashSet(enni1));
        json = target.buildConfig();
        expected =
                "{\"apps\":{" +
                        "\"org.onosproject.segmentrouting\":{" +
                        "\"xconnect\":{" +
                        "\"of:0000000000000001\":[{" +
                        "\"vlan\":" + vlan1.toString() + "," +
                        "\"ports\":[1, 2]," +
                        "\"name\": \"" + fc1.id().id() + "\"}" +
                        "]}}}}";
        assertEquals(json, parser.parse(expected).getAsJsonObject());

        // Add second fc
        target.addCrossconnect(fc2, inni, Sets.newHashSet(enni2));
        json = target.buildConfig();
        expected =
                "{\"apps\":{" +
                        "\"org.onosproject.segmentrouting\":{" +
                            "\"xconnect\":{" +
                                "\"of:0000000000000001\":[{" +
                                    "\"vlan\":" + vlan1.toString() + "," +
                                    "\"ports\":[1, 2]," +
                                    "\"name\": \"" + fc1.id().id() + "\"}," +
                                    "{\"vlan\":" + vlan2.toString() + "," +
                                    "\"ports\":[1, 3]," +
                                    "\"name\": \"" + fc2.id().id() + "\"}" +
                                    "]}}}}";
        assertEquals(json, parser.parse(expected).getAsJsonObject());

        // Remove first fc
        target.removeCrossconnect(fc1.id());
        json = target.buildConfig();
        expected =
                "{\"apps\":{" +
                        "\"org.onosproject.segmentrouting\":{" +
                        "\"xconnect\":{" +
                        "\"of:0000000000000001\":[{" +
                        "\"vlan\":" + vlan2.toString() + "," +
                        "\"ports\":[1, 3]," +
                        "\"name\": \"" + fc2.id().id() + "\"}" +
                        "]}}}}";
        assertEquals(json, parser.parse(expected).getAsJsonObject());
    }

    private class TestBigSwitchService
            extends AbstractListenerManager<BigSwitchEvent, BigSwitchListener>
            implements BigSwitchService {

        @Override
        public List<PortDescription> getPorts() {
            return null;
        }

        @Override
        public PortNumber getPort(ConnectPoint port) {
            return null;
        }

        @Override
        public Optional<ConnectPoint> connectPointFromVirtPort(PortNumber portNumber) {
            if (portNumber.toLong() == 1) {
                return Optional.of(inniPhyCp);
            }
            if (portNumber.toLong() == 2) {
                return Optional.of(enni1PhyCp);
            }
            if (portNumber.toLong() == 3) {
                return Optional.of(enni2PhyCp);
            }

            return Optional.empty();
        }

        @Override
        public DomainId domainId() {
            return null;
        }
    }
}
